//==============================================================================
//         Copyright 2003 - 2011 LASMEA UMR 6602 CNRS/Univ. Clermont II
//         Copyright 2009 - 2011 LRI    UMR 8623 CNRS/Univ Paris Sud XI
//
//          Distributed under the Boost Software License, Version 1.0.
//                 See accompanying file LICENSE.txt or copy at
//                     http://www.boost.org/LICENSE_1_0.txt
//==============================================================================
#ifndef NT2_TOOLBOX_COMBINATORIAL_FUNCTIONS_SCALAR_RAT_HPP_INCLUDED
#define NT2_TOOLBOX_COMBINATORIAL_FUNCTIONS_SCALAR_RAT_HPP_INCLUDED
#include <nt2/toolbox/combinatorial/functions/rat.hpp>
#include <nt2/include/constants/zero.hpp>
#include <nt2/include/functions/scalar/is_not_finite.hpp>
#include <nt2/include/functions/scalar/is_not_nan.hpp>
#include <nt2/include/functions/eye.hpp>
#include <nt2/include/functions/sign.hpp>
#include <nt2/include/functions/round.hpp>
#include <nt2/include/functions/is_finite.hpp>
#include <nt2/include/functions/horzcat.hpp>
#include <nt2/include/functions/vertcat.hpp>
#include <nt2/include/functions/scalar/is_eqz.hpp>
#include <nt2/include/functions/mtimes.hpp>
#include <nt2/include/functions/scalar/eps.hpp>
#include <nt2/include/functions/scalar/max.hpp>
#include <nt2/include/functions/scalar/rec.hpp>
#include <nt2/core/container/table/table.hpp>


/////////////////////////////////////////////////////////////////////////////
// Implementation when type A0 is arithmetic_
/////////////////////////////////////////////////////////////////////////////
namespace nt2 { namespace ext
{
  NT2_FUNCTOR_IMPLEMENTATION( nt2::tag::rat_, tag::cpu_
                              , (A0)
                              , ( scalar_<floating_<A0> > )
                              ( scalar_ <floating_<A0> > )
                              ( scalar_ <floating_ <A0> >)
    )
  {
    typedef void result_type;
    inline void operator()(A0 const& a0, A0 & n, A0 & d) const
    {
      A0 tol = A0(1.e-6)*nt2::abs(a0);
      rat(a0, tol, n, d);
    }
  };

  NT2_FUNCTOR_IMPLEMENTATION( nt2::tag::rat_, tag::cpu_
                              , (A0)
                              , ( scalar_<floating_<A0> >)
                              ( scalar_<floating_<A0> >)
                              ( scalar_<floating_<A0> >)
                              ( scalar_<floating_<A0> >)
    )
  {
    typedef void result_type;
    inline void operator()(A0 const& a0,A0 const& tol, A0 & n, A0 & d) const
    {
      typedef nt2::container::table<A0> tab_t;
      A0 x = a0;
      n = Zero<A0>();
      d = Zero<A0>();
      if (is_not_finite(x))
      { // Special case for inf, -inf, NaN
        if (is_not_nan(x)) n = nt2::sign(x);
      }
      else
      {
        size_t k = 0;
        tab_t c = nt2::eye(2, meta::as_<A0>());
        while (true)
        {
          ++k;
          d = nt2::round(x);
          if (nt2::is_finite(x))
          {
            x -= d;
            tab_t c0 =  nt2::vertcat(d, One<A0>());
            tab_t c1 = nt2::mtimes(c, c0);
            tab_t c2 = nt2::vertcat(c(1, 1), c(1, 2));
            c =        nt2::horzcat(c1, c2);
          }
          else // Special case for +/- inf
          {
            c =  horzcat(vertcat(x,Zero<A0>()), c(nt2::_, 1));
          }
          if (nt2::is_eqz(x) || (nt2::abs(c(1,1)/c(2,1) - a0) <= nt2::max(tol,nt2::eps(x))))
            break;
          x = rec(x);
        }
        n = c(1,1)/nt2::sign(c(2,1));
        d = nt2::abs(c(2,1));
      }
    }
  };
} }

// function [N,D] = rat(X,tol)
// %RAT    Rational approximation.
// %   [N,D] = RAT(X,tol) returns two integer matrices so that
// %   N./D is close to X in the sense that abs(N./D - X) <= tol.
// %   The rational approximations are generated by truncating continued
// %   fraction expansions.   tol = 1.e-6*norm(X(:),1) is the default.
// %
// %   S = RAT(X) or RAT(X,tol) returns the continued fraction
// %   representation as a string.
// %
// %   The same algorithm, with the default tol, is used internally
// %   by MATLAB for FORMAT RAT.
// %
// %   class support for input X:
// %      float: double, single
// %
// %   See also FORMAT, RATS.

// %   copyright 1984-2011 The MathWorks, Inc.
// %   $Revision: 5.18.4.7 $  $Date: 2011/06/15 08:04:40 $

// % Approximate x by
// %
// %                              1
// %         d1 + ----------------------------------
// %                                 1
// %              d2 + -----------------------------
// %                                   1
// %                   d3 + ------------------------
// %                                      1
// %                        d4 + -------------------
// %                                        1
// %                             d5 + --------------
// %                                           1
// %                                  d6 + ---------
// %                                             1
// %                                       d7 + ----
// %                                             d8
// %
// % The d's are obtained by repeatedly picking off the integer part and
// % then taking the reciprocal of the fractional part.  The accuracy of
// % the approximation increases exponentially with the number of terms
// % and is worst when x = sqrt(2).  For x = sqrt(2), the error with k
// % terms is about 2.68*(.173)^k, which is
// %
// %         1    4.6364e-01
// %         2    8.0210e-02
// %         3    1.3876e-02
// %         4    2.4006e-03
// %         5    4.1530e-04
// %         6    7.1847e-05
// %         7    1.2430e-05
// %         8    2.1503e-06
// %         9    3.7201e-07
// %        10    6.4357e-08




#endif
