# -*- coding: utf-8 -*-

# Copyright 2011 - 2013 Björn Larsson

# This file is part of pytvdbapi.
#
# pytvdbapi is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# pytvdbapi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with pytvdbapi.  If not, see <http://www.gnu.org/licenses/>.

"""
A module for managing banner information

Although some banners are related to a specific season, all banners will
be stored as a property of the related Show instance.
"""

from pytvdbapi import error
from pytvdbapi._compat import implements_to_string
from pytvdbapi.utils import InsensitiveDictionary


@implements_to_string
class Banner(object):
    """
    Representing a Banner as provided by `thetvdb.com <http://thetvdb.com>`_. More information about the
    data provided for a Banner can be found `here <http://thetvdb.com/wiki/index.php/API:banners.xml>`__.

    **Attributes:**

    The number and types of attributes that the Banner has is dependent on
    the type of Banner it is. Below is a description of the different
    attributes.

    Note: Wherever 'text' is given as the attribute type, this means that on Python 2.X it will be of type
    :class:`unicode` and on Python 3.X :class:`str`.

    *Common*:

    These attributes are present for all Banner objects.

    * BannerPath (text). The last part of the URL pointing to the image.\
        This is appended to the correct mirror address to form the full URL.
    * BannerType (text). This could be any of **fanart**, **season** or \
        **poster**. This value controls what other attributes are available \
        as described below.
    * BannerType2 (text). What this attribute contains will depend on the \
        type of Banner and will be described in each sub section below.
    * Language (text).
    * Rating (float).
    * RatingCount (int).
    * id (int).
    * banner_url (text). This is generated by **pytvdbapi** and is the full \
        URL for the banner.

    *fanart:*

    Additional to the common attributes, the following attributes are included
    on objects of type **fanart**.

    * BannerType2 (text). Contains the dimension of the image as text.
    * Colors (list).
    * SeriesName (bool).
    * ThumbnailPath (text).
    * VignettePath (text).

    *poster:*

    **poster** type does not contain any additional attributes.

    * BannerType2 (text). Contains the dimension of the image as a string.

    *season:*

    Additional to the common attributes, the following attributes are included
    on objects of type **season**.

    * BannerType2 (text). Contains the word 'season'
    * Season (int).

    Example::

        >>> from pytvdbapi import api
        >>> db = api.TVDB('B43FF87DE395DF56', banners=True)
        >>> show = db.get_series( 79349, "en" )  # Dexter
        >>> show.update()

        >>> assert len(show.banner_objects) > 0
        >>> banner = show.banner_objects[0]

        >>> print(banner.banner_url) #doctest: +ELLIPSIS
        http://thetvdb.com/banners/fanart/original/79349-...jpg

        >>> print(banner.Language)
        en

        Showing the different banner types and their attributes.

        >>> fanart = [b for b in show.banner_objects if b.BannerType == "fanart"]
        >>> dir(fanart[0])  # doctest: +NORMALIZE_WHITESPACE
        ['BannerPath', 'BannerType', 'BannerType2', 'Colors', 'Language',
        'Rating', 'RatingCount', 'SeriesName', 'ThumbnailPath', 'VignettePath',
        'banner_url', 'id']

        >>> print(fanart[0].BannerType2)
        1280x720


        >>> posters = [b for b in show.banner_objects if b.BannerType == "poster"]
        >>> dir(posters[0])  # doctest: +NORMALIZE_WHITESPACE
        ['BannerPath', 'BannerType', 'BannerType2', 'Language', 'Rating',
        'RatingCount', 'banner_url', 'id']

        >>> print(posters[0].BannerType2)
        680x1000


        >>> seasons = [b for b in show.banner_objects if b.BannerType == "season"]
        >>> dir(seasons[0])  # doctest: +NORMALIZE_WHITESPACE
        ['BannerPath', 'BannerType', 'BannerType2', 'Language', 'Rating',
        'RatingCount', 'Season', 'banner_url', 'id']

        >>> print(seasons[0].BannerType2)
        season
    """

    data = {}

    def __init__(self, mirror, data, show):
        self.mirror, self.show = mirror, show

        # pylint: disable=W0142
        self.data = InsensitiveDictionary(ignore_case=show.api.config['ignore_case'], **data)
        self.data['banner_url'] = self.mirror + u"/banners/" + self.BannerPath

    def __str__(self):
        return u'<Banner({1}) - {0}>'.format(self.id, self.BannerType)

    def __repr__(self):
        return self.__str__()

    def __getattr__(self, item):
        try:
            return self.data[item]
        except KeyError:
            raise error.TVDBAttributeError(
                u"Banner has no {0} attribute".format(item))

    def __dir__(self):
        return list(self.data.keys())
