#  _________________________________________________________________________
#
#  PyUtilib: A Python utility library.
#  Copyright (c) 2008 Sandia Corporation.
#  This software is distributed under the BSD License.
#  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
#  the U.S. Government retains certain rights in this software.
#  _________________________________________________________________________

__all__ = ['Connection', 'UnknownConnection', 'DirectConnection']


class Connection(object):

    def __init__(self, from_task=None, to_task=None):
        import task
        self.from_task = from_task if not from_task is None else task.NoTask
        self.to_task = set()
        if not to_task is None:  #pragma:nocover
            # This is not currently exercised.
            # I am raising an error until we know when/how this code is needed
            raise ValueError, "Setting a to_task in a Connection is not currently supported"
            if isinstance(to_task, task.Task):
                self.to_task.add(to_task)
            else:
                for t in to_task:
                    self.to_task.add(t)

    def get_value(self):
        raise ValueError, "There is no value to get in an abstract Connection object!"  #pragma:nocover

    def set_value(self, value):
        raise ValueError, "There is no value to set in an abstract Connection object!"  #pragma:nocover
    
    def __repr__(self):
        return str(self)

    def __str__(self):
        return "%s: from=(%s) to=(%s)" % (str(self.__class__.__name__), str(self.from_task), str(self.to_task))


class UnknownConnection(Connection):

    def __init__(self, from_task=None, to_task=None):
        Connection.__init__(self, from_task=from_task, to_task=to_task)


class DirectConnection(Connection):

    def __init__(self, value=None, from_task=None, to_task=None):
        Connection.__init__(self, from_task=from_task, to_task=to_task)
        self.set_value(value)

    def get_value(self):
        return self.value

    def set_value(self, value):
        self.value = value


class Connections(dict):

    def __init__(self, task):
        self._name_='Connections'
        self._task = task
        self._inputs=False
        self._outputs=False

    def set_name(self, name):
        self._name_ = name

    def declare(self, *args):
        for name in args:
            if self._inputs:
                 setattr(self, name, UnknownConnection())
            else:
                 setattr(self, name, DirectConnection(from_task=self._task))
        if len(args) == 1:
            return getattr(self, name)

    def __setitem__(self, name, val):
        self.__setattr__(name,val)

    def __getitem__(self, name):
        return self.__getattr__(name)

    def __setattr__(self, name, val):
        #if not name in ('_task'):
           #print "XXX",self.task.name if not self.task is None else "NONE",name,val
        if name[0] == '_':
            #dict.__setitem__(self, name, val)
            self.__dict__[name] = val
            return
        #if not self._declaration and name[0] != '_' and not self._outputs:
            #print self._declaration
            #raise ValueError, "Cannot set attribute values.  Connection object '%s' is not configured for outputs." % self._name_
        if name in ('inputs','outputs'):
            raise ValueError, "Cannot set attribute value with name '%s'" % name
        if not isinstance(val, Connection):
            val = DirectConnection(value=val)
            #raise ValueError, "Cannot set a Connections attribute to a value that is not a subclass of Connection"
        if self._inputs:
            val.to_task.add(self._task)
        else:
            val.from_task = self._task
        dict.__setitem__(self, name, val)
        self.__dict__[name] = val

    def __getattr__(self, name):
        try:
            return dict.__getitem__(self, name)
        except:
            if name[0] == '_':
                raise AttributeError, "Unknown attribute '%s'" % name

    def __repr__(self):
        attrs = sorted("%s = %r" % (k, v) for k, v in self.__dict__.iteritems() if not k.startswith("_"))
        return "%s(%s)" % (self.__class__.__name__, ", ".join(attrs))

    def __str__(self, nesting = 1, indent='', print_name=True):
        attrs = []
        if not print_name:
            nesting -= 1    #pragma:nocover
        indentation = indent+"    " * nesting
        for k, v in self.__dict__.iteritems():
            if not k.startswith("_"):
                text = [indentation, k, " = "]
                text.append(repr(v))
                attrs.append("".join(text))
        attrs.sort()
        if print_name:
            attrs.insert(0, indent+self._name_ + ":")
        if self._inputs:
            attrs.append("Mode:  inputs")
        if self._outputs:
            attrs.append("Mode:  outputs")
        if not self._task is None:
            attrs.append("Owner: "+str(self._task))
        return "\n".join(attrs)



class InputConnections(Connections):

    def __init__(self, task):
        Connections.__init__(self, task)
        self._inputs=True


class OutputConnections(Connections):

    def __init__(self, task):
        Connections.__init__(self, task)
        self._outputs=True

