#  _________________________________________________________________________
#
#  PyUtilib: A Python utility library.
#  Copyright (c) 2008 Sandia Corporation.
#  This software is distributed under the BSD License.
#  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
#  the U.S. Government retains certain rights in this software.
#  _________________________________________________________________________

# TODO: only set option values for variables that show up in a workflow's inputs
# TODO: add graceful management of exceptions
#       show the task tree, etc...

__all__ = ['Workflow']

from task import Task, EmptyTask, NoTask
from options import OptionParser


def _collect_parser_groups(t):
        for key in t._parser_group:
            #
            # NOTE: we are changing the properties of the group
            # instances here.  This is OK _only_ because we are
            # printing the help info and then terminating.
            #
            t._parser_group[key].parser = parser
            parser.add_option_group(t._parser_group[key])

class Workflow(object):

    def __init__(self):
        self.tasks = {}
        self.start_task = EmptyTask()
        self.add(self.start_task)
        self.final_task = EmptyTask()
        self.add(self.final_task)

    def add(self, task, loadall=True):
        if task.id == NoTask.id:
            return
        if task.id in self.tasks:
            return
        self.tasks[task.id] = task
        if not loadall:
            return
        for name in task.inputs:
            t = task.inputs[name].from_task
            if t.id != NoTask.id:
                self.add(t)
            else:
               if not name in self.start_task.outputs:
                   self.start_task.outputs.declare(name)
               #print "Z",self.start_task.outputs
               #print "Z",self.start_task.outputs[name]
               setattr(task.inputs, name, getattr(self.start_task.outputs, name))
               #print "Z",task.inputs
               #print "Z",self.start_task.outputs[name].from_task.id
               #print "Z",self.start_task.outputs[name].to_task.id
               #print "ZZZ",self.start_task.outputs
        for name in task.outputs:
            if len(task.outputs[name].to_task) > 0:
                for t in task.outputs[name].to_task:
                    self.add(t)
            else:
                if name in self.final_task.inputs:
                    raise ValueError, "Cannot declare a workplan with multiple output values that share the same name: %s" % name
                self.final_task.inputs.declare(name)
                setattr(self.final_task.inputs, name, task.outputs[name])

    def __call__(self, **kwds):
        for key in kwds:
            if key not in self.start_task.outputs:
                raise ValueError, "Cannot specify value for option %s.  Valid open names are %s" % (key, self.start_task.outputs.keys())
            self.start_task.outputs[key].set_value( kwds[key] )
        self.execute()
        ans = {}
        for key in self.final_task.inputs:
            ans[key] = self.final_task.inputs[key].get_value()
        # TODO: return a Container
        return ans

    def set_options(self, args):
        self._dfs_([self.start_task.id], lambda t: t.set_options(args))

    def print_help(self):
        parser = OptionParser()
        self._dfs_([self.start_task.id], _collect_parser_groups)
        parser.print_help()

    def execute(self):
        self._dfs_([self.start_task.id], lambda t: t.__call__())

    def __str__(self):
        return "\n".join(["Workflow:"]+self._dfs_([self.start_task.id], lambda t: str(t)))

    def _dfs_(self, indices, fn, touched=None):
        if touched is None:
            touched = set()
        ans=[]
        for i in indices:
            if i in touched:
                # With this design, this condition should never be triggered
                # TODO: verify that this is an O(n) search algorithm; I think it's
                # O(n^2)
                continue        #pragma:nocover
            ok=True
            #print "X",i
            #print "Y all keys",self.tasks.keys()
            #print "Y prev params",self.tasks[i].inputs.keys()
            #print "Y next paramsids",self.tasks[i].outputs.keys()
            #print "Y prev ids",self.tasks[i].prev_task_ids()
            #print "Y next ids",self.tasks[i].next_task_ids()
            task = self.tasks[i]
            for j in task.prev_task_ids():
                if j is NoTask.id or j in touched:
                    continue
                ok=False
                break
            if not ok:
                continue
            tmp = fn(task)
            if not tmp is None:
                ans.append(fn(task))
            touched.add(i)
            ans = ans + self._dfs_(task.next_task_ids(), fn, touched)    
        return ans

