#  _________________________________________________________________________
#
#  PyUtilib: A Python utility library.
#  Copyright (c) 2008 Sandia Corporation.
#  This software is distributed under the BSD License.
#  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
#  the U.S. Government retains certain rights in this software.
#  _________________________________________________________________________

__all__ = ['Task', 'EmptyTask', 'Component']

import connections
from options import OptionParser
import globals


class Task(object):

    def __init__(self, id=None, name=None):
        if not id is None:
            self.id = id
        else:
            self.id = globals.unique_id()
        if name is None:
            self.name = "Task"+str(self.id)
        else:
            self.name = name
        self.inputs = connections.InputConnections(self)
        self.inputs.set_name(self.name+"-inputs")
        self.outputs = connections.OutputConnections(self)
        self.outputs.set_name(self.name+"-outputs")
        self._resources = {}
        self._predecessors = []
        self._create_parser()

    def add_resource(self, res):
        self._resources[res.name] = res

    def resource(self, name):
        return self._resources[name]

    def next_tasks(self):
        return set(t for name in self.outputs for t in self.outputs[name].to_task)
            
    def prev_tasks(self):
        return set(self.inputs[name].from_task for name in self.inputs if self.inputs[name].from_task.id != NoTask.id) | set(task for task in self._predecessors)
           
    def next_task_ids(self):
        return set(task.id for task in self.next_tasks())
            
    def prev_task_ids(self):
        return set(task.id for task in self.prev_tasks())
           
    def execute(self):
        raise ValueError, "There is no default execution for an abstract Task object! Task=%s" % str(self) #pragma:nocover

    def busy(self):
        return [name for name in self._resources if not self._resources[name].available()]

    def __call__(self, *options, **kwds):
        """ 
        Copy the inputs into this Task's dictionary, 
        then execute the task, then copy 
        the outputs out of the dictionary.
        """
        self._call_start()
        busy = self.busy()
        if len(busy) > 0:
            raise IOError, "Cannot execute task %s.  Busy resources: %s" % (self.name, str(busy))
        # Set inputs
        for opt in options:
            #print "Y",opt
            self._set_inputs(opt)
        self._set_inputs(kwds)
        #
        for name, res in self._resources.iteritems():
            res.lock()
        for i in self.outputs:
            #print "z",i,getattr(self.outputs,i).get_value()
            setattr(self, i, None)
        for i in self.inputs:
            #print "X",i,self.inputs[i].get_value(),str(self.inputs[i])
            setattr(self, i, self.inputs[i].get_value())
        self.execute()
        for i in self.outputs:
            #print "Z",i,getattr(self.outputs,i).get_value()
            self.outputs[i].set_value( getattr(self, i) )
        for name, res in self._resources.iteritems():
            res.unlock()
        self._call_finish()

    def set_options(self, args):
        [self.options, args] = self.parser.parse_args(args)
        tmp = {}
        for opt in self.parser._long_opt:
            try:
                val = getattr(self.options,opt[2:])
                tmp[opt[2:]] = val
            except:
                pass
        self._set_inputs(tmp)

    def _call_start(self):
        pass

    def _call_finish(self):
        pass

    def _set_inputs(self, options):
        for key in options:
            self.inputs[key].set_value(options[key])

    def _create_parser(self):
        """
        Create the OptionParser object and populated it with option groups.
        """
        self.parser = OptionParser()
        self._parser_group = {}
        self._create_parser_groups()
        self.parser.ignore_invalid()
        for key in self._parser_group:
            self.parser.add_option_group(self._parser_group[key])

    def _create_parser_groups(self):
        """
        This method is called by the _create_parser method to setup the
        parser groups that are registered for this task.
        """

    def __repr__(self):
        return str(self)

    def __str__(self):
        return "%s prev: %s next: %s resources: %s" % (str(self.name),str(sorted(list(self.prev_task_ids()))),str(sorted(list(self.next_task_ids()))), str(sorted(self._resources.keys())))


"""
Alias for the Task class
"""
class Component(Task):

    def __init__(self, *args, **kwds):
        Task.__init__(self, *args, **kwds)


class EmptyTask(Task):

    def __init__(self, id=None, name=None):
        Task.__init__(self, id=None, name=None)

    def __call__(self, *args, **kwds):
        pass


NoTask = Task(id=0)

