#  Copyright (C) 2007 Maxim Fedorovsky, University of Fribourg (Switzerland).
#       email : Maxim.Fedorovsky@unifr.ch, mutable@yandex.ru
#
#  This file is part of PyVib2.
#
#  PyVib2 is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either VERSION 2 of the License, or
#  (at your option) any later VERSION.
#
#  PyVib2 is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with PyVib2; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""Module for writing files.

The following classes are exported :
    AbstractFileWriter      -- abstract base class for the writers
    FCHKFileWriter          -- Gaussian Formatted Checkpoint files
    VOAVIEWFileWriter       -- VOAVIEW files
    MOLDENFileWriter        -- MOLDEN files
    XMolXYZFileWriter       -- XMol XYZ files
    GaussianInputFileWriter -- Gaussian input files
    HESFileWriter           -- Hessian files of DALTON 1.x or 2.0

"""
__author__ = 'Maxim Fedorovsky'

import os.path
from math  import sqrt

from numpy import zeros

from  pyviblib.molecule         import Molecule
from  pyviblib.util.exceptions  import WriteError, ConstructorError
from  pyviblib.util.constants   import AMU2AU, BOHR2ANGSTROM, C_AU
from  pyviblib.util.misc        import SmartDict, Command, rests
from  pyviblib                  import APPNAME, VERSION

__all__ = ['AbstractFileWriter', 'FCHKFileWriter', 'VOAVIEWFileWriter',
           'MOLDENFileWriter', 'XMolXYZFileWriter',
           'GaussianInputFileWriter', 'HESFileWriter']


class AbstractFileWriter(object) :
  """Abstract base class for all writers.

  The following protected method is called in the constructor :
      _check_consistency()  -- check the data supplied by the user

  The following public method is exported :
      write()               -- write the data and close the file

  Subclasses *must* override these two methods.
  Otherwise a NotImplementedError is raised.

  The following read-only property is exposed :
      filename              -- filename argument supplied in the constructor

  """

  def __init__(self, filename, **data) :
    """Constructor of the class.

    Positional arguments :
    filename -- file name of file object

    Keyword arguments :    
    data     -- data to be written
    
    """
    if data is None :
      raise ConstructorError('Data must be supplied !')
    
    self._filename = filename
    self.__class__.filename = property(fget=Command(
      Command.fget_attr, '_filename'))
    
    self._file     = isinstance(filename, file) \
                     and self._filename or open(filename, 'w+') 
    self._data     = SmartDict(kw=data)

    # check consistency of the data before writing
    self._check_consistency()

  def _check_consistency(self) :
    """Check the consistency of the data supplied in the constructor.

    Subclasses *must* override the method.
    This implementation raises a NotImplementedError.
    
    """
    raise NotImplementedError('_check_consistency() method must be implemented')

  def _raise(self, desc) :
    """Raise a WriteError with a given description."""
    raise WriteError(self._filename, desc)

  def _gen_copyright() :
    """Generate the copyright string."""
    return ' <generated by %s-%s>' % (APPNAME, VERSION)

  _gen_copyright = staticmethod(_gen_copyright)

  def _gen_comment(comment, maxchar=80) :
    """Generate a comment.

    final comment = comment + copyright

    Positional arguments :
    comment -- original comment

    Keyword arguments :
    maxchar -- maximum length of the comment (default 80)

    """
    copyright_ = AbstractFileWriter._gen_copyright()

    comment = (comment or '').replace('\n', '')
    if maxchar is not None :      
      comment = comment[:maxchar - len(copyright_)]

    return '%s%s' % (comment, copyright_)

  _gen_comment = staticmethod(_gen_comment)

  def write(self) :
    """Write the data to a file.

    Subclasses *must* override the method.
    This implementation raises a NotImplementedError.
    
    """
    raise NotImplementedError('write() method *must* be implemented')


class MOLDENFileWriter(AbstractFileWriter) :
  """Writer for MOLDEN files.

  Currently the cartesian coordinates and the normal modes (if available) are
  written.
  
  """

  def __init__(self, filename, **data) :
    """Constructor of the class.

    Positional arguments :
    filename -- file name of file object

    Keyword arguments :    
    molecule -- pyviblib.molecule.Molecule (required)
    comment  -- comment
    
    """
    AbstractFileWriter.__init__(self, filename, **data)

  def _check_consistency(self) :
    """Check the consistency of the data."""
    if not isinstance(self._data['molecule'], Molecule) :
      raise ConstructorError('Invalid molecule argument')

  def write(self) :
    """Write the data."""
    str_to_write = ''
    
    # comment
    comment = self._gen_comment(self._data['comment'])

    # molecule
    mol = self._data['molecule']
      
    str_to_write = ''.join((
      str_to_write,
      '[Molden Format]\n[GEOMETRIES] XYZ\n%5d\n%s\n' % (mol.Natoms, comment)))

    # geometries in angstrom and bohr
    str_geom_bohr = ''
    format_f1  = '%16.8f'
    format_f3  = '%s %s %s\n' % (format_f1, format_f1, format_f1)
    format_sf3 = '%%2s %s' % format_f3
    
    for atom in mol.atoms :
      str_to_write = ''.join((
        str_to_write,
        format_sf3 % (atom.element.symbol, atom.coord[1], atom.coord[2],
                      atom.coord[3])))
      
      str_geom_bohr = ''.join((
        str_geom_bohr,
        format_sf3 %  (atom.element.symbol, 1./BOHR2ANGSTROM * atom.coord[1],
                        1./BOHR2ANGSTROM * atom.coord[2],
                        1./BOHR2ANGSTROM * atom.coord[3])))
    
    # normal modes if given
    if mol.freqs is not None and mol.Lx is not None :
      # frequencies
      str_to_write = ''.join((str_to_write, '[FREQ]\n'))
      for freq in mol.freqs[1:] :
        str_to_write = ''.join((str_to_write, '%8.2f\n' % freq))

      # coordinates in bohr
      str_to_write = ''.join((str_to_write, '[FR-COORD]\n%s' % str_geom_bohr))

      # Lx in amu
      str_to_write = ''.join((str_to_write, '[FR-NORM-COORD]\n'))
      for i in xrange(1, 1 + mol.NFreq) :
        str_to_write = ''.join((str_to_write, 'vibration%6d\n' % i))

        for j in xrange(1, 1 + mol.Natoms) :
          this_str = format_f3 % (sqrt(AMU2AU)* mol.Lx[i, j, 1],
                                  sqrt(AMU2AU)* mol.Lx[i, j, 2],
                                  sqrt(AMU2AU)* mol.Lx[i, j, 3])
          str_to_write = ''.join((str_to_write, this_str))
          
    # finally
    self._file.write(str_to_write)
    self._file.close()


class VOAVIEWFileWriter(AbstractFileWriter) :
  """Writer for VOAVIEW files."""

  def __init__(self, filename, **data) :
    """Constructor of the class.

    Positional arguments :
    filename -- file name of file object

    Keyword arguments :    
    molecule -- pyviblib.molecule.Molecule (required)
    
    """
    AbstractFileWriter.__init__(self, filename, **data)

  def _check_consistency(self) :
    """Check the consistency of the data."""
    if not isinstance(self._data['molecule'], Molecule) :
      raise ConstructorError('Invalid molecule argument supplied')

    if self._data['molecule'].raman_roa_tensors is None :
      print 'Warning : Raman / ROA tensors are not given => writing nulls.'

  def write(self) :
    """Write the data."""
    str_to_write = ''

    mol = self._data['molecule']
    
    # number of atoms
    str_to_write = ''.join((str_to_write,
                            '%13d%13d%13d\n' % (mol.Natoms, 0, 0)))

    # masses in a.m.u & coordinates
    str_masses = ''
    str_coords = ''
    for atom in mol.atoms :
      str_masses = ''.join((
        str_masses,
        '%13.6f %13d %13d\n' % (mol.masses[atom.index], 0, 0)))
      
      str_coords = ''.join((
        str_coords, '%13.6f %13.6f %13.6f\n' % tuple(atom.coord[1:])))

    str_to_write = ''.join((str_to_write, str_masses, str_coords))

    # wavenumbers in cm**(-1)
    # if not given : write nulls
    if mol.Lx is not None :
      NFreq = mol.NFreq
      freqs = mol.freqs  
    else :
      NFreq = max(3 * mol.Natoms - 6, 1)
      freqs = zeros(1 + NFreq, 'd')
    
    for i in xrange(NFreq, 0, -1) :
      str_to_write = ''.join((str_to_write,
                              '%13.2f %13d %13d\n' % (freqs[i], 0, 0)))

    # format for the Lx & tensors
    format = '%13.8f %13.8f %13.8f\n'

    # cartesian displacements Lx in a.u.
    # if not given : write nulls
    if mol.Lx is not None :
      Lx = mol.Lx
    else :
      Lx = zeros((1 + NFreq, 1 + mol.Natoms, 4), 'd')
      
    str_to_write = ''.join((str_to_write, self.__gen_Lx_p(Lx, True, format)))

    # if the tensors are not given - write zeros
    if mol.raman_roa_tensors is not None :
      str_to_write = ''.join((str_to_write,
                              self.__gen_PX(mol.raman_roa_tensors.PP, format)))
      str_to_write = ''.join((str_to_write,
                              self.__gen_PX(mol.raman_roa_tensors.PM, format)))
      str_to_write = ''.join((str_to_write,
                              self.__gen_PX(mol.raman_roa_tensors.PQ, format)))
      
    else :
      dummy_T = zeros((1 + mol.Natoms, 4, 4, 4), 'd')
      for i in xrange(3) :
        str_to_write = ''.join((str_to_write, self.__gen_PX(dummy_T, format)))

    # finally
    self._file.write(str_to_write)
    self._file.close()

  def __gen_Lx_p(Lx, reverse, format) :
    """Generate a string for displacements Lx.

    reverse specifies the direction of the iteration through vibrations.
    format is a format for the whole line.
    
    """
    str_to_write = ''

    range_ = range(1, Lx.shape[0])
    if reverse :
      range_.reverse()

    for p in range_ :
      for a in xrange(1, Lx.shape[1]) :
        str_to_write = ''.join((str_to_write, format % tuple(Lx[p, a, 1:])))

    return str_to_write

  __gen_Lx_p = staticmethod(__gen_Lx_p)

  def __gen_PX(T, format) :
    """Generate a string for an one-based tensor.

    format is a format for the whole line.
    
    """
    str_to_write = ''

    for a in xrange(1, T.shape[0]) :
      for j in xrange(1, 4) :
        for k in xrange(1, 4) :
          str_to_write = ''.join((str_to_write, format % tuple(T[a, 1:, j, k])))

    return str_to_write

  __gen_PX = staticmethod(__gen_PX)


class XMolXYZFileWriter(AbstractFileWriter) :
  """Writer for XMol XYZ files."""

  def __init__(self, filename, **data) :
    """Constructor of the class.

    Positional arguments :
    filename -- file name of file object

    Keyword arguments :    
    molecule -- pyviblib.molecule.Molecule (required)
    comment  -- comment
    
    """
    AbstractFileWriter.__init__(self, filename, **data)

  def _check_consistency(self) :
    """Check the consistency of the data."""
    if not isinstance(self._data['molecule'], Molecule) :
      raise ConstructorError('Invalid molecule argument')

  def write(self) :
    """Write the data."""
    str_to_write = ''
    
    # comment
    comment = self._gen_comment(self._data['comment'])

    # molecule
    mol = self._data['molecule']
      
    str_to_write = ''.join((str_to_write, '%5d\n%s\n' % (mol.Natoms, comment)))

    # geometries in angstrom
    format_f1  = '%16.6f'
    format_f3  = '%s %s %s\n' % (format_f1, format_f1, format_f1)
    format_sf3 = '%%2s %s' % format_f3
    
    for atom in mol.atoms :
      this_str = format_sf3 % (atom.element.symbol, atom.coord[1],
                               atom.coord[2], atom.coord[3])
      str_to_write = ''.join((str_to_write, this_str))
    # finally
    self._file.write(str_to_write)
    self._file.close()


class FCHKFileWriter(AbstractFileWriter) :
  """Writer for Gaussian formatted checkpoint files.

  The following items are written :

  Comment
  Dummy job line (always 'JobType Method Basis')
  Number of atoms
  Charge (always '0')
  Multiplicity (always '1')
  Atomic numbers
  Cartesian coordinates
  Hessian (if available)
  Derivatives of the polarizability tensor     [alpha] (if available)
  Derivatives of the optical rotation tensor   [G']    (if available)
  Derivatives of the D-Q polarizability tensor [A]     (if available)
  Derivatives of the electric dipole moment    [APT]   (if available)
  Derivatives of the magnetic dipole moment    [AAT]   (if available)

  """

  def __init__(self, filename, **data) :
    """Constructor of the class.

    Positional arguments :
    filename -- file name of file object

    Keyword arguments :    
    molecule -- pyviblib.molecule.Molecule (required)
    comment  -- comment (default None)
    hessian  -- hessian matrix (default None)
    
    """
    AbstractFileWriter.__init__(self, filename, **data)
    
  def _check_consistency(self) :
    """Check the consistency of the data."""
    if not isinstance(self._data['molecule'], Molecule) :
      raise ConstructorError('Invalid molecule argument')


  def __gen_PX(PX, format, n_proline) :
    """Generate the text for writing PP, PM or PQ.

    Positional arguments :
    PX        -- tensor to write, shape : (1 + Natoms, 4, 4, 4)
    format    -- format 
    n_proline -- numbers pro line

    Write nulls for first 1 + Natoms.

    """
    str_to_write = ''
    
    for a in xrange(1, 2 * PX.shape[0] - 1) :
      for q in xrange(1, 4) :
        for j in xrange(1, 4) :
          for i in xrange(1, 4) :
            no  = 27 * (a - 1) + 9 * (q - 1) + 3 * (j - 1) + i
            
            if PX.shape[0] > a :
              num = 0.
            else :
              num = PX[a - PX.shape[0] + 1, q, i, j]
              
            str_to_write = ''.join((str_to_write, format % num))
            
            if 0 == no % n_proline :
              str_to_write = ''.join((str_to_write, '\n'))
              
    return str_to_write

  __gen_PX = staticmethod(__gen_PX)

  def __gen_A(A, format, n_proline) :
    """Generate the text for writing the non-contracted A tensor.

    Positional arguments :
    A         -- uncontracted A tensor, shape : (1 + Natoms, 4, 4, 4, 4)
    format    -- format 
    n_proline -- numbers pro line

    Write nulls for first 1 + Natoms.

    """
    str_to_write = ''
    
    for a in xrange(1, 2 * A.shape[0] - 1) :
      for q1 in xrange(1, 4) :
        for q2 in xrange(1, 4) :
          for k in xrange(1, 7) :
            no  = 54 * (a - 1) + 18 * (q1 - 1) + 6 * (q2 - 1) + k
            
            if A.shape[0] > a :
              num = 0.
            else :
              if 4 > k :
                num = A[a - A.shape[0] + 1, q1, q2, k, k]
              elif 6 > k :
                num = A[a - A.shape[0] + 1, q1, q2, 1, k - 2]
              else :
                num = A[a - A.shape[0] + 1, q1, q2, 2, 3]
                              
            str_to_write = ''.join((str_to_write, format % num))
            
            if 0 == no % n_proline :
              str_to_write = ''.join((str_to_write, '\n'))
              
    return str_to_write

  __gen_A = staticmethod(__gen_A)

  def __gen_AXT(AXT, format, n_proline) :
    """Generate the text for writing APT or AAT.

    Positional arguments :
    AXT       -- tensor to write, shape : (1 + Natoms, 4, 4)
    format    -- format 
    n_proline -- numbers pro line

    Do *not* skip any elements !

    """
    str_to_write = ''
    
    for a in xrange(1, AXT.shape[0]) :
      for q in xrange(1, 4) :
        for i in xrange(1, 4) :
          no  = 9 * (a - 1) + 3 * (q - 1) + i
          
          num = AXT[a, q, i]
            
          str_to_write = ''.join((str_to_write, format % num))
          
          if 0 == no % n_proline :
            str_to_write = ''.join((str_to_write, '\n'))
              
    return str_to_write

  __gen_AXT = staticmethod(__gen_AXT)

  def write(self) :
    """Write the data."""
    str_to_write = ''
    
    pattern = '%-40s%4s%5s%12d\n'

    # format for numbers
    format_int = '%12d'
    format_real = '%16.8E'
    
    # number of numerics pro line
    n_proline = 5

    mol = self._data['molecule']
    
    ################ GO ################
    # first line is a comment
    str_to_write = ''.join((str_to_write,
                            self._gen_comment(self._data['comment']), '\n'))

    # second line is JobType, Method, Basis
    str_to_write = ''.join((str_to_write, 'JobType Method Basis\n'))

    # number of atoms
    str_to_write = ''.join(
      (str_to_write, pattern % ('Number of atoms', 'I', '', mol.Natoms)))    

    # charge and multiplicity - writing always 0 1
    str_to_write = ''.join(
      (str_to_write, pattern % ('Charge', 'I', '', 0)))
    str_to_write = ''.join(
      (str_to_write, pattern % ('Multiplicity', 'I', '', 1)))    
    
    # atomic numbers
    str_to_write = ''.join(
      (str_to_write, pattern % ('Atomic numbers', 'I', 'N=', mol.Natoms)))
    
    for atom in mol.atoms :
      str_to_write = ''.join((str_to_write, format_int % atom.element.atomno))
      
      if 0 == atom.index % n_proline :
        str_to_write = ''.join((str_to_write, '\n'))
    
    str_to_write = ''.join((str_to_write, '\n'))

    # cartesian coordinates
    str_to_write = ''.join((
      str_to_write,
      pattern % ('Current cartesian coordinates', 'R', 'N=', 3 * mol.Natoms)))

    for a in xrange(1, 1 + mol.Natoms) :
      for i in xrange(1, 4) :
        str_to_write = ''.join((
          str_to_write, format_real % (mol.coords[a][i] / BOHR2ANGSTROM)))
        
        if 0 == ((a - 1) * 3 + i) % n_proline :
          str_to_write = ''.join((str_to_write, '\n'))

    str_to_write = ''.join((str_to_write, '\n'))

    # hessian (if available)
    if self._data['hessian'] is not None :
      
      str_to_write = ''.join((
        str_to_write,
        pattern % ('Cartesian Force Constants', 'R', 'N=',
                   (3 * mol.Natoms) * (3 * mol.Natoms + 1) / 2 )))
      no = 0
      for i in xrange(1, 1 + 3 * mol.Natoms) :
        b, jj = rests(i)
        for j in xrange(1, 1 + i) :
          no += 1
          a, ii = rests(j)

          str_to_write = ''.join((
            str_to_write,
            format_real % self._data['hessian'][a, ii, b, jj]))

          if 0 == no % n_proline :
            str_to_write = ''.join((str_to_write, '\n'))

    # Raman/ROA tensors (if available)
    if mol.raman_roa_tensors is not None :
      
      str_to_write = ''.join(
        (str_to_write,
         '\n',
         pattern % ('NFreq for DFD properties', 'I', '', 2)))

      str_to_write = ''.join(
        (str_to_write,
         pattern % ('Frequencies for DFD properties', 'R', 'N=', 2)))

      str_to_write = ''.join(
        (str_to_write, format_real % 0.0,
         format_real % (45.563350 / mol.raman_roa_tensors.lambda_incident),
         '\n'))

      # alpha tensor gradient
      str_to_write = ''.join(
        (str_to_write,
         pattern % ('Derivative Alpha(-w,w)', 'R', 'N=', 54 * mol.Natoms)))
      str_to_write = ''.join(
        (str_to_write,
         self.__gen_PX(mol.raman_roa_tensors.PP, format_real, n_proline)))

      # G' tensor gradient - need to recalculate
      str_to_write = ''.join(
        (str_to_write,
         '\n',
         pattern % ('Derivative FD Optical Rotation Tensor', 'R', 'N=',
                    54 * mol.Natoms)))
      str_to_write = ''.join(
        (str_to_write,
         self.__gen_PX(mol.raman_roa_tensors.PM * sqrt(C_AU),
                       format_real, n_proline)))

      # non-contracted A tensor gradient - need to recalculate
      if mol.raman_roa_tensors.A is not None :
        str_to_write = ''.join(
          (str_to_write,
           '\n',
           pattern % ('Derivative D-Q polarizability', 'R', 'N=',
                      108 * mol.Natoms)))
        str_to_write = ''.join(
          (str_to_write,
           self.__gen_A(mol.raman_roa_tensors.A / 1.5,
                        format_real, n_proline)))
      else :
        print 'Warning : non-contracted A tensor is not supplied'

    # IR/VCD tensors (if available)
    if mol.ir_vcd_tensors is not None :
      
      # APT (electric dipole gradients)
      str_to_write = ''.join(
        (str_to_write,
         '\n',
         pattern % ('Dipole Derivatives', 'R', 'N=', 9 * mol.Natoms)))
      str_to_write = ''.join(
        (str_to_write,
         self.__gen_AXT(mol.ir_vcd_tensors.P, format_real, n_proline)))

      # AAT (magnetic dipole gradients)
      str_to_write = ''.join(
        (str_to_write,
         '\n',
         pattern % ('AAT', 'R', 'N=', 9 * mol.Natoms)))
      str_to_write = ''.join(
        (str_to_write,
         self.__gen_AXT(mol.ir_vcd_tensors.M, format_real, n_proline)))      

    # finally wring the buffer to the file
    self._file.write(str_to_write)
    self._file.close()


class HESFileWriter(AbstractFileWriter) :
  """Writer for hessian files of DALTON 1.x or 2.0 (*.hes)."""

  def __init__(self, filename, **data) :
    """Constructor of the class.

    Positional arguments :
    filename -- file name of file object

    Keyword arguments :    
    hessian  -- hessian matrix (required)
    coords   -- cartesian coordinates (required)
    inbohr   -- whether coordinates are given in a.u. (default False)

    """    
    AbstractFileWriter.__init__(self, filename, **data)

  def _check_consistency(self) :
    """Check the consistency of the data."""
    for p in ('hessian', 'coords') :
      if self._data[p] is None :
        raise ConstructorError('%s must be given' % p)

    # by default the coordinates are in angstroms
    self._data['inbohr'] = False

  def write(self) :
    """Write the data."""
    hessian = self._data['hessian']
    coords  = self._data['coords']
    Natoms  = hessian.shape[0] - 1
    
    str_to_write = ''
    str_to_write = ''.join((str_to_write, '%d\n\n' % (3 * Natoms)))

    # formats
    format_hes = '%20.11E\n'
    format_coord = '%17.10f\n'

    for b in xrange(1, 1 + Natoms) :
      for j in xrange(1, 4) :
        row = ''
        for a in xrange(1, 1 + Natoms) :
          for i in xrange(1, 4) :
            row = ''.join((row, format_hes % hessian[a, i, b, j]))
            
        str_to_write = ''.join((str_to_write, row, '\n'))

    for a in xrange(1, 1 + Natoms) :
      for i in xrange(1, 4) :
        if not self._data['inbohr'] :
          curcoord = coords[a, i] / BOHR2ANGSTROM
        else :
          curcoord = coords[a, i]
          
        str_to_write = ''.join((str_to_write, format_coord % curcoord))

    # empty line in the end
    str_to_write = ''.join((str_to_write, '\n'))

    # closing properly
    self._file.write(str_to_write)
    self._file.close()


class GaussianInputFileWriter(AbstractFileWriter) :
  """Writer for Gaussian input files.

  Example is given bellow :
  
  %chk=filebase.chk
  %mem=256MB
  %nproc=1
  # B3LYP/6-31G* OPT

  molecule_name <generated by PyVib2-version>

  0 1
  atomic coordinates (in angstrom)...

  """

  def __init__(self, filename, **data) :
    """Constructor of the class.

    Positional arguments :
    filename -- file name of file object

    Keyword arguments :    
    molecule -- pyviblib.molecule.Molecule (required)
    comment  -- comment
    
    """
    AbstractFileWriter.__init__(self, filename, **data)

  def _check_consistency(self) :
    """Check the consistency of the data."""
    if not isinstance(self._data['molecule'], Molecule) :
      raise ConstructorError('Invalid molecule argument')

  def write(self) :
    """Write the data."""
    str_to_write = ''
    mol = self._data['molecule']    

    # checkpoint file name
    if isinstance(self._filename, file) :
      name = mol.name or 'unknown_molecule'
    else :
      name = os.path.splitext(os.path.basename(self._filename))[0]

    str_to_write = ''.join((str_to_write, '%%chk=%s.chk\n' % name))

    # amount of memory
    str_to_write = ''.join((str_to_write, '%mem=256MB\n'))

    # number of processors
    str_to_write = ''.join((str_to_write, '%nproc=1\n'))

    # root section : optimization task
    str_to_write = ''.join((str_to_write, '# B3LYP/6-31G* OPT\n\n'))
    
    # comment
    comment = self._gen_comment(self._data['comment'])
    str_to_write = ''.join((str_to_write, '%s\n\n' % comment))

    # charge and multiplicity
    str_to_write = ''.join((str_to_write, ' 0 1\n'))

    # geometry in angstrom
    format_f1  = '%16.6f'
    format_f3  = '%s %s %s\n' % (format_f1, format_f1, format_f1)
    format_sf3 = '%%2s %s' % format_f3
    
    for atom in mol.atoms :
      this_str = format_sf3 % (atom.element.symbol, atom.coord[1],
                               atom.coord[2], atom.coord[3])
      str_to_write = ''.join((str_to_write, this_str))

    # empty line at the end
    str_to_write = ''.join((str_to_write, '\n'))
      
    # finally
    self._file.write(str_to_write)
    self._file.close()
