!    -*- F90 -*-
!
! AUTHOR  : ZHENFEI YUAN, TAIZHONG HU
! EMAIL   : ZFYUAN@MAIL.USTC.EDU.CN; THU@USTC.EDU.CN
! URL     : TAIZHONGLAB.USTC.EDU.CN/SOFTWARE/PYVINE.HTML
! DATE    : DEC 3, 2012
! LICENCE : GPL-V3
!
!
! THIS FILE CONSISTS OF ROUTINES FOR BIVARIATE JOE COPULA.
!
! LIST OF ROUTINES:
!
!   PBV_JOE_F
!   DBV_JOE_F
!   BV_JOE_LOGLIK_F
!   RBV_JOE_F
!   BV_JOE_HFUNC_F
!   BV_JOE_INV_HFUNC_F
!   BV_JOE_MLE


SUBROUTINE PBV_JOE_F(U1,U2,RES,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA
  INTEGER :: N

  ! LOCAL VARIABLES
  REAL(KIND=8) :: UD1,UD2,TINY_NUM=TINY(0D0)
  INTEGER :: I

  IF(DELTA.EQ.1D0)THEN
     RES = U1 * U2
     RETURN
  END IF

  DO I=1,N

     UD1 = (1-U1(I))**DELTA
     UD2 = (1-U2(I))**DELTA

     IF((UD1.LE.TINY_NUM).OR.(UD2.LE.TINY_NUM))THEN
        RES(I) = MIN(U1(I),U2(I))
     ELSE
        RES(I) = 1.0D00-&
             ((1-U1(I))**DELTA+(1-U2(I))**DELTA&
             -(1-U1(I))**DELTA*(1-U2(I))**DELTA)**(1.0D00/DELTA)
     END IF
     
  END DO

END SUBROUTINE PBV_JOE_F

SUBROUTINE DBV_JOE_F(U1,U2,RES,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA
  INTEGER :: N

  ! LOCAL VARIABLES
  REAL(KIND=8) :: Y,A,B,R,F_DELTA,G_DELTA,POSINF=HUGE(0.0D00)
  INTEGER :: I


  IF(DELTA.EQ.1.0D00)THEN
     RES = 1.0D00
     RETURN     
  END IF

  F_DELTA = DELTA / (2*DELTA-1)
  G_DELTA = DELTA**2 / (2*DELTA-1)
  

  DO I = 1,N

     A = 1-U1(I)
     B = 1-U2(I)
     Y = (1-A**DELTA)*(1-B**DELTA)
     R = A / B

     IF(R.GE.1)THEN
        IF(R**G_DELTA.GT.POSINF)THEN
           RES(I) = (DELTA-Y) / (R**DELTA * B)
        ELSE
           RES(I) = (DELTA-Y) &
                / (R**G_DELTA * B**F_DELTA + (1/R)**G_DELTA * A**F_DELTA -A**G_DELTA * B**G_DELTA)**(1/F_DELTA)
        END IF
     ELSE
        IF((1/R)**G_DELTA.GT.POSINF)THEN
           RES(I) = (DELTA-Y) / ((1/R)**DELTA * A)
        ELSE
           RES(I) = (DELTA-Y) &
                / (R**G_DELTA * B**F_DELTA + (1/R)**G_DELTA * A**F_DELTA -A**G_DELTA * B**G_DELTA)**(1/F_DELTA)
        END IF

     END IF

  END DO

END SUBROUTINE DBV_JOE_F

SUBROUTINE BV_JOE_HFUNC_F(U1,U2,RES,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA
  INTEGER :: N

  ! LOCAL VARIABLES

  IF(DELTA.EQ.1D0)THEN
     RES = U1
     RETURN
  END IF

  RES = (1-(1-U1)**DELTA) &
       / (((1-U1)/(1-U2))**DELTA+1-(1-U1)**DELTA)**(1-1/DELTA)

END SUBROUTINE BV_JOE_HFUNC_F



SUBROUTINE BV_JOE_INV_HFUNC_F(U1,U2,RES,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA
  INTEGER :: N

  ! LOCAL VARIABLES
  REAL(KIND=8) :: B,R
  REAL(KIND=8) , EXTERNAL :: INV_H_ROOT
  INTEGER :: I

  IF(DELTA.EQ.1D0)THEN
     RES = U1
     RETURN
  END IF

  DO I = 1,N
     B = 1 - U2(I)
     R = INV_H_ROOT(DELTA,U1(I),B)
     RES(I) = 1 - R*B
  END DO

END SUBROUTINE BV_JOE_INV_HFUNC_F


FUNCTION INV_H_ROOT(DELTA,W,B)

  IMPLICIT NONE

  REAL(KIND=8) :: DELTA,W,B,INV_H_ROOT

  ! LOCAL
  REAL(KIND=8) :: POSINF,FA,FB,TL,MCHP
  REAL(KIND=8) , EXTERNAL :: ZERO, TARGET_FUN

  FA = 1D-8
  FB = 1 / B - 1D-8
  TL = 1D-14
  MCHP = 1D-14
  POSINF = HUGE(0.0D00)

  INV_H_ROOT = ZERO(FA,FB,MCHP,TL,TARGET_FUN,DELTA,W,B)

END FUNCTION INV_H_ROOT

FUNCTION TARGET_FUN(R,DELTA,W,B)
  REAL(KIND=8) :: R,TARGET_FUN,DELTA,W,B,POSINF=HUGE(0D0)
  IF(R**DELTA.GE.POSINF)THEN
     TARGET_FUN = (1-DELTA) * LOG(R) + (1/DELTA-1) * LOG(1-B**DELTA) + LOG(1-(R*B)**DELTA) - LOG(W)
  ELSE
     TARGET_FUN = (1/DELTA-1) * LOG(1+R**DELTA * (1-B**DELTA)) + LOG(1-(R*B)**DELTA) - LOG(W)
  END IF
END FUNCTION TARGET_FUN


SUBROUTINE RBV_JOE_F(U1,U2,DELTA,N)

  IMPLICIT NONE
  
  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA
  INTEGER :: N

  CALL INIT_RANDOM_SEED()

  CALL RANDOM_NUMBER(RES)
  CALL RANDOM_NUMBER(U2)

  CALL BV_JOE_INV_HFUNC_F(RES,U2,U1,DELTA,N)

END SUBROUTINE RBV_JOE_F



FUNCTION ZERO ( A, B, MACHEP, T, F ,TARGET_DELTA,TARGET_W,TARGET_B)

!*****************************************************************************80
!
!! ZERO SEEKS THE ROOT OF A FUNCTION F(X) IN AN INTERVAL [A,B].
!
!  DISCUSSION:
!
!    THE INTERVAL [A,B] MUST BE A CHANGE OF SIGN INTERVAL FOR F.
!    THAT IS, F(A) AND F(B) MUST BE OF OPPOSITE SIGNS.  THEN
!    ASSUMING THAT F IS CONTINUOUS IMPLIES THE EXISTENCE OF AT LEAST
!    ONE VALUE C BETWEEN A AND B FOR WHICH F(C) = 0.
!
!    THE LOCATION OF THE ZERO IS DETERMINED TO WITHIN AN ACCURACY
!    OF 6 * MACHEPS * ABS ( C ) + 2 * T.
!
!  LICENSING:
!
!    THIS CODE IS DISTRIBUTED UNDER THE GNU LGPL LICENSE. 
!
!  MODIFIED:
!
!    12 APRIL 2008
!
!  AUTHOR:
!
!    ORIGINAL FORTRAN77 VERSION BY RICHARD BRENT.
!    FORTRAN90 VERSION BY JOHN BURKARDT.
!
!  REFERENCE:
!
!    RICHARD BRENT,
!    ALGORITHMS FOR MINIMIZATION WITHOUT DERIVATIVES,
!    DOVER, 2002,
!    ISBN: 0-486-41998-3,
!    LC: QA402.5.B74.
!
!  PARAMETERS:
!
!    INPUT, REAL ( KIND = 8 ) A, B, THE ENDPOINTS OF THE CHANGE OF 
!    SIGN INTERVAL.
!
!    INPUT, REAL ( KIND = 8 ) MACHEP, AN ESTIMATE FOR THE RELATIVE MACHINE
!    PRECISION.
!
!    INPUT, REAL ( KIND = 8 ) T, A POSITIVE ERROR TOLERANCE.
!
!    INPUT, EXTERNAL REAL ( KIND = 8 ) F, THE NAME OF A USER-SUPPLIED
!    FUNCTION, OF THE FORM "FUNCTION F ( X )", WHICH EVALUATES THE
!    FUNCTION WHOSE ZERO IS BEING SOUGHT.
!
!    OUTPUT, REAL ( KIND = 8 ) ZERO, THE ESTIMATED VALUE OF A ZERO OF
!    THE FUNCTION F.
!
  IMPLICIT NONE

  INTERFACE
     FUNCTION F(TARGET_R,TARGET_DELTA,TARGET_W,TARGET_B)
       IMPLICIT NONE
       REAL(KIND=8) :: TARGET_R,TARGET_DELTA,TARGET_W,TARGET_B,F
     END FUNCTION F
  END INTERFACE

  REAL ( KIND = 8 ) :: TARGET_DELTA, TARGET_W, TARGET_B

  REAL ( KIND = 8 ) A
  REAL ( KIND = 8 ) B
  REAL ( KIND = 8 ) C
  REAL ( KIND = 8 ) D
  REAL ( KIND = 8 ) E
  ! REAL ( KIND = 8 ) F!ZFYUAN COMMENTED
  REAL ( KIND = 8 ) FA
  REAL ( KIND = 8 ) FB
  REAL ( KIND = 8 ) FC
  REAL ( KIND = 8 ) M
  REAL ( KIND = 8 ) MACHEP
  REAL ( KIND = 8 ) P
  REAL ( KIND = 8 ) Q
  REAL ( KIND = 8 ) R
  REAL ( KIND = 8 ) S
  REAL ( KIND = 8 ) SA
  REAL ( KIND = 8 ) SB
  REAL ( KIND = 8 ) T
  REAL ( KIND = 8 ) TOL
  REAL ( KIND = 8 ) ZERO
  !
  !  MAKE LOCAL COPIES OF A AND B.
  !
  SA = A
  SB = B
  ! FA = F ( SA )
  ! FB = F ( SB )! ZFYUAN COMMENTED. ADD TWO LINES BELOW. DEC 28, 2012.
  FA = F ( SA ,TARGET_DELTA,TARGET_W,TARGET_B)
  FB = F ( SB ,TARGET_DELTA,TARGET_W,TARGET_B)

  C = SA
  FC = FA
  E = SB - SA
  D = E

  DO

     IF ( ABS ( FC ) < ABS ( FB ) ) THEN

        SA = SB
        SB = C
        C = SA
        FA = FB
        FB = FC
        FC = FA

     END IF

     TOL = 2.0D+00 * MACHEP * ABS ( SB ) + T
     M = 0.5D+00 * ( C - SB )

     IF ( ABS ( M ) <= TOL .OR. FB == 0.0D+00 ) THEN
        EXIT
     END IF

     IF ( ABS ( E ) < TOL .OR. ABS ( FA ) <= ABS ( FB ) ) THEN

        E = M
        D = E

     ELSE

        S = FB / FA

        IF ( SA == C ) THEN

           P = 2.0D+00 * M * S
           Q = 1.0D+00 - S

        ELSE

           Q = FA / FC
           R = FB / FC
           P = S * ( 2.0D+00 * M * A * ( Q - R ) - ( SB - SA ) * ( R - 1.0D+00 ) )
           Q = ( Q - 1.0D+00 ) * ( R - 1.0D+00 ) * ( S - 1.0D+00 )

        END IF

        IF ( 0.0D+00 < P ) THEN
           Q = - Q
        ELSE
           P = - P
        END IF

        S = E
        E = D

        IF ( 2.0D+00 * P < 3.0D+00 * M * Q - ABS ( TOL * Q ) .AND. &
             P < ABS ( 0.5D+00 * S * Q ) ) THEN
           D = P / Q
        ELSE
           E = M
           D = E
        END IF

     END IF

     SA = SB
     FA = FB

     IF ( TOL < ABS ( D ) ) THEN
        SB = SB + D
     ELSE IF ( 0.0D+00 < M ) THEN
        SB = SB + TOL
     ELSE
        SB = SB - TOL
     END IF

     ! FB = F ( SB )! ZFYUAN COMMENTED, ADD ONE LINE BELOW, DEC 28, 2012.
     FB = F(SB, TARGET_DELTA,TARGET_W,TARGET_B)

     IF ( ( 0.0D+00 < FB .AND. 0.0D+00 < FC ) .OR. &
          ( FB <= 0.0D+00 .AND. FC <= 0.0D+00 ) ) THEN
        C = SA
        FC = FA
        E = SB - SA
        D = E
     END IF

  END DO

  ZERO = SB

  RETURN
END FUNCTION ZERO


FUNCTION BV_JOE_LOGLIK_F(U1,U2,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA,BV_JOE_LOGLIK_F
  INTEGER :: N

  CALL DBV_JOE_F(U1,U2,RES,DELTA,N)
  BV_JOE_LOGLIK_F = SUM(LOG(RES))
  

END FUNCTION BV_JOE_LOGLIK_F


SUBROUTINE BV_JOE_INIT(U1,U2,NNUM,LOW_INIT,UPP_INIT)
  IMPLICIT NONE

  INTEGER               :: NNUM
  REAL(KIND=8)          :: U1(NNUM),U2(NNUM),LOW_INIT,UPP_INIT
  ! LOCAL
  REAL(KIND=8),EXTERNAL :: BV_JOE_LOGLIK_F
  REAL(KIND=8)          :: LOGLIK_L,LOGLIK_R,STEP=2D0

  LOW_INIT = 1.001D0
  
  LOGLIK_L = BV_JOE_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
  LOGLIK_R = BV_JOE_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  DO WHILE(LOGLIK_L.LT.LOGLIK_R)
     LOW_INIT = LOW_INIT + STEP
     
     LOGLIK_L = BV_JOE_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
     LOGLIK_R = BV_JOE_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  END DO

  UPP_INIT = LOW_INIT + STEP
  LOW_INIT = MAX(LOW_INIT - STEP,1.001D0)

END SUBROUTINE BV_JOE_INIT


SUBROUTINE BV_JOE_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  ! LOCAL
  REAL(KIND=8)           :: EPS = 1.5D-8, TOL = 1D-3, LOW, UPP, X
  REAL(KIND=8), EXTERNAL :: BV_JOE_LOGLIK_F, LOCAL_MIN

  CALL BV_JOE_INIT(U1,U2,NNUM,LOW,UPP)

  LL = -LOCAL_MIN(LOW,UPP,EPS,TOL,BV_JOE_LOGLIK_F,U1,U2,NNUM,X)

  RES(1) = X
  RES(2) = 0D0

! CONTAINS
  
!   FUNCTION TARGET_FUN(DELTA)
!     IMPLICIT NONE
!     REAL(KIND=8) :: DELTA, TARGET_FUN
!     TARGET_FUN = -BV_JOE_LOGLIK_F(U1,U2,DELTA,NNUM)
!   END FUNCTION TARGET_FUN

END SUBROUTINE BV_JOE_MLE

