!    -*- F90 -*-
!
! AUTHOR  : ZHENFEI YUAN, TAIZHONG HU
! EMAIL   : ZFYUAN@MAIL.USTC.EDU.CN; THU@USTC.EDU.CN
! URL     : TAIZHONGLAB.USTC.EDU.CN/SOFTWARE/PYVINE.HTML
! DATE    : DEC 3, 2012
! LICENCE : GPL-V3
!
!
! THIS FILE CONSISTS OF ROUTINES FOR BIVARIATE NORMAL COPULA.
!
! LIST OF ROUTINES:
!
!   PBV_NORM_F
!   DBV_NORM_F
!   BV_NORM_LOGLIK_F
!   RBV_NORM_F
!   BV_NORM_HFUNC_F
!   BV_NORM_INV_HFUNC_F
!   BV_NORM_MLE



SUBROUTINE PBV_NORM_F(U1,U2,RES,RHO,N)
  
  REAL(KIND=8) :: U1(N), U2(N)
  REAL(KIND=8) :: RES(N)
  REAL(KIND=8) :: RHO
  INTEGER :: N

  INTEGER :: I

  REAL(KIND=8) :: QU1,QU2
  REAL(KIND=8) , EXTERNAL :: BIVNOR

  DO I = 1,N
     IF(U1(I)*U2(I).EQ.0)THEN
        RES(I) = 0.0
     ELSE IF(U1(I).EQ.1)THEN
        RES(I) = U2(I)
     ELSE IF(U2(I).EQ.1)THEN
        RES(I) = U1(I)
     ELSE
        CALL QNORM(U1(I),QU1)
        CALL QNORM(U2(I),QU2)
        QU1 = - QU1
        QU2 = - QU2
        RES(I) = BIVNOR(QU1,QU2,RHO)
     END IF
  END DO

END SUBROUTINE PBV_NORM_F


SUBROUTINE DBV_NORM_F(U1,U2,RES,RHO,N)

  REAL(KIND=8) :: U1(N),U2(N),RES(N)
  REAL(KIND=8) :: RHO
  INTEGER :: N

  REAL(KIND=8) :: QU1(N),QU2(N),G1,G2(N)

  CALL QNORM_VEC(U1,QU1,N)
  CALL QNORM_VEC(U2,QU2,N)

  G1 = 1.0 / SQRT(1.0-RHO**2)
  G2 = -0.5 * ( RHO**2 * QU1**2 - 2*RHO*QU1*QU2 + RHO**2 * QU2**2)&
       / ( 1.-RHO**2 )

  RES = EXP(G2) * G1

END SUBROUTINE DBV_NORM_F

SUBROUTINE RBV_NORM_F(U1,U2,RHO,N)

  REAL(KIND=8) :: U1(N),U2(N)
  REAL(KIND=8) :: RHO

  REAL(KIND=8) :: RES(N)
  INTEGER :: N

  CALL INIT_RANDOM_SEED()
  CALL RANDOM_NUMBER(U1)
  CALL RANDOM_NUMBER(U2)

  IF(RHO.EQ.1)THEN
     U1 = U2
  ELSE IF(RHO.EQ.-1)THEN
     U1 = 1 - U2
  ELSE
     CALL BV_NORM_INV_HFUNC_F(U1,U2,RES,RHO,N)
     U1 = RES
  END IF
  
END SUBROUTINE RBV_NORM_F

SUBROUTINE BV_NORM_HFUNC_F(U1,U2,RES,RHO,N)
  REAL(KIND=8) :: U1(N),U2(N),RES(N)
  REAL(KIND=8) :: QU1(N),QU2(N),TMP(N)
  REAL(KIND=8) :: RHO
  INTEGER :: N

  CALL QNORM_VEC(U1,QU1,N)
  CALL QNORM_VEC(U2,QU2,N)

  TMP = (QU1 - RHO * QU2) / SQRT(1-RHO**2)

  CALL PNORM_VEC(TMP,RES,N)

END SUBROUTINE BV_NORM_HFUNC_F


SUBROUTINE BV_NORM_INV_HFUNC_F(U1,U2,RES,RHO,N)

  REAL(KIND=8) :: U1(N), U2(N), RES(N)
  REAL(KIND=8) :: RHO  
  INTEGER :: N

  REAL(KIND=8) :: QU1(N), QU2(N) , PARA

  CALL QNORM_VEC(U1,QU1,N)
  CALL QNORM_VEC(U2,QU2,N)

  PARA = SQRT(1.0 - RHO**2)

  CALL PNORM_VEC(QU1*PARA+QU2*RHO,RES,N)

END SUBROUTINE BV_NORM_INV_HFUNC_F




FUNCTION BV_NORM_LOGLIK_F(U1,U2,RHO,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),RHO,BV_NORM_LOGLIK_F
  INTEGER :: N

  CALL DBV_NORM_F(U1,U2,RES,RHO,N)

  BV_NORM_LOGLIK_F = SUM(LOG(RES))
  
END FUNCTION BV_NORM_LOGLIK_F



SUBROUTINE BV_NORM_INIT(U1,U2,NNUM,INIT)
  IMPLICIT NONE

  INTEGER      :: NNUM
  REAL(KIND=8) :: U1(NNUM),U2(NNUM),INIT
  ! LOCAL
  REAL(KIND=8) :: QU1(NNUM),QU2(NNUM)

  CALL QNORM_VEC(U1,QU1,NNUM)
  CALL QNORM_VEC(U2,QU2,NNUM)

  QU1 = QU1 - SUM(QU1) / NNUM
  QU2 = QU2 - SUM(QU2) / NNUM

  INIT = SUM(QU1*QU2) / SQRT( SUM(QU1**2) * SUM(QU2**2) )

END SUBROUTINE BV_NORM_INIT



SUBROUTINE BV_NORM_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  ! LOCAL
  REAL(KIND=8)           :: EPS = 1.5D-08,TOL = 1D-4,LOW, UPP , X
  REAL(KIND=8), EXTERNAL :: BV_NORM_LOGLIK_F,LOCAL_MIN

  CALL BV_NORM_INIT(U1,U2,NNUM,X)

  LOW = MAX(-9.99D-1 , X-5D-2 )
  UPP = MIN( 9.99D-1 , X+5D-2 )

  LL = -LOCAL_MIN(LOW,UPP,EPS,TOL,BV_NORM_LOGLIK_F,U1,U2,NNUM,X)

  RES(1) = X
  RES(2) = 0D0

! CONTAINS

!   FUNCTION TARGET_FUN(RHO)
!     IMPLICIT NONE
!     REAL(KIND=8) :: RHO,TARGET_FUN
!     TARGET_FUN = -BV_NORM_LOGLIK_F(U1,U2,RHO,NNUM)
!   END FUNCTION TARGET_FUN

END SUBROUTINE BV_NORM_MLE

