!    -*- F90 -*-
!
! AUTHOR  : ZHENFEI YUAN, TAIZHONG HU
! EMAIL   : ZFYUAN@MAIL.USTC.EDU.CN; THU@USTC.EDU.CN
! URL     : TAIZHONGLAB.USTC.EDU.CN/SOFTWARE/PYVINE.HTML
! DATE    : DEC 3, 2012
! LICENCE : GPL-V3
!
!
! 'PROB.F90' PROVIDES FUNCTIONS AND SUBROUTINES FOR ONE-DIMENSIONAL
! PROBABILITY FUNCTIONS OF NORMAL AND STUDENT T DISTRIBUTIONS.
!
! LIST OF ROUTINES
!
!   PNORM
!   PNORM_VEC
!   QNORM
!   QNORM_VEC
!   QNORM_MAT
!   QT
!   QT_VEC
!   PT
!   PT_VEC
!   DT
!   DT_VEC
!
! THE FOLLOWING ROUTINES COME FROM JOHN BURKARDT'S HOMEPAGE, AND
! THANKS FOR HIS TREMENDOUS CONTRIBUTION FOR SCIENTIFIC COMPUTING. FOR
! HIS HOMEPAGE, SEE : 'PEOPLE.SC.FSU.EDU/~JBURKARDT'
!
!   NORMAL
!   NORMDEV
!   STUDENT
!   T_QUANTILE
!   BIVNOR



FUNCTION NORMAL(X)RESULT(U)
  REAL(KIND=8) :: X,U
  U = 0.5 + 0.5 * ERF(X/SQRT(2.0D+00))  
END FUNCTION NORMAL



FUNCTION BIVNOR ( AH, AK, R )

!*****************************************************************************80
!
!! BIVNOR COMPUTES THE BIVARIATE NORMAL CDF.
!
!  DISCUSSION:
!
!    BIVNOR COMPUTES THE PROBABILITY FOR TWO NORMAL VARIATES X AND Y
!    WHOSE CORRELATION IS R, THAT AH <= X AND AK <= Y.
!
!  LICENSING:
!
!    THIS CODE IS DISTRIBUTED UNDER THE GNU LGPL LICENSE.
!
!  MODIFIED:
!
!    13 APRIL 2012
!
!  AUTHOR:
!
!    ORIGINAL FORTRAN77 VERSION BY THOMAS DONNELLY.
!    FORTRAN90 VERSION BY JOHN BURKARDT.
!
!  REFERENCE:
!
!    THOMAS DONNELLY,
!    ALGORITHM 462: BIVARIATE NORMAL DISTRIBUTION,
!    COMMUNICATIONS OF THE ACM,
!    OCTOBER 1973, VOLUME 16, NUMBER 10, PAGE 638.
!
!  PARAMETERS:
!
!    INPUT, REAL ( KIND = 8 ) AH, AK, THE LOWER LIMITS OF INTEGRATION.
!
!    INPUT, REAL ( KIND = 8 ) R, THE CORRELATION BETWEEN X AND Y.
!
!    OUTPUT, REAL ( KIND = 8 ) BIVNOR, THE BIVARIATE NORMAL CDF.
!
!  LOCAL PARAMETERS:
!
!    LOCAL, INTEGER ( KIND = 4 ) IDIG, THE NUMBER OF SIGNIFICANT DIGITS
!    TO THE RIGHT OF THE DECIMAL POINT DESIRED IN THE ANSWER.
!
  IMPLICIT NONE

  REAL ( KIND = 8 ) :: A2
  REAL ( KIND = 8 ) :: AH
  REAL ( KIND = 8 ) :: AK
  REAL ( KIND = 8 ) :: AP
  REAL ( KIND = 8 ) :: B
  REAL ( KIND = 8 ) :: BIVNOR
  REAL ( KIND = 8 ) :: CN
  REAL ( KIND = 8 ) :: CON
  REAL ( KIND = 8 ) :: CONEX
  REAL ( KIND = 8 ) :: EX
  REAL ( KIND = 8 ) :: G2
  REAL ( KIND = 8 ) :: GAUSS
  REAL ( KIND = 8 ) :: GH
  REAL ( KIND = 8 ) :: GK
  REAL ( KIND = 8 ) :: GW
  REAL ( KIND = 8 ) :: H2
  REAL ( KIND = 8 ) :: H4
  INTEGER ( KIND = 4 ) :: I
  INTEGER ( KIND = 4 ), PARAMETER :: IDIG = 15
  INTEGER ( KIND = 4 ) :: IS
  REAL ( KIND = 8 ) :: R
  REAL ( KIND = 8 ) :: RR
  REAL ( KIND = 8 ) :: S1
  REAL ( KIND = 8 ) :: S2
  REAL ( KIND = 8 ) :: SGN
  REAL ( KIND = 8 ) :: SN
  REAL ( KIND = 8 ) :: SP
  REAL ( KIND = 8 ) :: SQR
  REAL ( KIND = 8 ) :: T
  REAL ( KIND = 8 ), PARAMETER :: TWOPI = 6.283185307179587D+00
  REAL ( KIND = 8 ) :: W2
  REAL ( KIND = 8 ) :: WH
  REAL ( KIND = 8 ) :: WK

  GAUSS ( T ) = ( 1.0D+00 + ERF ( T / SQRT ( 2.0D+00 ) ) ) / 2.0D+00

!  GAUSS IS A UNIVARIATE LOWER NORMAL TAIL AREA CALCULATED HERE FROM THE
!  CENTRAL ERROR FUNCTION ERF.
!
  B = 0.0D+00

  GH = GAUSS ( - AH ) / 2.0D+00
  GK = GAUSS ( - AK ) / 2.0D+00

  IF ( R == 0.0D+00 ) THEN
    B = 4.0D+000 * GH * GK
    B = MAX ( B, 0.0D+00 )
    B = MIN ( B, 1.0D+00 )
    BIVNOR = B
    RETURN
  END IF

  RR = ( 1.0D+00 + R ) * ( 1.0D+00 - R )

  IF ( RR < 0.0D+00 ) THEN
    WRITE ( *, '(A)' ) ' '
    WRITE ( *, '(A)' ) 'BIVNOR - FATAL ERROR!'
    WRITE ( *, '(A)' ) '  1 < |R|.'
    STOP
  END IF

  IF ( RR == 0.0D+00 ) THEN

    IF ( R < 0.0D+00 ) THEN

      IF ( AH + AK < 0.0D+00 ) THEN
        B = 2.0D+00 * ( GH + GK ) - 1.0D+00
      END IF

    ELSE

      IF ( AH - AK < 0.0D+00 ) THEN
        B = 2.0D+00 * GK
      ELSE
        B = 2.0D+00 * GH
      END IF

    END IF

    B = MAX ( B, 0.0D+00 )
    B = MIN ( B, 1.0D+00 )
    BIVNOR = B
    RETURN

  END IF

  SQR = SQRT ( RR )

  IF ( IDIG == 15 ) THEN
    CON = TWOPI * 1.0D-15 / 2.0D+00
  ELSE
    CON = TWOPI / 2.0D+00
    DO I = 1, IDIG
      CON = CON / 10.0D+00
    END DO
  END IF
!
!  (0,0)
!
  IF ( AH == 0.0D+00 .AND. AK == 0.0D+00 ) THEN
    B = 0.25D+00 + ASIN ( R ) / TWOPI
    B = MAX ( B, 0.0D+00 )
    B = MIN ( B, 1.0D+00 )
    BIVNOR = B
    RETURN
  END IF
!
!  (0,NONZERO)
!
  IF ( AH == 0.0D+00 .AND. AK /= 0.0D+00 ) THEN

    B = GK
    WH = -AK
    WK = ( AH / AK - R ) / SQR
    GW = 2.0D+00 * GK
    IS = 1
!
!  (NONZERO,0)
!
  ELSE IF ( AH /= 0.0D+00 .AND. AK == 0.0D+00 ) THEN

    B = GH
    WH = -AH
    WK = ( AK / AH - R ) / SQR
    GW = 2.0D+00 * GH
    IS = -1
!
!  (NONZERO,NONZERO)
!
  ELSE IF ( AH /= 0.0 .AND. AK /= 0.0 ) THEN

    B = GH + GK
    IF ( AH * AK < 0.0D+00 ) THEN
      B = B - 0.5D+00
    END IF
    WH = - AH
    WK = ( AK / AH - R ) / SQR
    GW = 2.0D+00 * GH
    IS = -1

  END IF

  DO

    SGN = -1.0D+00
    T = 0.0D+00

    IF ( WK /= 0.0D+00 ) THEN

      IF ( ABS ( WK ) == 1.0D+00 ) THEN

        T = WK * GW * ( 1.0D+00 - GW ) / 2.0D+00
        B = B + SGN * T

      ELSE

        IF ( 1.0D+00 < ABS ( WK ) ) THEN

          SGN = -SGN
          WH = WH * WK
          G2 = GAUSS ( WH )
          WK = 1.0D+00 / WK

          IF ( WK < 0.0D+00 ) THEN
            B = B + 0.5D+00
          END IF

          B = B - ( GW + G2 ) / 2.0D+00 + GW * G2

        END IF

        H2 = WH * WH
        A2 = WK * WK
        H4 = H2 / 2.0D+00
        EX = EXP ( - H4 )
        W2 = H4 * EX
        AP = 1.0D+00
        S2 = AP - EX
        SP = AP
        S1 = 0.0D+00
        SN = S1
        CONEX = ABS ( CON / WK )

        DO

          CN = AP * S2 / ( SN + SP )
          S1 = S1 + CN

          IF ( ABS ( CN ) <= CONEX ) THEN
            EXIT
          END IF

          SN = SP
          SP = SP + 1.0D+00
          S2 = S2 - W2
          W2 = W2 * H4 / SP
          AP = - AP * A2

        END DO

        T = ( ATAN ( WK ) - WK * S1 ) / TWOPI
        B = B + SGN * T

      END IF

    END IF

    IF ( 0 <= IS ) THEN
      EXIT
    END IF

    IF ( AK == 0.0D+00 ) THEN
      EXIT
    END IF

    WH = -AK
    WK = ( AH / AK - R ) / SQR
    GW = 2.0D+00 * GK
    IS = 1

  END DO

  B = MAX ( B, 0.0D+00 )
  B = MIN ( B, 1.0D+00 )
  BIVNOR = B

  RETURN
END FUNCTION BIVNOR


FUNCTION NORMDEV(P)

  !*****************************************************************************80
  !
  !! R8_NORMAL_01_CDF_INVERSE INVERTS THE STANDARD NORMAL CDF.
  !
  !  DISCUSSION:
  !
  !    THE RESULT IS ACCURATE TO ABOUT 1 PART IN 10**16.
  !
  !  LICENSING:
  !
  !    THIS CODE IS DISTRIBUTED UNDER THE GNU LGPL LICENSE.
  !
  !  MODIFIED:
  !
  !    27 DECEMBER 2004
  !
  !  AUTHOR:
  !
  !    ORIGINAL FORTRAN77 VERSION BY MICHAEL WICHURA.
  !    FORTRAN90 VERSION BY JOHN BURKARDT.
  !
  !  REFERENCE:
  !
  !    MICHAEL WICHURA,
  !    THE PERCENTAGE POINTS OF THE NORMAL DISTRIBUTION,
  !    ALGORITHM AS 241,
  !    APPLIED STATISTICS,
  !    VOLUME 37, NUMBER 3, PAGES 477-484, 1988.
  !
  !  PARAMETERS:
  !
  !    INPUT, REAL ( KIND = 8 ) P, THE VALUE OF THE CUMULATIVE PROBABILITY 
  !    DENSITITY FUNCTION.  0 < P < 1.  IF P IS OUTSIDE THIS RANGE,
  !    AN "INFINITE" VALUE WILL BE RETURNED.
  !
  !    OUTPUT, REAL ( KIND = 8 ) D_NORMAL_01_CDF_INVERSE, THE NORMAL DEVIATE 
  !    VALUE WITH THE PROPERTY THAT THE PROBABILITY OF A STANDARD NORMAL 
  !    DEVIATE BEING LESS THAN OR EQUAL TO THE VALUE IS P.
  !
  IMPLICIT NONE

  REAL ( KIND = 8 ), PARAMETER, DIMENSION ( 8 ) :: A = (/ &
       3.3871328727963666080D+00, &
       1.3314166789178437745D+02, &
       1.9715909503065514427D+03, &
       1.3731693765509461125D+04, &
       4.5921953931549871457D+04, &
       6.7265770927008700853D+04, &
       3.3430575583588128105D+04, &
       2.5090809287301226727D+03 /)
  REAL ( KIND = 8 ), PARAMETER, DIMENSION ( 8 ) :: B = (/ &
       1.0D+00, &
       4.2313330701600911252D+01, &
       6.8718700749205790830D+02, &
       5.3941960214247511077D+03, &
       2.1213794301586595867D+04, &
       3.9307895800092710610D+04, &
       2.8729085735721942674D+04, &
       5.2264952788528545610D+03 /)
  REAL   ( KIND = 8 ), PARAMETER, DIMENSION ( 8 ) :: C = (/ &
       1.42343711074968357734D+00, &
       4.63033784615654529590D+00, &
       5.76949722146069140550D+00, &
       3.64784832476320460504D+00, &
       1.27045825245236838258D+00, &
       2.41780725177450611770D-01, &
       2.27238449892691845833D-02, &
       7.74545014278341407640D-04 /)
  REAL ( KIND = 8 ), PARAMETER :: CONST1 = 0.180625D+00
  REAL ( KIND = 8 ), PARAMETER :: CONST2 = 1.6D+00
  REAL ( KIND = 8 ), PARAMETER, DIMENSION ( 8 ) :: D = (/ &
       1.0D+00, &
       2.05319162663775882187D+00, &
       1.67638483018380384940D+00, &
       6.89767334985100004550D-01, &
       1.48103976427480074590D-01, &
       1.51986665636164571966D-02, &
       5.47593808499534494600D-04, &
       1.05075007164441684324D-09 /)
  REAL ( KIND = 8 ), PARAMETER, DIMENSION ( 8 ) :: E = (/ &
       6.65790464350110377720D+00, &
       5.46378491116411436990D+00, &
       1.78482653991729133580D+00, &
       2.96560571828504891230D-01, &
       2.65321895265761230930D-02, &
       1.24266094738807843860D-03, &
       2.71155556874348757815D-05, &
       2.01033439929228813265D-07 /)
  REAL ( KIND = 8 ), PARAMETER, DIMENSION ( 8 ) :: F = (/ &
       1.0D+00, &
       5.99832206555887937690D-01, &
       1.36929880922735805310D-01, &
       1.48753612908506148525D-02, &
       7.86869131145613259100D-04, &
       1.84631831751005468180D-05, &
       1.42151175831644588870D-07, &
       2.04426310338993978564D-15 /)
  REAL ( KIND = 8 ) P
  REAL ( KIND = 8 ) Q
  REAL ( KIND = 8 ) R
  REAL ( KIND = 8 ) NORMDEV
  ! REAL ( KIND = 8 ),EXTERNAL:: R8POLY_VALUE
  REAL ( KIND = 8 ), PARAMETER :: SPLIT1 = 0.425D+00
  REAL ( KIND = 8 ), PARAMETER :: SPLIT2 = 5.0D+00

  IF ( P <= 0.0D+00 ) THEN
     NORMDEV = - HUGE ( P )
     RETURN
  END IF

  IF ( 1.0D+00 <= P ) THEN
     NORMDEV = HUGE ( P )
     RETURN
  END IF

  Q = P - 0.5D+00

  IF ( ABS ( Q ) <= SPLIT1 ) THEN

     R = CONST1 - Q * Q
     NORMDEV = Q * R8POLY_VALUE ( 8, A, R ) &
          / R8POLY_VALUE ( 8, B, R )

  ELSE

     IF ( Q < 0.0D+00 ) THEN
        R = P
     ELSE
        R = 1.0D+00 - P
     END IF

     IF ( R <= 0.0D+00 ) THEN
        NORMDEV = - 1.0D+00
        STOP
     END IF

     R = SQRT ( -LOG ( R ) )

     IF ( R <= SPLIT2 ) THEN

        R = R - CONST2
        NORMDEV = R8POLY_VALUE ( 8, C, R ) &
             / R8POLY_VALUE ( 8, D, R )

     ELSE

        R = R - SPLIT2
        NORMDEV = R8POLY_VALUE ( 8, E, R ) &
             / R8POLY_VALUE ( 8, F, R )

     END IF

     IF ( Q < 0.0D+00 ) THEN
        NORMDEV = - NORMDEV
     END IF

  END IF

  RETURN

CONTAINS

  FUNCTION R8POLY_VALUE ( N, A, X )

    !*****************************************************************************80
    !
    !! R8POLY_VALUE EVALUATES AN R8POLY
    !
    !  DISCUSSION:
    !
    !    FOR SANITY'S SAKE, THE VALUE OF N INDICATES THE NUMBER OF 
    !    COEFFICIENTS, OR MORE PRECISELY, THE ORDER OF THE POLYNOMIAL,
    !    RATHER THAN THE DEGREE OF THE POLYNOMIAL.  THE TWO QUANTITIES
    !    DIFFER BY 1, BUT CAUSE A GREAT DEAL OF CONFUSION.
    !
    !    GIVEN N AND A, THE FORM OF THE POLYNOMIAL IS:
    !
    !      P(X) = A(1) + A(2) * X + ... + A(N-1) * X^(N-2) + A(N) * X^(N-1)
    !
    !  LICENSING:
    !
    !    THIS CODE IS DISTRIBUTED UNDER THE GNU LGPL LICENSE.
    !
    !  MODIFIED:
    !
    !    13 AUGUST 2004
    !
    !  AUTHOR:
    !
    !    JOHN BURKARDT
    !
    !  PARAMETERS:
    !
    !    INPUT, INTEGER ( KIND = 4 ) N, THE ORDER OF THE POLYNOMIAL.
    !
    !    INPUT, REAL ( KIND = 8 ) A(N), THE COEFFICIENTS OF THE POLYNOMIAL.
    !    A(1) IS THE CONSTANT TERM.
    !
    !    INPUT, REAL ( KIND = 8 ) X, THE POINT AT WHICH THE POLYNOMIAL IS 
    !    TO BE EVALUATED.
    !
    !    OUTPUT, REAL ( KIND = 8 ) R8POLY_VALUE, THE VALUE OF THE POLYNOMIAL AT X.
    !
    IMPLICIT NONE

    INTEGER ( KIND = 4 ) N

    REAL ( KIND = 8 ) A(N)
    INTEGER ( KIND = 4 ) I
    REAL ( KIND = 8 ) R8POLY_VALUE
    REAL ( KIND = 8 ) X

    R8POLY_VALUE = 0.0D+00
    DO I = N, 1, -1
       R8POLY_VALUE = R8POLY_VALUE * X + A(I)
    END DO

    RETURN
    
  END FUNCTION R8POLY_VALUE

END FUNCTION NORMDEV



SUBROUTINE PNORM_VEC(X,U,N)
  REAL(KIND=8) :: X(N),U(N)
  INTEGER::N
  U = .5 + .5 * ERF(X/SQRT(2.0D+00))
END SUBROUTINE PNORM_VEC



SUBROUTINE PNORM(X,U)
  REAL(KIND=8) :: X,U
  U = 0.5 + 0.5 * ERF(X/SQRT(2.0D+00))  
END SUBROUTINE PNORM



SUBROUTINE QNORM_VEC(U,X,N)
  REAL(KIND=8) :: U(N),X(N)
  INTEGER :: N

  INTEGER :: I
  REAL(KIND=8) , EXTERNAL :: NORMDEV
  
  DO I = 1,N
     X(I) = NORMDEV(U(I))
  END DO
  
END SUBROUTINE QNORM_VEC


SUBROUTINE QNORM(U,X)
  
  REAL(KIND=8) :: U,X

  REAL(KIND=8) , EXTERNAL :: NORMDEV

  X = NORMDEV(U)

END SUBROUTINE QNORM


FUNCTION T_QUANTILE(P, NDF, NORMDEV) RESULT(FN_VAL)

  !  CALCULATES THE TWO-TAIL QUANTILES OF STUDENT'S T-DISTRIBUTION.
  !  THE USER MUST SUPPLY A FUNCTION `NORMDEV' TO RETURN QUANTILES OF THE
  !  THE NORMAL DISTRIBUTION, SUCH AS APPLIED STATISTICS ALGORITHM AS 241.

  !  TRANSLATION FROM ALGOL BY ALAN MILLER, CSIRO DIVISION OF MATHEMATICS
  !  & STATISTICS, CLAYTON, VICTORIA 3169, AUSTRALIA OF:

  !  ALGORITHM 396: STUDENT'S T-QUANTILES BY G.W. HILL
  !  COMM. A.C.M., VOL.13(10), 619-620, OCTOBER 1970

  !  N.B. THE NUMBER OF DEGREES OF FREEDOM IS NOT NECESSARILY AN INTEGER

  !  ARGUMENTS:
  !  P       DOUBLE PRECISION    AREA UNDER BOTH TAILS
  !  NDF     DOUBLE PRECISION    NUMBER OF DEGREES OF FREEDOM
  !  NORMAL  DOUBLE PRECISION    EXTERNAL FUNCTION TO RETURN QUANTILES
  !                              OF THE STANDARD NORMAL CURVE
  !  IER     INTEGER             ERROR INDICATOR = 0 FOR NORMAL EXIT
  !                                              = 1 IF NDF < 1

  !  ELF90-COMPATIBLE VERSION BY ALAN MILLER
  !  LATEST REVISION - 7 AUGUST 1997

  IMPLICIT NONE
  ! INTEGER, PARAMETER    :: DP = SELECTED_REAL_KIND(12, 60)
  REAL (KIND=8) :: P, NDF
  REAL (KIND=8) :: FN_VAL

  INTERFACE
     FUNCTION NORMDEV(P) RESULT(X)
       IMPLICIT NONE
       ! INTEGER, PARAMETER    :: DP = SELECTED_REAL_KIND(12, 60)
       REAL (KIND=8), INTENT(IN) :: P
       REAL (KIND=8)             :: X
     END FUNCTION NORMDEV
  END INTERFACE

  !     LOCAL VARIABLES

  REAL (KIND=8) , EXTERNAL :: STUDENT,NORMAL
  REAL (KIND=8)            :: A, B, C, D, PROB, X, Y , TDT , TPT , TZT
  REAL (KIND=8), PARAMETER :: HALF_PI = 1.5707963267949D0, EPS = 1.D-12, ONE = 1.D0

  IF (NDF < ONE)THEN
     WRITE(*,*)  'ERROR IN T_QUANTILE: NDF<1: NDF = ', NDF
     STOP
  ELSE IF(P > ONE)THEN
     WRITE(*,*)  'ERROR IN T_QUANTILE: P > 1'
     STOP
  ELSE IF(P <= 0D0)THEN
     WRITE(*,*)  'ERROR IN T_QUANTILE: P<= 0'
     STOP
  END IF

  IF ( ABS(NDF - 2.D0) < EPS ) THEN
     FN_VAL = SQRT(2.D0 / ( P * (2.D0 - P) ) - 2.D0)
     RETURN

  ELSE IF (NDF < ONE+EPS) THEN
     PROB = P * HALF_PI
     FN_VAL = COS(PROB) / SIN(PROB)
     RETURN

  ELSE
     A = ONE / (NDF - 0.5D0)
     B = 48.D0 / A**2
     C = ((20700.D0 * A / B - 98.D0) * A - 16.D0) * A + 96.36D0
     D = ((94.5D0 / (B + C) - 3.D0) / B + ONE) * SQRT(A * HALF_PI)* NDF
     X = D * P
     Y = X ** (2.D0 / NDF)

     IF (Y > 0.05D0 + A) THEN

        !     ASYMPTOTIC INVERSE EXPANSION ABOUT NORMAL

        X = NORMDEV(0.5 * P)
        Y = X**2
        IF (NDF < 5.D0) C = C + 0.3D0 * (NDF - 4.5D0) * (X + 0.6D0)
        C = (((0.05D0 * D * X - 5.D0) * X - 7.D0) * X - 2.D0) * X + B+ C
        Y = (((((0.4D0*Y + 6.3D0) * Y + 36.D0) * Y + 94.5D0) / C - Y  &
             - 3.D0) / B + ONE) * X
        Y = A * Y**2
        ! IF (Y > 0.002D0) THEN
        !    Y = EXP(Y) - ONE
        ! ELSE
        !    Y = 0.5D0 * Y**2 + Y
        ! END IF
        !!! THE AMENDMENT HERE IS ACCORDING TO "REMARK ON ALGORITHM 396"
        IF(Y.GT.1.0D-01)THEN
           Y = EXP(Y) - ONE
        ELSE
           Y = ((Y+4.0D00) * Y + 12.0D00) * Y * Y / 24.0D00 + Y
        END IF
     ELSE

        Y = ((ONE / (((NDF + 6.D0) / (NDF * Y) - 0.089D0 * D -  &
             0.822D0) * (NDF + 2.D0) * 3.D0) + 0.5D0 / (NDF + 4.D0))  &
             * Y - ONE) * (NDF + ONE) / (NDF + 2.D0) + ONE / Y
     END IF
  END IF
  FN_VAL = SQRT(NDF * Y)

  ! 	- APPLY 2-TERM TAYLOR EXPANSION AS IN
  !       HILL, G.W (1981) "REMARK ON ALGO.396", ACM TOMS 7, 250-1
  CALL DT(FN_VAL,TDT,NDF)
  TPT = STUDENT(FN_VAL,NDF,NORMAL)
  TZT = 0.5D00*(TPT-P) / TDT

  FN_VAL = (NDF+1.0D00) * FN_VAL * TZT * TZT * 0.5D00 &
       / (FN_VAL * FN_VAL + NDF) + TZT + FN_VAL

  RETURN
END FUNCTION T_QUANTILE


SUBROUTINE QT(U,X,DF)


  ! INTEGER, PARAMETER    :: DP = SELECTED_REAL_KIND(12, 60)
  REAL(KIND=8) :: U,X,DF  
  REAL(KIND=8) , EXTERNAL :: T_QUANTILE , NORMDEV

  IF(U .LT. 0.5D00)THEN
     X = - T_QUANTILE(U*2.0D00,DF,NORMDEV)
  ELSE
     X = T_QUANTILE(2.0D00-U*2.0D00,DF,NORMDEV)
  END IF

END SUBROUTINE QT


SUBROUTINE QT_VEC(U,X,DF,N)

  REAL(KIND=8) :: U(N),X(N),DF
  REAL(KIND=8) , EXTERNAL :: T_QUANTILE , NORMDEV
  INTEGER :: N,I

  DO I = 1,N

     IF(U(I) .LT. 0.5D00)THEN
        X(I) = - T_QUANTILE(U(I)*2.0D00,DF,NORMDEV)
     ELSE
        X(I) = T_QUANTILE(2.0D00-U(I)*2.0D00,DF,NORMDEV)
     END IF
  END DO

END SUBROUTINE QT_VEC



FUNCTION STUDENT(T, NDF, NORMAL) RESULT(FN_VAL)

  !  CALCULATES THE TWO-TAILED PROBABILITY OF STUDENT'S T.
  !  THE USER MUST SUPPLY A FUNCTION `NORMAL' TO CALCULATE THE AREA UNDER THE
  !  NORMAL DISTRIBUTION CURVE, SUCH AS APPLIED STATISTICS ALGORITHM AS 66.

  !  TRANSLATION FROM ALGOL BY ALAN MILLER, CSIRO DIVISION OF MATHEMATICS
  !  & STATISTICS, CLAYTON, VICTORIA 3169, AUSTRALIA OF:

  !  ALGORITHM 395: STUDENT'S T-DISTRIBUTION BY G.W. HILL
  !  COMM. A.C.M., VOL.13(10), 617-619, OCTOBER 1970

  !  N.B. THE NUMBER OF DEGREES OF FREEDOM IS NOT NECESSARILY AN INTEGER

  !  ARGUMENTS:
  !  T       DOUBLE PRECISION    VALUE OF STUDENT'S T
  !  NDF     DOUBLE PRECISION    NUMBER OF DEGREES OF FREEDOM
  !  NORMAL  DOUBLE PRECISION    EXTERNAL FUNCTION TO RETURN THE
  !                              AREA UNDER THE STANDARD NORMAL CURVE
  !  IER     INTEGER             ERROR INDICATOR = 0 FOR NORMAL EXIT
  !                                              = 1 IF NDF < 1

  !  ELF90-COMPATIBLE VERSION BY ALAN MILLER
  !  LATEST REVISION - 7 AUGUST 1997

  IMPLICIT NONE
  ! INTEGER, PARAMETER    :: DP = SELECTED_REAL_KIND(12, 60)
  REAL (KIND=8), INTENT(IN) :: T, NDF
  REAL (KIND=8)             :: FN_VAL

  INTERFACE
     FUNCTION NORMAL(X) RESULT(FX)
       IMPLICIT NONE
       REAL (KIND=8), INTENT(IN) :: X
       REAL (KIND=8)             :: FX
     END FUNCTION NORMAL
  END INTERFACE

  !     LOCAL VARIABLES

  REAL (KIND=8)            :: A, B, N, T2, Y, Z
  REAL (KIND=8), PARAMETER :: EPS = 1.D-12, ONE = 1.D0, TWOONPI = 0.63661977236758D0
  INTEGER              :: J

  IF (NDF < ONE) THEN
     WRITE(*, *) 'ERROR IN STUDENT: NUMBER OF DEGREES OF FREEDOM < 1'
     RETURN
  END IF

  T2 = T*T
  Y = T2/NDF
  B = ONE + Y
  Z = ONE

  IF ( (NDF - INT(NDF)) > EPS .OR. (NDF > 20.D0 - EPS .AND. NDF > T2)  &
       .OR.  NDF > 200.D0 ) THEN

     !     ASYMPTOTIC SERIES FOR LARGE OR NON-INTEGER NDF

     IF (Y > 1.D-06) Y = LOG(B)
     A = NDF - 0.5D0
     B = 48.D0 * A**2
     Y = A * Y
     Y = (((((-0.4D0*Y - 3.3D0)*Y - 24.D0)*Y - 85.5D0) /  &
          (0.8D0*Y**2 + 100.D0 + B) + Y +3.D0) / B + ONE) * SQRT(Y)
     ! ZFYUAN
     FN_VAL = 2.D0 * NORMAL(-Y)
     RETURN

  ELSE IF ( NDF < 20.D0 .AND. T2 < 4.D0) THEN

     !     NESTED SUMMATION OF COSINE SERIES

     Y = SQRT(Y)
     A = Y
     IF (NDF == ONE) A = 0.D0
     N = NDF

     !     TAIL SERIES EXPANSION FOR LARGE T-VALUES

  ELSE
     A = SQRT(B)
     Y = A * NDF
     J = 0
10   J = J + 2
     IF (ABS(A-Z) > EPS) THEN
        Z = A
        Y = Y * (J - 1) / (B * J)
        A = A + Y / (NDF + J)
        GO TO 10
     END IF
     N = NDF + 2.D0
     Z = 0.D0
     Y = 0.D0
     A = - A
  END IF

  !     THIS IS THE `LOOP' FROM THE ALGOL CODE.
  !     IT REQUIRED JUMPING BETWEEN DIFFERENT PARTS OF AN IF () THEN
  !     ELSE IF () .. BLOCK.

20 N = N - 2.D0
  IF (N > 1.D0) THEN
     A = (N - 1.D0) * A / (B * N) + Y
     GO TO 20
  END IF

  IF ( ABS(N) < EPS ) THEN
     A = A / SQRT(B)
  ELSE
     A = (ATAN(Y) + A/B) * TWOONPI
  END IF
  FN_VAL = Z - A
  RETURN
END FUNCTION STUDENT


SUBROUTINE PT(X,U,NU)

  IMPLICIT NONE

  REAL(KIND=8) :: X,U,NU
  REAL(KIND=8) , EXTERNAL :: STUDENT,NORMAL

  IF(X.LE.0)THEN
     U = STUDENT(-X,NU,NORMAL) / 2.0D00
  ELSE
     U = 1.0D00 - STUDENT(X,NU,NORMAL) / 2.0D00
  END IF

END SUBROUTINE PT

SUBROUTINE PT_VEC(X,U,NU,N)
  IMPLICIT NONE

  REAL(KIND=8) :: X(N),U(N),NU
  REAL(KIND=8) , EXTERNAL :: STUDENT,NORMAL
  INTEGER :: N,I

  DO I = 1,N
     IF(X(I).LE.0)THEN
        U(I) = STUDENT(-X(I),NU,NORMAL) / 2.0D00
     ELSE
        U(I) = 1.0D00 - STUDENT(X(I),NU,NORMAL) / 2.0D00
     END IF
  END DO

END SUBROUTINE PT_VEC


SUBROUTINE DT(X,F_X,NU)

  IMPLICIT NONE

  REAL(KIND=8) :: X,F_X,NU
  REAL(KIND=8) , PARAMETER :: PI = 3.1415926535897932

  F_X = EXP(LOG_GAMMA(NU/2.0D00+0.5D00)-LOG_GAMMA(NU/2.0D00)) &
       / SQRT(PI * NU) &
       * ( 1 + X ** 2.0D00 / NU ) ** (-(NU+1.0D00)/2.0D00)

END SUBROUTINE DT



SUBROUTINE DT_VEC(X,F_X,NU,N)

  IMPLICIT NONE

  REAL(KIND=8) :: X(N),F_X(N),NU
  REAL(KIND=8) , PARAMETER :: PI = 3.1415926535897932
  INTEGER :: N,I

  DO I = 1,N
     F_X(I) = EXP(LOG_GAMMA(NU/2.0D00+0.5D00)-LOG_GAMMA(NU/2.0D00)) &
          / SQRT(PI * NU) &
          * ( 1 + X(I) ** 2.0D00 / NU ) ** (-(NU+1.0D00)/2.0D00)
  END DO

END SUBROUTINE DT_VEC

