SUBROUTINE QAGS ( F, A, B, EPSABS, EPSREL, RESULT, ABSERR, NEVAL, IER, BB1,BB2,NU )

!*****************************************************************************80
!
!! QAGS ESTIMATES THE INTEGRAL OF A FUNCTION.
!
!  DISCUSSION:
!
!    THE ROUTINE CALCULATES AN APPROXIMATION RESULT TO A DEFINITE INTEGRAL   
!      I = INTEGRAL OF F OVER (A,B),
!    HOPEFULLY SATISFYING
!      || I - RESULT || <= MAX ( EPSABS, EPSREL * ||I|| ).
!
!  AUTHOR:
!
!    ROBERT PIESSENS, ELISE DE DONCKER-KAPENGER, 
!    CHRISTIAN UEBERHUBER, DAVID KAHANER
!
!  REFERENCE:
!
!    ROBERT PIESSENS, ELISE DE DONCKER-KAPENGER, 
!    CHRISTIAN UEBERHUBER, DAVID KAHANER,
!    QUADPACK, A SUBROUTINE PACKAGE FOR AUTOMATIC INTEGRATION,
!    SPRINGER VERLAG, 1983
!
!  PARAMETERS:
!
!    INPUT, EXTERNAL REAL(KIND=8) F, THE NAME OF THE FUNCTION ROUTINE, OF THE FORM
!      FUNCTION F ( X )
!      REAL(KIND=8) F
!      REAL(KIND=8) X
!    WHICH EVALUATES THE INTEGRAND FUNCTION.
!
!    INPUT, REAL(KIND=8) A, B, THE LIMITS OF INTEGRATION.
!
!    INPUT, REAL(KIND=8) EPSABS, EPSREL, THE ABSOLUTE AND RELATIVE ACCURACY REQUESTED.
!
!    OUTPUT, REAL(KIND=8) RESULT, THE ESTIMATED VALUE OF THE INTEGRAL.
!
!    OUTPUT, REAL(KIND=8) ABSERR, AN ESTIMATE OF || I - RESULT ||.
!
!    OUTPUT, INTEGER NEVAL, THE NUMBER OF TIMES THE INTEGRAL WAS EVALUATED.
!
!    OUTPUT, INTEGER IER, ERROR FLAG.
!                     IER = 0 NORMAL AND RELIABLE TERMINATION OF THE
!                             ROUTINE. IT IS ASSUMED THAT THE REQUESTED
!                             ACCURACY HAS BEEN ACHIEVED.
!                     IER > 0 ABNORMAL TERMINATION OF THE ROUTINE
!                             THE ESTIMATES FOR INTEGRAL AND ERROR ARE
!                             LESS RELIABLE. IT IS ASSUMED THAT THE
!                             REQUESTED ACCURACY HAS NOT BEEN ACHIEVED.
!                         = 1 MAXIMUM NUMBER OF SUBDIVISIONS ALLOWED
!                             HAS BEEN ACHIEVED. ONE CAN ALLOW MORE SUB-
!                             DIVISIONS BY INCREASING THE DATA VALUE OF
!                             LIMIT IN QAGS (AND TAKING THE ACCORDING
!                             DIMENSION ADJUSTMENTS INTO ACCOUNT).
!                             HOWEVER, IF THIS YIELDS NO IMPROVEMENT
!                             IT IS ADVISED TO ANALYZE THE INTEGRAND
!                             IN ORDER TO DETERMINE THE INTEGRATION
!                             DIFFICULTIES. IF THE POSITION OF A
!                             LOCAL DIFFICULTY CAN BE DETERMINED (E.G.
!                             SINGULARITY, DISCONTINUITY WITHIN THE
!                             INTERVAL) ONE WILL PROBABLY GAIN FROM
!                             SPLITTING UP THE INTERVAL AT THIS POINT
!                             AND CALLING THE INTEGRATOR ON THE SUB-
!                             RANGES. IF POSSIBLE, AN APPROPRIATE
!                             SPECIAL-PURPOSE INTEGRATOR SHOULD BE USED,
!                             WHICH IS DESIGNED FOR HANDLING THE TYPE
!                             OF DIFFICULTY INVOLVED.
!                         = 2 THE OCCURRENCE OF ROUNDOFF ERROR IS DETEC-
!                             TED, WHICH PREVENTS THE REQUESTED
!                             TOLERANCE FROM BEING ACHIEVED.
!                             THE ERROR MAY BE UNDER-ESTIMATED.
!                         = 3 EXTREMELY BAD INTEGRAND BEHAVIOR OCCURS
!                             AT SOME  POINTS OF THE INTEGRATION
!                             INTERVAL.
!                         = 4 THE ALGORITHM DOES NOT CONVERGE. ROUNDOFF
!                             ERROR IS DETECTED IN THE EXTRAPOLATION
!                             TABLE. IT IS PRESUMED THAT THE REQUESTED
!                             TOLERANCE CANNOT BE ACHIEVED, AND THAT THE
!                             RETURNED RESULT IS THE BEST WHICH CAN BE
!                             OBTAINED.
!                         = 5 THE INTEGRAL IS PROBABLY DIVERGENT, OR
!                             SLOWLY CONVERGENT. IT MUST BE NOTED THAT
!                             DIVERGENCE CAN OCCUR WITH ANY OTHER VALUE
!                             OF IER.
!                         = 6 THE INPUT IS INVALID, BECAUSE
!                             EPSABS < 0 AND EPSREL < 0,
!                             RESULT, ABSERR AND NEVAL ARE SET TO ZERO.
!
!  LOCAL PARAMETERS:
!
!           ALIST     - LIST OF LEFT END POINTS OF ALL SUBINTERVALS
!                       CONSIDERED UP TO NOW
!           BLIST     - LIST OF RIGHT END POINTS OF ALL SUBINTERVALS
!                       CONSIDERED UP TO NOW
!           RLIST(I)  - APPROXIMATION TO THE INTEGRAL OVER
!                       (ALIST(I),BLIST(I))
!           RLIST2    - ARRAY OF DIMENSION AT LEAST LIMEXP+2 CONTAINING
!                       THE PART OF THE EPSILON TABLE WHICH IS STILL
!                       NEEDED FOR FURTHER COMPUTATIONS
!           ELIST(I)  - ERROR ESTIMATE APPLYING TO RLIST(I)
!           MAXERR    - POINTER TO THE INTERVAL WITH LARGEST ERROR
!                       ESTIMATE
!           ERRMAX    - ELIST(MAXERR)
!           ERLAST    - ERROR ON THE INTERVAL CURRENTLY SUBDIVIDED
!                       (BEFORE THAT SUBDIVISION HAS TAKEN PLACE)
!           AREA      - SUM OF THE INTEGRALS OVER THE SUBINTERVALS
!           ERRSUM    - SUM OF THE ERRORS OVER THE SUBINTERVALS
!           ERRBND    - REQUESTED ACCURACY MAX(EPSABS,EPSREL*
!                       ABS(RESULT))
!           *****1    - VARIABLE FOR THE LEFT INTERVAL
!           *****2    - VARIABLE FOR THE RIGHT INTERVAL
!           LAST      - INDEX FOR SUBDIVISION
!           NRES      - NUMBER OF CALLS TO THE EXTRAPOLATION ROUTINE
!           NUMRL2    - NUMBER OF ELEMENTS CURRENTLY IN RLIST2. IF AN
!                       APPROPRIATE APPROXIMATION TO THE COMPOUNDED
!                       INTEGRAL HAS BEEN OBTAINED IT IS PUT IN
!                       RLIST2(NUMRL2) AFTER NUMRL2 HAS BEEN INCREASED
!                       BY ONE.
!           SMALL     - LENGTH OF THE SMALLEST INTERVAL CONSIDERED
!                       UP TO NOW, MULTIPLIED BY 1.5
!           ERLARG    - SUM OF THE ERRORS OVER THE INTERVALS LARGER
!                       THAN THE SMALLEST INTERVAL CONSIDERED UP TO NOW
!           EXTRAP    - LOGICAL VARIABLE DENOTING THAT THE ROUTINE IS
!                       ATTEMPTING TO PERFORM EXTRAPOLATION I.E. BEFORE
!                       SUBDIVIDING THE SMALLEST INTERVAL WE TRY TO
!                       DECREASE THE VALUE OF ERLARG.
!           NOEXT     - LOGICAL VARIABLE DENOTING THAT EXTRAPOLATION
!                       IS NO LONGER ALLOWED (TRUE VALUE)
!
  IMPLICIT NONE

  INTERFACE
     FUNCTION F(X,BB1,BB2,NU)
       IMPLICIT NONE
       REAL(KIND=8) :: X, BB1, BB2,NU,F
     END FUNCTION F
  END INTERFACE
  REAL(KIND=8) :: BB1,BB2,NU

  INTEGER, PARAMETER :: LIMIT = 500

  REAL(KIND=8) A
  REAL(KIND=8) ABSEPS
  REAL(KIND=8) ABSERR
  REAL(KIND=8) ALIST(LIMIT)
  REAL(KIND=8) AREA
  REAL(KIND=8) AREA1
  REAL(KIND=8) AREA12
  REAL(KIND=8) AREA2
  REAL(KIND=8) A1
  REAL(KIND=8) A2
  REAL(KIND=8) B
  REAL(KIND=8) BLIST(LIMIT)
  REAL(KIND=8) B1
  REAL(KIND=8) B2
  REAL(KIND=8) CORREC
  REAL(KIND=8) DEFABS
  REAL(KIND=8) DEFAB1
  REAL(KIND=8) DEFAB2
  REAL(KIND=8) DRES
  REAL(KIND=8) ELIST(LIMIT)
  REAL(KIND=8) EPSABS
  REAL(KIND=8) EPSREL
  REAL(KIND=8) ERLARG
  REAL(KIND=8) ERLAST
  REAL(KIND=8) ERRBND
  REAL(KIND=8) ERRMAX
  REAL(KIND=8) ERROR1
  REAL(KIND=8) ERROR2
  REAL(KIND=8) ERRO12
  REAL(KIND=8) ERRSUM
  REAL(KIND=8) ERTEST
  LOGICAL EXTRAP
  ! REAL(KIND=8), EXTERNAL :: F  ! ZFYUAN COMMENTED. FRI DEC 28, 19:49, 2012
  INTEGER ID
  INTEGER IER
  INTEGER IERRO
  INTEGER IORD(LIMIT)
  INTEGER IROFF1
  INTEGER IROFF2
  INTEGER IROFF3
  INTEGER JUPBND
  INTEGER K
  INTEGER KSGN
  INTEGER KTMIN
  INTEGER LAST
  LOGICAL NOEXT
  INTEGER MAXERR
  INTEGER NEVAL
  INTEGER NRES
  INTEGER NRMAX
  INTEGER NUMRL2
  REAL(KIND=8) RESABS
  REAL(KIND=8) RESEPS
  REAL(KIND=8) RESULT
  REAL(KIND=8) RES3LA(3)
  REAL(KIND=8) RLIST(LIMIT)
  REAL(KIND=8) RLIST2(52)
  REAL(KIND=8) SMALL
!
!  THE DIMENSION OF RLIST2 IS DETERMINED BY THE VALUE OF
!  LIMEXP IN QEXTR (RLIST2 SHOULD BE OF DIMENSION
!  (LIMEXP+2) AT LEAST).
!
!  TEST ON VALIDITY OF PARAMETERS.
!
  IER = 0
  NEVAL = 0
  LAST = 0
  RESULT = 0.0D+00
  ABSERR = 0.0D+00
  ALIST(1) = A
  BLIST(1) = B
  RLIST(1) = 0.0D+00
  ELIST(1) = 0.0D+00

  IF ( EPSABS < 0.0D+00 .AND. EPSREL < 0.0D+00 ) THEN
    IER = 6
    RETURN
  END IF
!
!  FIRST APPROXIMATION TO THE INTEGRAL.
!
  IERRO = 0
  CALL QK21 ( F, A, B, RESULT, ABSERR, DEFABS, RESABS, BB1,BB2,NU )
!
!  TEST ON ACCURACY.
!
  DRES = ABS ( RESULT )
  ERRBND = MAX ( EPSABS, EPSREL * DRES )
  LAST = 1
  RLIST(1) = RESULT
  ELIST(1) = ABSERR
  IORD(1) = 1

  IF ( ABSERR <= 1.0E+02 * EPSILON ( DEFABS ) * DEFABS .AND. &
    ABSERR > ERRBND ) THEN
    IER = 2
  END IF

  IF ( LIMIT == 1 ) THEN
    IER = 1
  END IF

  IF ( IER /= 0 .OR. (ABSERR <= ERRBND .AND. ABSERR /= RESABS ) .OR. &
    ABSERR == 0.0D+00 ) GO TO 140
!
!  INITIALIZATION.
!
  RLIST2(1) = RESULT
  ERRMAX = ABSERR
  MAXERR = 1
  AREA = RESULT
  ERRSUM = ABSERR
  ABSERR = HUGE ( ABSERR )
  NRMAX = 1
  NRES = 0
  NUMRL2 = 2
  KTMIN = 0
  EXTRAP = .FALSE.
  NOEXT = .FALSE.
  IROFF1 = 0
  IROFF2 = 0
  IROFF3 = 0

  IF ( DRES >= (1.0D+00-5.0E+01* EPSILON ( DEFABS ) ) * DEFABS ) THEN
    KSGN = 1
  ELSE
    KSGN = -1
  END IF

  DO LAST = 2, LIMIT
!
!  BISECT THE SUBINTERVAL WITH THE NRMAX-TH LARGEST ERROR ESTIMATE.
!
    A1 = ALIST(MAXERR)
    B1 = 5.0E-01 * ( ALIST(MAXERR) + BLIST(MAXERR) )
    A2 = B1
    B2 = BLIST(MAXERR)
    ERLAST = ERRMAX
    ! CALL QK21 ( F, A1, B1, AREA1, ERROR1, RESABS, DEFAB1 )
    ! CALL QK21 ( F, A2, B2, AREA2, ERROR2, RESABS, DEFAB2 )
    CALL QK21 ( F, A1, B1, AREA1, ERROR1, RESABS, DEFAB1 ,BB1,BB2,NU)
    CALL QK21 ( F, A2, B2, AREA2, ERROR2, RESABS, DEFAB2 ,BB1,BB2,NU)
!
!  IMPROVE PREVIOUS APPROXIMATIONS TO INTEGRAL AND ERROR
!  AND TEST FOR ACCURACY.
!
    AREA12 = AREA1+AREA2
    ERRO12 = ERROR1+ERROR2
    ERRSUM = ERRSUM+ERRO12-ERRMAX
    AREA = AREA+AREA12-RLIST(MAXERR)

    IF ( DEFAB1 == ERROR1 .OR. DEFAB2 == ERROR2 ) GO TO 15

    IF ( ABS ( RLIST(MAXERR) - AREA12) > 1.0E-05 * ABS(AREA12) &
      .OR. ERRO12 < 9.9E-01 * ERRMAX ) GO TO 10

    IF ( EXTRAP ) THEN
      IROFF2 = IROFF2+1
    ELSE
      IROFF1 = IROFF1+1
    END IF

10  CONTINUE

    IF ( LAST > 10 .AND. ERRO12 > ERRMAX ) THEN
      IROFF3 = IROFF3+1
    END IF

15  CONTINUE

    RLIST(MAXERR) = AREA1
    RLIST(LAST) = AREA2
    ERRBND = MAX ( EPSABS, EPSREL*ABS(AREA) )
!
!  TEST FOR ROUNDOFF ERROR AND EVENTUALLY SET ERROR FLAG.
!
    IF ( IROFF1+IROFF2 >= 10 .OR. IROFF3 >= 20 ) THEN
      IER = 2
    END IF

    IF ( IROFF2 >= 5 ) THEN
      IERRO = 3
    END IF
!
!  SET ERROR FLAG IN THE CASE THAT THE NUMBER OF SUBINTERVALS
!  EQUALS LIMIT.
!
    IF ( LAST == LIMIT ) THEN
      IER = 1
    END IF
!
!  SET ERROR FLAG IN THE CASE OF BAD INTEGRAND BEHAVIOR
!  AT A POINT OF THE INTEGRATION RANGE.
!
    IF ( MAX ( ABS(A1),ABS(B2)) <= (1.0D+00+1.0E+03* EPSILON ( A1 ) )* &
      (ABS(A2)+1.0E+03* TINY ( A2 ) ) ) THEN
      IER = 4
    END IF
!
!  APPEND THE NEWLY-CREATED INTERVALS TO THE LIST.
!
    IF ( ERROR2 <= ERROR1 ) THEN
      ALIST(LAST) = A2
      BLIST(MAXERR) = B1
      BLIST(LAST) = B2
      ELIST(MAXERR) = ERROR1
      ELIST(LAST) = ERROR2
    ELSE
      ALIST(MAXERR) = A2
      ALIST(LAST) = A1
      BLIST(LAST) = B1
      RLIST(MAXERR) = AREA2
      RLIST(LAST) = AREA1
      ELIST(MAXERR) = ERROR2
      ELIST(LAST) = ERROR1
    END IF
!
!  CALL QSORT TO MAINTAIN THE DESCENDING ORDERING
!  IN THE LIST OF ERROR ESTIMATES AND SELECT THE SUBINTERVAL
!  WITH NRMAX-TH LARGEST ERROR ESTIMATE (TO BE BISECTED NEXT).
!
    CALL QSORT ( LIMIT, LAST, MAXERR, ERRMAX, ELIST, IORD, NRMAX )

    IF ( ERRSUM <= ERRBND ) GO TO 115

    IF ( IER /= 0 ) THEN
      EXIT
    END IF

    IF ( LAST == 2 ) GO TO 80
    IF ( NOEXT ) GO TO 90

    ERLARG = ERLARG-ERLAST

    IF ( ABS(B1-A1) > SMALL ) THEN
      ERLARG = ERLARG+ERRO12
    END IF
!
!  TEST WHETHER THE INTERVAL TO BE BISECTED NEXT IS THE
!  SMALLEST INTERVAL.
!
    IF ( .NOT. EXTRAP ) THEN
      IF ( ABS(BLIST(MAXERR)-ALIST(MAXERR)) > SMALL ) GO TO 90
      EXTRAP = .TRUE.
      NRMAX = 2
    END IF

!40  CONTINUE
!
!  THE SMALLEST INTERVAL HAS THE LARGEST ERROR.
!  BEFORE BISECTING DECREASE THE SUM OF THE ERRORS OVER THE
!  LARGER INTERVALS (ERLARG) AND PERFORM EXTRAPOLATION.
!
    IF ( IERRO /= 3 .AND. ERLARG > ERTEST ) THEN

      ID = NRMAX
      JUPBND = LAST

      IF ( LAST > (2+LIMIT/2) ) THEN
        JUPBND = LIMIT+3-LAST
      END IF

      DO K = ID, JUPBND
        MAXERR = IORD(NRMAX)
        ERRMAX = ELIST(MAXERR)
        IF ( ABS(BLIST(MAXERR)-ALIST(MAXERR)) > SMALL ) THEN
          GO TO 90
        END IF
        NRMAX = NRMAX+1
      END DO

    END IF
!
!  PERFORM EXTRAPOLATION.
!
!60  CONTINUE

    NUMRL2 = NUMRL2+1
    RLIST2(NUMRL2) = AREA
    CALL QEXTR ( NUMRL2, RLIST2, RESEPS, ABSEPS, RES3LA, NRES )
    KTMIN = KTMIN+1

    IF ( KTMIN > 5 .AND. ABSERR < 1.0E-03 * ERRSUM ) THEN
      IER = 5
    END IF

    IF ( ABSEPS < ABSERR ) THEN

      KTMIN = 0
      ABSERR = ABSEPS
      RESULT = RESEPS
      CORREC = ERLARG
      ERTEST = MAX ( EPSABS,EPSREL*ABS(RESEPS))

      IF ( ABSERR <= ERTEST ) THEN
        EXIT
      END IF

    END IF
!
!  PREPARE BISECTION OF THE SMALLEST INTERVAL.
!
    IF ( NUMRL2 == 1 ) THEN
      NOEXT = .TRUE.
    END IF

    IF ( IER == 5 ) THEN
      EXIT
    END IF

    MAXERR = IORD(1)
    ERRMAX = ELIST(MAXERR)
    NRMAX = 1
    EXTRAP = .FALSE.
    SMALL = SMALL * 5.0E-01
    ERLARG = ERRSUM
    GO TO 90

80  CONTINUE

    SMALL = ABS ( B - A ) * 3.75E-01
    ERLARG = ERRSUM
    ERTEST = ERRBND
    RLIST2(2) = AREA

90  CONTINUE

  END DO
!
!  SET FINAL RESULT AND ERROR ESTIMATE.
!
  IF ( ABSERR == HUGE ( ABSERR ) ) THEN
    GO TO 115
  END IF

  IF ( IER + IERRO == 0 ) THEN
    GO TO 110
  END IF

  IF ( IERRO == 3 ) THEN
    ABSERR = ABSERR + CORREC
  END IF

  IF ( IER == 0 ) THEN
    IER = 3
  END IF

  IF ( RESULT /= 0.0D+00.AND.AREA /= 0.0D+00 ) THEN
    GO TO 105
  END IF

  IF ( ABSERR > ERRSUM ) GO TO 115
  IF ( AREA == 0.0D+00 ) GO TO 130
  GO TO 110

105 CONTINUE

  IF ( ABSERR/ABS(RESULT) > ERRSUM/ABS(AREA) ) GO TO 115
!
!  TEST ON DIVERGENCE.
!
110 CONTINUE

  IF ( KSGN == (-1).AND.MAX ( ABS(RESULT),ABS(AREA)) <=  &
   DEFABS*1.0E-02 ) GO TO 130

  IF ( 1.0E-02 > (RESULT/AREA) .OR. (RESULT/AREA) > 1.0E+02 &
   .OR. ERRSUM > ABS(AREA) ) THEN
    IER = 6
  END IF

  GO TO 130
!
!  COMPUTE GLOBAL INTEGRAL SUM.
!
115 CONTINUE

  RESULT = SUM ( RLIST(1:LAST) )

  ABSERR = ERRSUM

130 CONTINUE
 
  IF ( 2 < IER ) THEN
    IER = IER - 1
  END IF

140 CONTINUE

  NEVAL = 42*LAST-21

  RETURN
END SUBROUTINE QAGS





SUBROUTINE QEXTR ( N, EPSTAB, RESULT, ABSERR, RES3LA, NRES )

!*****************************************************************************80
!
!! QEXTR CARRIES OUT THE EPSILON EXTRAPOLATION ALGORITHM.
!
!  DISCUSSION:
!
!    THE ROUTINE DETERMINES THE LIMIT OF A GIVEN SEQUENCE OF APPROXIMATIONS, 
!    BY MEANS OF THE EPSILON ALGORITHM OF P. WYNN.  AN ESTIMATE OF THE 
!    ABSOLUTE ERROR IS ALSO GIVEN.  THE CONDENSED EPSILON TABLE IS COMPUTED.
!    ONLY THOSE ELEMENTS NEEDED FOR THE COMPUTATION OF THE NEXT DIAGONAL
!    ARE PRESERVED.
!
!  AUTHOR:
!
!    ROBERT PIESSENS, ELISE DE DONCKER-KAPENGER, 
!    CHRISTIAN UEBERHUBER, DAVID KAHANER
!
!  REFERENCE:
!
!    ROBERT PIESSENS, ELISE DE DONCKER-KAPENGER, 
!    CHRISTIAN UEBERHUBER, DAVID KAHANER,
!    QUADPACK, A SUBROUTINE PACKAGE FOR AUTOMATIC INTEGRATION,
!    SPRINGER VERLAG, 1983
!
!  PARAMETERS:
!
!    INPUT, INTEGER N, INDICATES THE ENTRY OF EPSTAB WHICH CONTAINS
!    THE NEW ELEMENT IN THE FIRST COLUMN OF THE EPSILON TABLE.
!
!    INPUT/OUTPUT, REAL(KIND=8) EPSTAB(52), THE TWO LOWER DIAGONALS OF THE TRIANGULAR
!    EPSILON TABLE.  THE ELEMENTS ARE NUMBERED STARTING AT THE RIGHT-HAND 
!    CORNER OF THE TRIANGLE.
!
!    OUTPUT, REAL(KIND=8) RESULT, THE ESTIMATED VALUE OF THE INTEGRAL.
!
!    OUTPUT, REAL(KIND=8) ABSERR, ESTIMATE OF THE ABSOLUTE ERROR COMPUTED FROM
!    RESULT AND THE 3 PREVIOUS RESULTS.
!
!    ?, REAL(KIND=8) RES3LA(3), THE LAST 3 RESULTS.
!
!    INPUT/OUTPUT, INTEGER NRES, THE NUMBER OF CALLS TO THE ROUTINE.  THIS
!    SHOULD BE ZERO ON THE FIRST CALL, AND IS AUTOMATICALLY UPDATED
!    BEFORE RETURN.
!
!  LOCAL PARAMETERS:
!
!           E0     - THE 4 ELEMENTS ON WHICH THE
!           E1       COMPUTATION OF A NEW ELEMENT IN
!           E2       THE EPSILON TABLE IS BASED
!           E3                 E0
!                        E3    E1    NEW
!                              E2
!           NEWELM - NUMBER OF ELEMENTS TO BE COMPUTED IN THE NEW
!                    DIAGONAL
!           ERROR  - ERROR = ABS(E1-E0)+ABS(E2-E1)+ABS(NEW-E2)
!           RESULT - THE ELEMENT IN THE NEW DIAGONAL WITH LEAST VALUE
!                    OF ERROR
!           LIMEXP IS THE MAXIMUM NUMBER OF ELEMENTS THE EPSILON TABLE
!           CAN CONTAIN. IF THIS NUMBER IS REACHED, THE UPPER DIAGONAL
!           OF THE EPSILON TABLE IS DELETED.
!
  IMPLICIT NONE

  REAL(KIND=8) ABSERR
  REAL(KIND=8) DELTA1
  REAL(KIND=8) DELTA2
  REAL(KIND=8) DELTA3
  REAL(KIND=8) EPSINF
  REAL(KIND=8) EPSTAB(52)
  REAL(KIND=8) ERROR
  REAL(KIND=8) ERR1
  REAL(KIND=8) ERR2
  REAL(KIND=8) ERR3
  REAL(KIND=8) E0
  REAL(KIND=8) E1
  REAL(KIND=8) E1ABS
  REAL(KIND=8) E2
  REAL(KIND=8) E3
  INTEGER I
  INTEGER IB
  INTEGER IB2
  INTEGER IE
  INTEGER INDX
  INTEGER K1
  INTEGER K2
  INTEGER K3
  INTEGER LIMEXP
  INTEGER N
  INTEGER NEWELM
  INTEGER NRES
  INTEGER NUM
  REAL(KIND=8) RES
  REAL(KIND=8) RESULT
  REAL(KIND=8) RES3LA(3)
  REAL(KIND=8) SS
  REAL(KIND=8) TOL1
  REAL(KIND=8) TOL2
  REAL(KIND=8) TOL3

  NRES = NRES+1
  ABSERR = HUGE ( ABSERR )
  RESULT = EPSTAB(N)

  IF ( N < 3 ) THEN
    ABSERR = MAX ( ABSERR,0.5D+00* EPSILON ( RESULT ) *ABS(RESULT))
    RETURN
  END IF

  LIMEXP = 50
  EPSTAB(N+2) = EPSTAB(N)
  NEWELM = (N-1)/2
  EPSTAB(N) = HUGE ( EPSTAB(N) )
  NUM = N
  K1 = N

  DO I = 1, NEWELM

    K2 = K1-1
    K3 = K1-2
    RES = EPSTAB(K1+2)
    E0 = EPSTAB(K3)
    E1 = EPSTAB(K2)
    E2 = RES
    E1ABS = ABS(E1)
    DELTA2 = E2-E1
    ERR2 = ABS(DELTA2)
    TOL2 = MAX ( ABS(E2),E1ABS)* EPSILON ( E2 )
    DELTA3 = E1-E0
    ERR3 = ABS(DELTA3)
    TOL3 = MAX ( E1ABS,ABS(E0))* EPSILON ( E0 )
!
!  IF E0, E1 AND E2 ARE EQUAL TO WITHIN MACHINE ACCURACY, CONVERGENCE 
!  IS ASSUMED.
!
    IF ( ERR2 <= TOL2 .AND. ERR3 <= TOL3 ) THEN
      RESULT = RES
      ABSERR = ERR2+ERR3
      ABSERR = MAX ( ABSERR,0.5D+00* EPSILON ( RESULT ) *ABS(RESULT))
      RETURN
    END IF

    E3 = EPSTAB(K1)
    EPSTAB(K1) = E1
    DELTA1 = E1-E3
    ERR1 = ABS(DELTA1)
    TOL1 = MAX ( E1ABS,ABS(E3))* EPSILON ( E3 )
!
!  IF TWO ELEMENTS ARE VERY CLOSE TO EACH OTHER, OMIT A PART
!  OF THE TABLE BY ADJUSTING THE VALUE OF N.
!
    IF ( ERR1 <= TOL1 .OR. ERR2 <= TOL2 .OR. ERR3 <= TOL3 ) GO TO 20

    SS = 1.0D+00/DELTA1+1.0D+00/DELTA2-1.0D+00/DELTA3
    EPSINF = ABS ( SS*E1 )
!
!  TEST TO DETECT IRREGULAR BEHAVIOR IN THE TABLE, AND
!  EVENTUALLY OMIT A PART OF THE TABLE ADJUSTING THE VALUE OF N.
!
    IF ( EPSINF > 1.0E-04 ) GO TO 30

20  CONTINUE

    N = I+I-1
    EXIT
!
!  COMPUTE A NEW ELEMENT AND EVENTUALLY ADJUST THE VALUE OF RESULT.
!
30  CONTINUE

    RES = E1+1.0D+00/SS
    EPSTAB(K1) = RES
    K1 = K1-2
    ERROR = ERR2+ABS(RES-E2)+ERR3

    IF ( ERROR <= ABSERR ) THEN
      ABSERR = ERROR
      RESULT = RES
    END IF

  END DO
!
!  SHIFT THE TABLE.
!
  IF ( N == LIMEXP ) THEN
    N = 2*(LIMEXP/2)-1
  END IF

  IF ( (NUM/2)*2 == NUM ) THEN
    IB = 2
  ELSE
    IB = 1
  END IF

  IE = NEWELM+1

  DO I = 1, IE
    IB2 = IB+2
    EPSTAB(IB) = EPSTAB(IB2)
    IB = IB2
  END DO

  IF ( NUM /= N ) THEN

    INDX = NUM-N+1

    DO I = 1, N
      EPSTAB(I)= EPSTAB(INDX)
      INDX = INDX+1
    END DO

  END IF

  IF ( NRES < 4 ) THEN
    RES3LA(NRES) = RESULT
    ABSERR = HUGE ( ABSERR )
  ELSE
    ABSERR = ABS(RESULT-RES3LA(3))+ABS(RESULT-RES3LA(2)) &
      +ABS(RESULT-RES3LA(1))
    RES3LA(1) = RES3LA(2)
    RES3LA(2) = RES3LA(3)
    RES3LA(3) = RESULT
  END IF

  ABSERR = MAX ( ABSERR,0.5D+00* EPSILON ( RESULT ) *ABS(RESULT))

  RETURN
END




SUBROUTINE QK21 ( F, A, B, RESULT, ABSERR, RESABS, RESASC, BB1,BB2,NU )

!*****************************************************************************80
!
!! QK21 CARRIES OUT A 21 POINT GAUSS-KRONROD QUADRATURE RULE.
!
!  DISCUSSION:
!
!    THIS ROUTINE APPROXIMATES
!      I = INTEGRAL ( A <= X <= B ) F(X) DX
!    WITH AN ERROR ESTIMATE, AND
!      J = INTEGRAL ( A <= X <= B ) | F(X) | DX
!
!  AUTHOR:
!
!    ROBERT PIESSENS, ELISE DE DONCKER-KAPENGER, 
!    CHRISTIAN UEBERHUBER, DAVID KAHANER
!
!  REFERENCE:
!
!    ROBERT PIESSENS, ELISE DE DONCKER-KAPENGER, 
!    CHRISTIAN UEBERHUBER, DAVID KAHANER,
!    QUADPACK, A SUBROUTINE PACKAGE FOR AUTOMATIC INTEGRATION,
!    SPRINGER VERLAG, 1983
!
!  PARAMETERS:
!
!    INPUT, EXTERNAL REAL(KIND=8) F, THE NAME OF THE FUNCTION ROUTINE, OF THE FORM
!      FUNCTION F ( X )
!      REAL(KIND=8) F
!      REAL(KIND=8) X
!    WHICH EVALUATES THE INTEGRAND FUNCTION.
!
!    INPUT, REAL(KIND=8) A, B, THE LIMITS OF INTEGRATION.
!
!    OUTPUT, REAL(KIND=8) RESULT, THE ESTIMATED VALUE OF THE INTEGRAL.
!    RESULT IS COMPUTED BY APPLYING THE 21-POINT KRONROD RULE (RESK) 
!    OBTAINED BY OPTIMAL ADDITION OF ABSCISSAE TO THE 10-POINT GAUSS 
!    RULE (RESG).
!
!    OUTPUT, REAL(KIND=8) ABSERR, AN ESTIMATE OF | I - RESULT |.
!
!    OUTPUT, REAL(KIND=8) RESABS, APPROXIMATION TO THE INTEGRAL OF THE ABSOLUTE
!    VALUE OF F.
!
!    OUTPUT, REAL(KIND=8) RESASC, APPROXIMATION TO THE INTEGRAL | F-I/(B-A) | 
!    OVER [A,B].
!
  IMPLICIT NONE

  INTERFACE                     ! ZFYUAN ADDED HERE.
     FUNCTION F(X,BB1,BB2,NU)
       IMPLICIT NONE
       REAL(KIND=8) :: X, BB1, BB2,NU,F
     END FUNCTION F
  END INTERFACE
  REAL(KIND=8) :: BB1,BB2,NU

  REAL(KIND=8) A
  REAL(KIND=8) ABSC
  REAL(KIND=8) ABSERR
  REAL(KIND=8) B
  REAL(KIND=8) CENTR
  REAL(KIND=8) DHLGTH
  ! REAL(KIND=8), EXTERNAL :: F ! ZFYUAN COMMENTED, FRI DEC 28 2012
  REAL(KIND=8) FC
  REAL(KIND=8) FSUM
  REAL(KIND=8) FVAL1
  REAL(KIND=8) FVAL2
  REAL(KIND=8) FV1(10)
  REAL(KIND=8) FV2(10)
  REAL(KIND=8) HLGTH
  INTEGER J
  INTEGER JTW
  INTEGER JTWM1
  REAL(KIND=8) RESABS
  REAL(KIND=8) RESASC
  REAL(KIND=8) RESG
  REAL(KIND=8) RESK
  REAL(KIND=8) RESKH
  REAL(KIND=8) RESULT
  REAL(KIND=8) WG(5)
  REAL(KIND=8) WGK(11)
  REAL(KIND=8) XGK(11)
!
!           THE ABSCISSAE AND WEIGHTS ARE GIVEN FOR THE INTERVAL (-1,1).
!           BECAUSE OF SYMMETRY ONLY THE POSITIVE ABSCISSAE AND THEIR
!           CORRESPONDING WEIGHTS ARE GIVEN.
!
!           XGK    - ABSCISSAE OF THE 21-POINT KRONROD RULE
!                    XGK(2), XGK(4), ...  ABSCISSAE OF THE 10-POINT
!                    GAUSS RULE
!                    XGK(1), XGK(3), ...  ABSCISSAE WHICH ARE OPTIMALLY
!                    ADDED TO THE 10-POINT GAUSS RULE
!
!           WGK    - WEIGHTS OF THE 21-POINT KRONROD RULE
!
!           WG     - WEIGHTS OF THE 10-POINT GAUSS RULE
!
  DATA XGK(1),XGK(2),XGK(3),XGK(4),XGK(5),XGK(6),XGK(7),XGK(8), &
    XGK(9),XGK(10),XGK(11)/ &
       9.956571630258081E-01,     9.739065285171717E-01, &
       9.301574913557082E-01,     8.650633666889845E-01, &
       7.808177265864169E-01,     6.794095682990244E-01, &
       5.627571346686047E-01,     4.333953941292472E-01, &
       2.943928627014602E-01,     1.488743389816312E-01, &
       0.000000000000000D+00/
!
  DATA WGK(1),WGK(2),WGK(3),WGK(4),WGK(5),WGK(6),WGK(7),WGK(8), &
    WGK(9),WGK(10),WGK(11)/ &
       1.169463886737187E-02,     3.255816230796473E-02, &
       5.475589657435200E-02,     7.503967481091995E-02, &
       9.312545458369761E-02,     1.093871588022976E-01, &
       1.234919762620659E-01,     1.347092173114733E-01, &
       1.427759385770601E-01,     1.477391049013385E-01, &
       1.494455540029169E-01/
!
  DATA WG(1),WG(2),WG(3),WG(4),WG(5)/ &
       6.667134430868814E-02,     1.494513491505806E-01, &
       2.190863625159820E-01,     2.692667193099964E-01, &
       2.955242247147529E-01/
!
!
!           LIST OF MAJOR VARIABLES
!
!           CENTR  - MID POINT OF THE INTERVAL
!           HLGTH  - HALF-LENGTH OF THE INTERVAL
!           ABSC   - ABSCISSA
!           FVAL*  - FUNCTION VALUE
!           RESG   - RESULT OF THE 10-POINT GAUSS FORMULA
!           RESK   - RESULT OF THE 21-POINT KRONROD FORMULA
!           RESKH  - APPROXIMATION TO THE MEAN VALUE OF F OVER (A,B),
!                    I.E. TO I/(B-A)
!
  CENTR = 5.0E-01*(A+B)
  HLGTH = 5.0E-01*(B-A)
  DHLGTH = ABS(HLGTH)
!
!  COMPUTE THE 21-POINT KRONROD APPROXIMATION TO THE
!  INTEGRAL, AND ESTIMATE THE ABSOLUTE ERROR.
!
  RESG = 0.0D+00
  ! FC = F(CENTR) ! ZFYUAN COMMENTED ADD NEXT LINE.
  FC = F(CENTR, BB1,BB2,NU)
  
  RESK = WGK(11)*FC
  RESABS = ABS(RESK)

  DO J = 1, 5
    JTW = 2*J
    ABSC = HLGTH*XGK(JTW)
    ! FVAL1 = F(CENTR-ABSC)
    ! FVAL2 = F(CENTR+ABSC)!ZFYUAN COMMENT ADD TWO LINES BELOW
    FVAL1 = F(CENTR-ABSC,BB1,BB2,NU)
    FVAL2 = F(CENTR+ABSC,BB1,BB2,NU)
    
    FV1(JTW) = FVAL1
    FV2(JTW) = FVAL2
    FSUM = FVAL1+FVAL2
    RESG = RESG+WG(J)*FSUM
    RESK = RESK+WGK(JTW)*FSUM
    RESABS = RESABS+WGK(JTW)*(ABS(FVAL1)+ABS(FVAL2))
  END DO

  DO J = 1, 5
    JTWM1 = 2*J-1
    ABSC = HLGTH*XGK(JTWM1)
    ! FVAL1 = F(CENTR-ABSC)
    ! FVAL2 = F(CENTR+ABSC)! ZFYUAN COMMENT ADD TWO LINES BELOW
    FVAL1 = F(CENTR-ABSC,BB1,BB2,NU)
    FVAL2 = F(CENTR+ABSC,BB1,BB2,NU)
    FV1(JTWM1) = FVAL1
    FV2(JTWM1) = FVAL2
    FSUM = FVAL1+FVAL2
    RESK = RESK+WGK(JTWM1)*FSUM
    RESABS = RESABS+WGK(JTWM1)*(ABS(FVAL1)+ABS(FVAL2))
  END DO

  RESKH = RESK*5.0E-01
  RESASC = WGK(11)*ABS(FC-RESKH)

  DO J = 1, 10
    RESASC = RESASC+WGK(J)*(ABS(FV1(J)-RESKH)+ABS(FV2(J)-RESKH))
  END DO

  RESULT = RESK*HLGTH
  RESABS = RESABS*DHLGTH
  RESASC = RESASC*DHLGTH
  ABSERR = ABS((RESK-RESG)*HLGTH)

  IF ( RESASC /= 0.0D+00.AND.ABSERR /= 0.0D+00) THEN
    ABSERR = RESASC*MIN ( 1.0D+00,(2.0E+02*ABSERR/RESASC)**1.5D+00)
  END IF

  IF ( RESABS > TINY ( RESABS ) /(5.0E+01* EPSILON ( RESABS ) )) THEN
    ABSERR = MAX (( EPSILON ( RESABS ) *5.0E+01)*RESABS,ABSERR)
  END IF

  RETURN
END SUBROUTINE QK21



SUBROUTINE QSORT ( LIMIT, LAST, MAXERR, ERMAX, ELIST, IORD, NRMAX )

!*****************************************************************************80
!
!! QSORT MAINTAINS THE ORDER OF A LIST OF LOCAL ERROR ESTIMATES.
!
!  DISCUSSION:
!
!    THIS ROUTINE MAINTAINS THE DESCENDING ORDERING IN THE LIST OF THE 
!    LOCAL ERROR ESTIMATES RESULTING FROM THE INTERVAL SUBDIVISION PROCESS. 
!    AT EACH CALL TWO ERROR ESTIMATES ARE INSERTED USING THE SEQUENTIAL 
!    SEARCH TOP-DOWN FOR THE LARGEST ERROR ESTIMATE AND BOTTOM-UP FOR THE
!    SMALLEST ERROR ESTIMATE.
!
!  AUTHOR:
!
!    ROBERT PIESSENS, ELISE DE DONCKER-KAPENGER, 
!    CHRISTIAN UEBERHUBER, DAVID KAHANER
!
!  REFERENCE:
!
!    ROBERT PIESSENS, ELISE DE DONCKER-KAPENGER, 
!    CHRISTIAN UEBERHUBER, DAVID KAHANER,
!    QUADPACK, A SUBROUTINE PACKAGE FOR AUTOMATIC INTEGRATION,
!    SPRINGER VERLAG, 1983
!
!  PARAMETERS:
!
!    INPUT, INTEGER LIMIT, THE MAXIMUM NUMBER OF ERROR ESTIMATES THE LIST CAN
!    CONTAIN.
!
!    INPUT, INTEGER LAST, THE CURRENT NUMBER OF ERROR ESTIMATES.
!
!    INPUT/OUTPUT, INTEGER MAXERR, THE INDEX IN THE LIST OF THE NRMAX-TH 
!    LARGEST ERROR.
!
!    OUTPUT, REAL(KIND=8) ERMAX, THE NRMAX-TH LARGEST ERROR = ELIST(MAXERR).
!
!    INPUT, REAL(KIND=8) ELIST(LIMIT), CONTAINS THE ERROR ESTIMATES.
!
!    INPUT/OUTPUT, INTEGER IORD(LAST).  THE FIRST K ELEMENTS CONTAIN 
!    POINTERS TO THE ERROR ESTIMATES SUCH THAT ELIST(IORD(1)) THROUGH
!    ELIST(IORD(K)) FORM A DECREASING SEQUENCE, WITH
!      K = LAST 
!    IF 
!      LAST <= (LIMIT/2+2), 
!    AND OTHERWISE
!      K = LIMIT+1-LAST.
!
!    INPUT/OUTPUT, INTEGER NRMAX.
!
  IMPLICIT NONE

  INTEGER LAST

  REAL(KIND=8) ELIST(LAST)
  REAL(KIND=8) ERMAX
  REAL(KIND=8) ERRMAX
  REAL(KIND=8) ERRMIN
  INTEGER I
  INTEGER IBEG
  INTEGER IORD(LAST)
  INTEGER ISUCC
  INTEGER J
  INTEGER JBND
  INTEGER JUPBN
  INTEGER K
  INTEGER LIMIT
  INTEGER MAXERR
  INTEGER NRMAX
!
!  CHECK WHETHER THE LIST CONTAINS MORE THAN TWO ERROR ESTIMATES.
!
  IF ( LAST <= 2 ) THEN
    IORD(1) = 1
    IORD(2) = 2
    GO TO 90
  END IF
!
!  THIS PART OF THE ROUTINE IS ONLY EXECUTED IF, DUE TO A
!  DIFFICULT INTEGRAND, SUBDIVISION INCREASED THE ERROR
!  ESTIMATE. IN THE NORMAL CASE THE INSERT PROCEDURE SHOULD
!  START AFTER THE NRMAX-TH LARGEST ERROR ESTIMATE.
!
  ERRMAX = ELIST(MAXERR)

  DO I = 1, NRMAX-1

    ISUCC = IORD(NRMAX-1)

    IF ( ERRMAX <= ELIST(ISUCC) ) THEN
      EXIT
    END IF

    IORD(NRMAX) = ISUCC
    NRMAX = NRMAX-1

  END DO
!
!  COMPUTE THE NUMBER OF ELEMENTS IN THE LIST TO BE MAINTAINED
!  IN DESCENDING ORDER.  THIS NUMBER DEPENDS ON THE NUMBER OF
!  SUBDIVISIONS STILL ALLOWED.
!
  JUPBN = LAST

  IF ( (LIMIT/2+2) < LAST ) THEN
    JUPBN = LIMIT+3-LAST
  END IF

  ERRMIN = ELIST(LAST)
!
!  INSERT ERRMAX BY TRAVERSING THE LIST TOP-DOWN, STARTING
!  COMPARISON FROM THE ELEMENT ELIST(IORD(NRMAX+1)).
!
  JBND = JUPBN-1
  IBEG = NRMAX+1

  DO I = IBEG, JBND
    ISUCC = IORD(I)
    IF ( ELIST(ISUCC) <= ERRMAX ) THEN
      GO TO 60
    END IF
    IORD(I-1) = ISUCC
  END DO

  IORD(JBND) = MAXERR
  IORD(JUPBN) = LAST
  GO TO 90
!
!  INSERT ERRMIN BY TRAVERSING THE LIST BOTTOM-UP.
!
60 CONTINUE

  IORD(I-1) = MAXERR
  K = JBND

  DO J = I, JBND
    ISUCC = IORD(K)
    IF ( ERRMIN < ELIST(ISUCC) ) THEN
      GO TO 80
    END IF
    IORD(K+1) = ISUCC
    K = K-1
  END DO

  IORD(I) = LAST
  GO TO 90

80 CONTINUE

  IORD(K+1) = LAST
!
!  SET MAXERR AND ERMAX.
!
90 CONTINUE

  MAXERR = IORD(NRMAX)
  ERMAX = ELIST(MAXERR)

  RETURN
END SUBROUTINE QSORT
