!    -*- F90 -*-
!
! AUTHOR  : ZHENFEI YUAN, TAIZHONG HU
! EMAIL   : ZFYUAN@MAIL.USTC.EDU.CN; THU@USTC.EDU.CN
! URL     : TAIZHONGLAB.USTC.EDU.CN/SOFTWARE/PYVINE.HTML
! DATE    : DEC 3, 2012
! LICENCE : GPL (>= 2)
!
!
! THIS FILE CONSISTS OF ROUTINES FOR BIVARIATE T COPULA.
!
! LIST OF ROUTINES:
!
!   PBV_T_F
!   DBV_T_F
!   BV_T_LOGLIK_F
!   RBV_T_F
!   BV_T_HFUNC_F
!   BV_T_INV_HFUNC_F
!   BV_T_MLE




FUNCTION F_T(X,B1,B2,NU)
  IMPLICIT NONE
  REAL(KIND=8) :: F_T
  REAL(KIND=8) X,B1,B2,NU

  F_T = (1 + &
       (B1**2 + B2**2 - 2 * SIN(X)*B1*B2) / (NU* COS(X)** 2) ) ** (-NU/2)

END FUNCTION F_T



SUBROUTINE PBV_T_F(U1,U2,RES,RHO,NU,N)
  !
  !  CALCULATES PROBABILITY FUNCTION OF BIVARIATE STUDENT T
  !  COPULAS.
  !
  !  REALIZATION OF ALAN GENZ' 'GENERALIZED PLACKETT BVT FORMULA'
  !  DESCRIBED IN "NUMERICAL COMPUTATION OF RECTANGULAR BIVARIATE AND
  !  TRIVARIATE NORMAL AND T PROBABILITIES".
  !
  IMPLICIT NONE
  
  REAL(KIND=8) :: U1(N),U2(N),RES(N)
  REAL(KIND=8) :: RHO,NU
  REAL(KIND=8) :: B1,B2
  REAL(KIND=8) :: LB,UB
  REAL(KIND=8) , PARAMETER :: PI = 3.1415926535897932
  REAL(KIND=8) :: EPSABS = 1D-8, EPSREL = 1D-8, ABSERR,NEVAL
  INTEGER :: N,S,INFO,I
  REAL(KIND=8) :: C
  REAL(KIND=8),EXTERNAL :: F_T

  DO I = 1,N
     
     IF(U1(I)*U2(I).EQ.0)THEN
        RES(I) = 0
     ELSE IF(U1(I).EQ.1)THEN
        RES(I) = U2(I)
     ELSE IF(U2(I).EQ.1)THEN
        RES(I) = U1(I)
     ELSE
        
        IF(RHO.GE.0)THEN
           S = 1
           C = MIN(U1(I),U2(I))
        ELSE
           S = -1
           C = MAX(0.0D0,U1(I)+U2(I)-1)
        END IF

        LB = S * PI/2
        UB = ASIN(RHO)

        CALL QT(U1(I),B1,NU)
        CALL QT(U2(I),B2,NU)        

        IF(LB.LE.UB)THEN
           CALL QAGS(F_T,LB,UB,EPSABS,EPSREL,RES(I),ABSERR,NEVAL,INFO,B1,B2,NU)
        ELSE
           CALL QAGS(F_T,UB,LB,EPSABS,EPSREL,RES(I),ABSERR,NEVAL,INFO,B1,B2,NU)
           RES(I) = -RES(I)
        END IF

        RES(I) = C + 1/(2*PI) * RES(I)

     END IF

  END DO

END SUBROUTINE PBV_T_F




SUBROUTINE DBV_T_F(U1,U2,RES,RHO,NU,N)
  IMPLICIT NONE
  
  REAL(KIND=8) :: U1(N),U2(N),RES(N),RHO,NU
  INTEGER :: N
  
  REAL(KIND=8) :: QU1(N),QU2(N)
  REAL(KIND=8) :: C1,C2(N)
  REAL(KIND=8) , PARAMETER :: PI = 3.1415926535897932

  CALL QT_VEC(U1,QU1,NU,N)
  CALL QT_VEC(U2,QU2,NU,N)  


  C1 = EXP(LOG_GAMMA(NU/2)+LOG_GAMMA(NU/2+1)-2*LOG_GAMMA(NU/2+0.5D00))&
       / SQRT(1-RHO**2)
  C2 = (1+(QU1**2+QU2**2-2*RHO*QU1*QU2)/NU/(1-RHO**2))&
       **(NU/2+1)
  RES = C1 / C2 * (1+ QU1**2/NU)**(NU/2+0.5)&
       * (1+QU2**2/NU)**(NU/2+0.5)

END SUBROUTINE DBV_T_F



SUBROUTINE RBV_T_F(U1,U2,RHO,NU,N)
  IMPLICIT NONE

  REAL(KIND=8) :: RHO,NU
  INTEGER :: N

  REAL(KIND=8) :: U1(N),U2(N),RES(N)

  CALL INIT_RANDOM_SEED()

  CALL RANDOM_NUMBER(RES)
  CALL RANDOM_NUMBER(U2)

  CALL BV_T_INV_HFUNC_F(RES,U2,U1,RHO,NU,N)

END SUBROUTINE RBV_T_F



SUBROUTINE BV_T_HFUNC_F(U1,U2,RES,RHO,NU,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),RHO,NU
  INTEGER :: N

  REAL(KIND=8) :: QU1(N),QU2(N),I1(N),I2(N)

  CALL QT_VEC(U1,QU1,NU,N)
  CALL QT_VEC(U2,QU2,NU,N)  

  I1 = QU1 - RHO * QU2
  I2 = SQRT((NU+QU2**2)*(1-RHO**2)/(NU+1.0D00))
  CALL PT_VEC(I1 / I2,RES,NU+1,N)

END SUBROUTINE BV_T_HFUNC_F


SUBROUTINE BV_T_INV_HFUNC_F(U1,U2,RES,RHO,NU,N)
  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),RHO,NU
  INTEGER :: N
  
  REAL(KIND=8) :: QU1(N),QU2(N),CV(N)

  CALL QT_VEC(U1,QU1,NU+1,N)
  CALL QT_VEC(U2,QU2,NU,N)

  CV = SQRT((NU+QU2**2) * (1-RHO**2) / (NU+1))

  CALL PT_VEC(QU1 * CV + RHO * QU2,RES,NU,N)
  
END SUBROUTINE BV_T_INV_HFUNC_F




FUNCTION BV_T_LOGLIK_F(U1,U2,RHO,NU,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),RHO,NU,BV_T_LOGLIK_F
  INTEGER :: N

  CALL DBV_T_F(U1,U2,RES,RHO,NU,N)

  BV_T_LOGLIK_F = SUM(LOG(RES))

END FUNCTION BV_T_LOGLIK_F


!! THE BLOCK COMMENTED BELOW GUESS THE INITIAL VALUE FOR BIVARIATE T
!! MLE USING LBFGSB ALGORITHM, HOWEVER THIS GUESSING ROUTINE FAILS
!! WHEN CORRELATION CLOSE TO 1 OR -1. AND IT IS REPLACED WITH THE
!! VERSION USING BRENT LOCAL_MIN FUNCTION.
!! ZHENFEI YUAN , COMMENTED ON NOV 28, 2012

! SUBROUTINE BV_T_INIT(U1,U2,NNUM,INIT)
!   !
!   ! SUBROUTINE FOR GUESSING THE INITIAL VALUE FOR L-BFGS-B
!   ! OPTIMIZATION OF BIVARIATE T COPULA MLE.
!   !
!   ! FOR BIVARIATE T COPULAS, CORRELATIOIN PAR IS GUESSED FROM PEARSON
!   ! CORRELATION, AND GUESS OF DEGREE OF FREEDOM IS OBTAINED FROM
!   ! OPTIMIZATION WITH FIXED RHO.
!   ! 
!   IMPLICIT NONE

!   INTEGER      :: NNUM
!   REAL(KIND=8) :: U1(NNUM),U2(NNUM),INIT(2)
!   ! LOCAL
!   REAL(KIND=8) :: QU1(NNUM),QU2(NNUM)
!   REAL(KIND=8)           :: EPS = 1.4901161193847656D-08  

!   INTEGER,  PARAMETER    :: N = 1, M = 5, IPRINT = 0
!   REAL(KIND=8), PARAMETER:: FACTR  = 1.0D+7, PGTOL  = 1.0D-5

!   CHARACTER(LEN=60)      :: TASK, CSAVE
!   LOGICAL                :: LSAVE(4)
!   INTEGER                :: ISAVE(44)
!   REAL(KIND=8)           :: F
!   REAL(KIND=8)           :: DSAVE(29)
!   INTEGER,  ALLOCATABLE  :: NBD(:), IWA(:)
!   REAL(KIND=8), ALLOCATABLE  :: X(:), L(:), U(:), G(:), WA(:)
  
!   REAL(KIND=8), EXTERNAL :: BV_T_LOGLIK_F

!   CALL QNORM_VEC(U1,QU1,NNUM)
!   CALL QNORM_VEC(U2,QU2,NNUM)

!   QU1 = QU1 - SUM(QU1)/NNUM
!   QU2 = QU2 - SUM(QU2)/NNUM

!   INIT(1) = SUM(QU1*QU2) / SQRT( SUM(QU1**2) * SUM(QU2**2) )

!   ALLOCATE ( NBD(N), X(N), L(N), U(N), G(N) )
!   ALLOCATE ( IWA(3*N) )
!   ALLOCATE ( WA(2*M*N + 5*N + 11*M*M + 8*M) )
!   NBD = 2

!   L = 2.1D0
!   U = 1D3

!   X = 4D0

!   TASK = 'START'

!   ! THE BEGINNING OF THE LOOP
  
!   DO WHILE(TASK(1:2).EQ.'FG'.OR.TASK.EQ.'NEW_X'.OR. &
!                TASK.EQ.'START') 
         
!      ! THIS IS THE CALL TO THE L-BFGS-B CODE.
         
!      CALL SETULB ( N, M, X, L, U, NBD, F, G, FACTR, PGTOL, &
!           WA, IWA, TASK, IPRINT,&
!           CSAVE, LSAVE, ISAVE, DSAVE )
         
!      IF (TASK(1:2) .EQ. 'FG') THEN
!         ! COMPUTE FUNCTION VALUE F
!         F = BV_T_LOGLIK_F(U1,U2,INIT(1),X(1),NNUM)
        
!         ! COMPUTE GRADIENT G FOR THE SAMPLE PROBLEM.
!         G = (BV_T_LOGLIK_F(U1,U2,INIT(1),X(1)+EPS,NNUM) - F) / EPS

!         WRITE(*,*)G

!      END IF

!   END DO
  
!   INIT(2) = X(1)

!   WRITE(*,*)"INITIAL",INIT
! END SUBROUTINE BV_T_INIT


! THE INITAL GUESSING ROUTINE USING BRENT'S LOCAL_MIN ALGORITHM


SUBROUTINE BV_T_INIT(U1,U2,NNUM,INIT)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),INIT(2)
  ! LOCAL
  REAL(KIND=8)           :: QU1(NNUM),QU2(NNUM),X
  REAL(KIND=8)           :: LOW, UPP, LOGLIK_L, LOGLIK_R, STEP
  REAL(KIND=8)           :: EXP = 1D-8,TOL = 1D-2,FN_VAL
  REAL(KIND=8), EXTERNAL :: BV_T_LOGLIK_F, LOCAL_MIN_T

  STEP = 2D0

  CALL QNORM_VEC(U1,QU1,NNUM)
  CALL QNORM_VEC(U2,QU2,NNUM)

  QU1 = QU1 - SUM(QU1)/NNUM
  QU2 = QU2 - SUM(QU2)/NNUM

  INIT(1) = SUM(QU1*QU2) / SQRT( SUM(QU1**2) * SUM(QU2**2) )

  LOW = 2D0
  UPP = 1D2
  
  LOGLIK_L = BV_T_LOGLIK_F(U1,U2,INIT(1),LOW,NNUM)
  LOGLIK_R = BV_T_LOGLIK_F(U1,U2,INIT(1),LOW+STEP,NNUM)

  DO WHILE(LOGLIK_L.LT.LOGLIK_R.AND.LOW.LT.UPP)
     LOW = LOW + STEP
     
     LOGLIK_L = BV_T_LOGLIK_F(U1,U2,INIT(1),LOW,NNUM)
     LOGLIK_R = BV_T_LOGLIK_F(U1,U2,INIT(1),LOW+STEP,NNUM)
  END DO

  IF(LOW.GE.UPP)THEN
     X = 1D2
  ELSE
     UPP = LOW + STEP
     LOW = MAX(LOW - STEP,2D0)
     FN_VAL = -LOCAL_MIN_T(LOW,UPP,EXP,TOL,BV_T_LOGLIK_F,U1,U2,INIT(1),NNUM,X)
  END IF

  INIT(2) = X

! CONTAINS 

!   FUNCTION TARGET_FUN(NU)
!     REAL(KIND=8)   :: NU,TARGET_FUN
!     TARGET_FUN = -BV_T_LOGLIK_F(U1,U2,INIT(1),NU,NNUM)
!   END FUNCTION TARGET_FUN

END SUBROUTINE BV_T_INIT


SUBROUTINE BV_T_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER,  PARAMETER    :: N = 2, M = 5, IPRINT = -1
  REAL(KIND=8), PARAMETER:: FACTR  = 1.0D+7, PGTOL  = 1.0D-5

  CHARACTER(LEN=60)      :: TASK, CSAVE
  LOGICAL                :: LSAVE(4)
  INTEGER                :: ISAVE(44)
  REAL(KIND=8)           :: F
  REAL(KIND=8)           :: DSAVE(29)
  INTEGER,  ALLOCATABLE  :: NBD(:), IWA(:)
  REAL(KIND=8), ALLOCATABLE  :: X(:), L(:), U(:), G(:), WA(:)
  

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  REAL(KIND=8)           :: EPS = 1.5D-08 ! EPS IS SQUARE ROOT OF
                                          ! MACHINE PRECISION
  REAL(KIND=8), EXTERNAL :: BV_T_LOGLIK_F

  !  ALLOCATE DYNAMIC ARRAYS

  ALLOCATE ( NBD(N), X(N), L(N), U(N), G(N) )
  ALLOCATE ( IWA(3*N) )
  ALLOCATE ( WA(2*M*N + 5*N + 11*M*M + 8*M) )

  NBD = 2

  CALL BV_T_INIT(U1,U2,NNUM,X)

  ! ABS(RHO) BOUNDS CLOSE TO ONE WILL CAUSE ABNORMAL
  L = (/MAX(-9.9D-1,X(1)-5D-2) , MAX(2D0,X(2)-2D0)/)
  U = (/MIN( 9.9D-1,X(1)+5D-2) , MIN(1D2,X(2)+2D0)/)

  !   WE START THE ITERATION BY INITIALIZING TASK.
 
  TASK = 'START'

  !   THE BEGINNING OF THE LOOP
 
  DO WHILE(TASK(1:2).EQ.'FG'.OR.TASK.EQ.'NEW_X'.OR. &
               TASK.EQ.'START') 
         
     !  THIS IS THE CALL TO THE L-BFGS-B CODE.
         
     CALL SETULB ( N, M, X, L, U, NBD, F, G, FACTR, PGTOL, &
          WA, IWA, TASK, IPRINT,&
          CSAVE, LSAVE, ISAVE, DSAVE )
         
     IF (TASK(1:2) .EQ. 'FG') THEN
        ! COMPUTE FUNCTION VALUE F
        F = -BV_T_LOGLIK_F(U1,U2,X(1),X(2),NNUM)
        
        ! COMPUTE GRADIENT G FOR THE SAMPLE PROBLEM.
        G(1) = (-BV_T_LOGLIK_F(U1,U2,X(1)+EPS,X(2),NNUM) - F) / EPS
        G(2) = (-BV_T_LOGLIK_F(U1,U2,X(1),X(2)+EPS,NNUM) - F) / EPS
        
     END IF

  END DO

  RES(1:2) = X
  LL = -F

END SUBROUTINE BV_T_MLE


