!    -*- F90 -*-
!
! AUTHOR  : ZHENFEI YUAN, TAIZHONG HU
! EMAIL   : ZFYUAN@MAIL.USTC.EDU.CN; THU@USTC.EDU.CN
! URL     : TAIZHONGLAB.USTC.EDU.CN/SOFTWARE/PYVINE.HTML
! DATE    : DEC 3, 2012
! LICENCE : GPL (>= 2)
!
! THIS FORTRAN 90 SOURCE FILE IS A SUPPORT FILE FOR MULTI PROCESSING
! FOR THE MLE CARRIED ON EDGES OF EACH VINE TREE, ALSO FOR COMPUTING
! THE LIKELIHOOD VALUE OF EACH VINE TREE.
!
!
! LIST OF ROUTINES:
!
!     MULTI_PROCESS_MLE
!     MULTI_PROCESS_LOGLIK
!     AD_STAT
!     ADINF
!     KENDALLTAU




SUBROUTINE DBV_NORM_F(U1,U2,RES,RHO,N)

  REAL(KIND=8) :: U1(N),U2(N),RES(N)
  REAL(KIND=8) :: RHO
  INTEGER :: N

  REAL(KIND=8) :: QU1(N),QU2(N),G1,G2(N)

  CALL QNORM_VEC(U1,QU1,N)
  CALL QNORM_VEC(U2,QU2,N)

  G1 = 1.0 / SQRT(1.0-RHO**2)
  
  G2 = -0.5 * (RHO**2 * (QU1**2+QU2**2) - 2*RHO*QU1*QU2)&
       / ( 1.-RHO**2 )

  RES = EXP(G2) * G1

END SUBROUTINE DBV_NORM_F


FUNCTION BV_NORM_LOGLIK_F(U1,U2,RHO,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),RHO,BV_NORM_LOGLIK_F
  INTEGER :: N

  CALL DBV_NORM_F(U1,U2,RES,RHO,N)

  BV_NORM_LOGLIK_F = SUM(LOG(RES))
  
END FUNCTION BV_NORM_LOGLIK_F



SUBROUTINE BV_NORM_INIT(U1,U2,NNUM,INIT)
  IMPLICIT NONE

  INTEGER      :: NNUM
  REAL(KIND=8) :: U1(NNUM),U2(NNUM),INIT
  ! LOCAL
  REAL(KIND=8) :: QU1(NNUM),QU2(NNUM)

  CALL QNORM_VEC(U1,QU1,NNUM)
  CALL QNORM_VEC(U2,QU2,NNUM)

  QU1 = QU1 - SUM(QU1) / NNUM
  QU2 = QU2 - SUM(QU2) / NNUM

  INIT = SUM(QU1*QU2) / SQRT( SUM(QU1**2) * SUM(QU2**2) )

END SUBROUTINE BV_NORM_INIT



SUBROUTINE BV_NORM_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  ! LOCAL
  REAL(KIND=8)           :: EPS = 1.5D-08,TOL = 1D-4,LOW, UPP , X
  REAL(KIND=8), EXTERNAL :: BV_NORM_LOGLIK_F,LOCAL_MIN

  CALL BV_NORM_INIT(U1,U2,NNUM,X)

  LOW = MAX(-9.99D-1 , X-5D-2 )
  UPP = MIN( 9.99D-1 , X+5D-2 )

  LL = -LOCAL_MIN(LOW,UPP,EPS,TOL,BV_NORM_LOGLIK_F,U1,U2,NNUM,X)

  RES(1) = X
  RES(2) = 0D0

! CONTAINS

!   FUNCTION TARGET_FUN(RHO)
!     IMPLICIT NONE
!     REAL(KIND=8) :: RHO,TARGET_FUN
!     TARGET_FUN = -BV_NORM_LOGLIK_F(U1,U2,RHO,NNUM)
!   END FUNCTION TARGET_FUN

END SUBROUTINE BV_NORM_MLE





SUBROUTINE DBV_T_F(U1,U2,RES,RHO,NU,N)
  IMPLICIT NONE
  
  REAL(KIND=8) :: U1(N),U2(N),RES(N),RHO,NU
  INTEGER :: N
  
  REAL(KIND=8) :: QU1(N),QU2(N)
  REAL(KIND=8) :: C1,C2(N)
  REAL(KIND=8) , PARAMETER :: PI = 3.1415926535897932

  CALL QT_VEC(U1,QU1,NU,N)
  CALL QT_VEC(U2,QU2,NU,N)  


  C1 = EXP(LOG_GAMMA(NU/2)+LOG_GAMMA(NU/2+1)-2*LOG_GAMMA(NU/2+0.5D00))&
       / SQRT(1-RHO**2)

  C2 = (1+(QU1**2+QU2**2-2*RHO*QU1*QU2)/NU/(1-RHO**2))&
       **(NU/2+1)
     
  RES = C1 / C2 * (1+ QU1**2/NU)**(NU/2+0.5)&
       * (1+QU2**2/NU)**(NU/2+0.5)

END SUBROUTINE DBV_T_F



FUNCTION BV_T_LOGLIK_F(U1,U2,RHO,NU,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),RHO,NU,BV_T_LOGLIK_F
  INTEGER :: N

  CALL DBV_T_F(U1,U2,RES,RHO,NU,N)

  BV_T_LOGLIK_F = SUM(LOG(RES))

END FUNCTION BV_T_LOGLIK_F




SUBROUTINE BV_T_INIT(U1,U2,NNUM,INIT)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),INIT(2)
  ! LOCAL
  REAL(KIND=8)           :: QU1(NNUM),QU2(NNUM),X
  REAL(KIND=8)           :: LOW, UPP, LOGLIK_L, LOGLIK_R, STEP
  REAL(KIND=8)           :: EXP = 1D-8,TOL = 1D-2,FN_VAL
  REAL(KIND=8), EXTERNAL :: BV_T_LOGLIK_F, LOCAL_MIN_T

  STEP = 2D0

  CALL QNORM_VEC(U1,QU1,NNUM)
  CALL QNORM_VEC(U2,QU2,NNUM)

  QU1 = QU1 - SUM(QU1)/NNUM
  QU2 = QU2 - SUM(QU2)/NNUM

  INIT(1) = SUM(QU1*QU2) / SQRT( SUM(QU1**2) * SUM(QU2**2) )

  LOW = 2D0
  UPP = 1D2
  
  LOGLIK_L = BV_T_LOGLIK_F(U1,U2,INIT(1),LOW,NNUM)
  LOGLIK_R = BV_T_LOGLIK_F(U1,U2,INIT(1),LOW+STEP,NNUM)

  DO WHILE(LOGLIK_L.LT.LOGLIK_R.AND.LOW.LT.UPP)
     LOW = LOW + STEP
     
     LOGLIK_L = BV_T_LOGLIK_F(U1,U2,INIT(1),LOW,NNUM)
     LOGLIK_R = BV_T_LOGLIK_F(U1,U2,INIT(1),LOW+STEP,NNUM)
  END DO

  IF(LOW.GE.UPP)THEN
     X = 1D2
  ELSE
     UPP = LOW + STEP
     LOW = MAX(LOW - STEP,2D0)
     FN_VAL = -LOCAL_MIN_T(LOW,UPP,EXP,TOL,BV_T_LOGLIK_F,U1,U2,INIT(1),NNUM,X)
  END IF

  INIT(2) = X

! CONTAINS 

!   FUNCTION TARGET_FUN(NU)
!     REAL(KIND=8)   :: NU,TARGET_FUN
!     TARGET_FUN = -BV_T_LOGLIK_F(U1,U2,INIT(1),NU,NNUM)
!   END FUNCTION TARGET_FUN

END SUBROUTINE BV_T_INIT


SUBROUTINE BV_T_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER,  PARAMETER    :: N = 2, M = 5, IPRINT = -1
  REAL(KIND=8), PARAMETER:: FACTR  = 1.0D+7, PGTOL  = 1.0D-5

  CHARACTER(LEN=60)      :: TASK, CSAVE
  LOGICAL                :: LSAVE(4)
  INTEGER                :: ISAVE(44)
  REAL(KIND=8)           :: F
  REAL(KIND=8)           :: DSAVE(29)
  INTEGER,  ALLOCATABLE  :: NBD(:), IWA(:)
  REAL(KIND=8), ALLOCATABLE  :: X(:), L(:), U(:), G(:), WA(:)
  

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  REAL(KIND=8)           :: EPS = 1.5D-08 ! EPS IS SQUARE ROOT OF
                                          ! MACHINE PRECISION
  REAL(KIND=8), EXTERNAL :: BV_T_LOGLIK_F

  !  ALLOCATE DYNAMIC ARRAYS

  ALLOCATE ( NBD(N), X(N), L(N), U(N), G(N) )
  ALLOCATE ( IWA(3*N) )
  ALLOCATE ( WA(2*M*N + 5*N + 11*M*M + 8*M) )

  NBD = 2

  CALL BV_T_INIT(U1,U2,NNUM,X)

  ! ABS(RHO) BOUNDS CLOSE TO ONE WILL CAUSE ABNORMAL
  L = (/MAX(-9.9D-1,X(1)-5D-2) , MAX(2D0,X(2)-2D0)/)
  U = (/MIN( 9.9D-1,X(1)+5D-2) , MIN(1D2,X(2)+2D0)/)

  !   WE START THE ITERATION BY INITIALIZING TASK.
 
  TASK = 'START'

  !   THE BEGINNING OF THE LOOP
 
  DO WHILE(TASK(1:2).EQ.'FG'.OR.TASK.EQ.'NEW_X'.OR. &
               TASK.EQ.'START') 
         
     !  THIS IS THE CALL TO THE L-BFGS-B CODE.
         
     CALL SETULB ( N, M, X, L, U, NBD, F, G, FACTR, PGTOL, &
          WA, IWA, TASK, IPRINT,&
          CSAVE, LSAVE, ISAVE, DSAVE )
         
     IF (TASK(1:2) .EQ. 'FG') THEN
        ! COMPUTE FUNCTION VALUE F
        F = -BV_T_LOGLIK_F(U1,U2,X(1),X(2),NNUM)
        
        ! COMPUTE GRADIENT G FOR THE SAMPLE PROBLEM.
        G(1) = (-BV_T_LOGLIK_F(U1,U2,X(1)+EPS,X(2),NNUM) - F) / EPS
        G(2) = (-BV_T_LOGLIK_F(U1,U2,X(1),X(2)+EPS,NNUM) - F) / EPS
        
     END IF

  END DO

  RES(1:2) = X
  LL = -F

END SUBROUTINE BV_T_MLE



SUBROUTINE PBV_CLAYTON_F(U1,U2,RES,DELTA,N)
  
  IMPLICIT NONE
  
  REAL(KIND=8) :: U1(N),U2(N),RES(N)
  REAL(KIND=8) :: DELTA,V,POSINF=HUGE(0.0D00)
  INTEGER::N,I

  IF(DELTA.EQ.0.0D00)THEN
     RES = U1 * U2
  ELSE
     DO I=1,N
        V = MIN(U1(I),U2(I))
        IF(((1/U1(I))**DELTA+(1/U2(I))**DELTA-1).GT.POSINF)THEN
           RES(I) = V
        ELSE
           RES(I) = ((1/U1(I))**DELTA+(1/U2(I))**DELTA-1)**(-1/DELTA)
        END IF
     END DO
  END IF
  
END SUBROUTINE PBV_CLAYTON_F

SUBROUTINE DBV_CLAYTON_F(U1,U2,RES,DELTA,N)
  REAL(KIND=8) :: U1(N),U2(N),RES(N),C(N),R(N),U1D(N),U2D(N),RD(N)
  REAL(KIND=8) :: DELTA
  INTEGER::N

  IF(DELTA.EQ.0.0D00)THEN
     RES = 1.
  ELSE
     CALL PBV_CLAYTON_F(U1,U2,C,DELTA,N)
     R = U1/U2
     RD = R ** DELTA
     U1D = U1 ** DELTA
     U2D = U2 ** DELTA
     RES = (1+DELTA) * C / (U1*U2) / (RD+1/RD+2-2*U1D-2*U2D+U1D*U2D)
  END IF

END SUBROUTINE DBV_CLAYTON_F


FUNCTION BV_CLAYTON_LOGLIK_F(U1,U2,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA,BV_CLAYTON_LOGLIK_F
  INTEGER :: N

  CALL DBV_CLAYTON_F(U1,U2,RES,DELTA,N)

  BV_CLAYTON_LOGLIK_F = SUM(LOG(RES))
  

END FUNCTION BV_CLAYTON_LOGLIK_F



SUBROUTINE BV_CLAYTON_INIT(U1,U2,NNUM,LOW_INIT,UPP_INIT)
  
  IMPLICIT NONE
  
  INTEGER               :: NNUM
  REAL(KIND=8)          :: U1(NNUM),U2(NNUM)
  REAL(KIND=8)          :: LOW_INIT,UPP_INIT
  ! LOCAL
  REAL(KIND=8),EXTERNAL :: BV_CLAYTON_LOGLIK_F
  REAL(KIND=8)          :: LOGLIK_L,LOGLIK_R,STEP

  LOW_INIT = 1D-3

  STEP = 5D0

  LOGLIK_L = BV_CLAYTON_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
  LOGLIK_R = BV_CLAYTON_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  DO WHILE(LOGLIK_L.LT.LOGLIK_R)
     LOW_INIT = LOW_INIT + STEP
     
     LOGLIK_L = BV_CLAYTON_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
     LOGLIK_R = BV_CLAYTON_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  END DO

  UPP_INIT = LOW_INIT + STEP
  LOW_INIT = MAX(LOW_INIT - STEP, 1D-3)

END SUBROUTINE BV_CLAYTON_INIT


SUBROUTINE BV_CLAYTON_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  ! EPS IS USED FOR COMPUTING G, WHICH IS GRADIENT OF F, THIS NUMBER
  ! COMES FROM THE _EPSILON USED IN SCIPY.OPTIMIZE.APPROX_FPRIME, AND
  ! _EPSILON = SQRT(NUMPY.FINFO(FLOAT).EPS)
  REAL(KIND=8)           :: EPS = 1.5D-08, TOL = 1D-3, LOW, UPP, X
  REAL(KIND=8), EXTERNAL :: BV_CLAYTON_LOGLIK_F, LOCAL_MIN

  CALL BV_CLAYTON_INIT(U1,U2,NNUM,LOW,UPP)

  LL = -LOCAL_MIN(LOW,UPP,EPS,TOL,BV_CLAYTON_LOGLIK_F,U1,U2,NNUM,X)

  RES(1) = X
  RES(2) = 0D0

! CONTAINS
  
!   FUNCTION TARGET_FUN(DELTA)
!     IMPLICIT NONE
!     REAL(KIND=8) :: DELTA, TARGET_FUN
!     TARGET_FUN = -BV_CLAYTON_LOGLIK_F(U1,U2,DELTA,NNUM)
!   END FUNCTION TARGET_FUN
  
END SUBROUTINE BV_CLAYTON_MLE



SUBROUTINE PBV_GUMBEL_F(U1,U2,RES,DELTA,N)
  REAL(KIND=8) :: U1(N),U2(N),RES(N)
  REAL(KIND=8) :: DELTA,POSINF=HUGE(1.0D+00),UMAX,UMIN,B
  INTEGER::N,I

  IF(DELTA==1.)THEN
     RES = U1 * U2
  ELSE
     DO I=1,N
        IF((U1(I)==0).OR.(U2(I)==0))THEN
           RES(I) = 0
        ELSE IF(U1(I)==1)THEN
           RES(I) = U2(I)
        ELSE IF(U2(I)==1)THEN
           RES(I) = U1(I)
        ELSE
           UMAX = MAX(U1(I),U2(I))
           UMIN = MIN(U1(I),U2(I))
           IF((LOG(1/UMIN)/LOG(1/UMAX))**DELTA>POSINF)THEN
              RES(I) = UMIN
           ELSE
              B = (1+(LOG(1/UMIN)/LOG(1/UMAX))**DELTA)**(1/DELTA)&
                   *LOG(1/UMAX)
              RES(I) = EXP(-B)
           END IF
        END IF
     END DO
  END IF
END SUBROUTINE PBV_GUMBEL_F


SUBROUTINE DBV_GUMBEL_F(U1,U2,RES,DELTA,N)

  IMPLICIT NONE
  
  REAL(KIND=8) :: U1(N),U2(N),RES(N),C(N),UD1(N),UD2(N)
  REAL(KIND=8) :: DELTA,POSINF=HUGE(1.0D+00),R
  INTEGER::N,I

  IF(DELTA.EQ.1.0D00)THEN
     RES = 1.0D00
  ELSE
     
     CALL PBV_GUMBEL_F(U1,U2,C,DELTA,N)

     UD1 = LOG(1/U1)
     UD2 = LOG(1/U2)

     DO I = 1,N
        
        R = UD1(I) / UD2(I)
        
        IF(R.LT.1.0D00)THEN
           R = 1.0D00 / R
        END IF

        IF(R**(2.0D00*DELTA).GT.POSINF)THEN
           RES(I) = C(I) / (U1(I)*U2(I)) &
                * ( R**(1-DELTA) &
                + (DELTA-1) * R**(-DELTA) / MIN(UD1(I),UD2(I)))
        ELSE
           RES(I) = C(I) / (U1(I)*U2(I)) &
                * ( R**(DELTA-1) / (1+R**DELTA)**(2-2/DELTA) &
                + (DELTA-1) * R**(DELTA-1) / MIN(UD1(I),UD2(I)) / &
                (1+R**DELTA)**(2-1/DELTA))
        END IF

     END DO

  END IF

END SUBROUTINE DBV_GUMBEL_F




FUNCTION BV_GUMBEL_LOGLIK_F(U1,U2,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA,BV_GUMBEL_LOGLIK_F
  INTEGER :: N

  CALL DBV_GUMBEL_F(U1,U2,RES,DELTA,N)

  BV_GUMBEL_LOGLIK_F = SUM(LOG(RES))
  

END FUNCTION BV_GUMBEL_LOGLIK_F




SUBROUTINE BV_GUMBEL_INIT(U1,U2,NNUM,LOW_INIT,UPP_INIT)
  IMPLICIT NONE

  INTEGER               :: NNUM
  REAL(KIND=8)          :: U1(NNUM),U2(NNUM),LOW_INIT,UPP_INIT
  ! LOCAL
  REAL(KIND=8),EXTERNAL :: BV_GUMBEL_LOGLIK_F
  REAL(KIND=8)          :: LOGLIK_L,LOGLIK_R,STEP

  LOW_INIT = 1.001D0

  STEP = 5D0

  LOGLIK_L = BV_GUMBEL_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
  LOGLIK_R = BV_GUMBEL_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  DO WHILE(LOGLIK_L.LT.LOGLIK_R)
     LOW_INIT = LOW_INIT + STEP
     
     LOGLIK_L = BV_GUMBEL_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
     LOGLIK_R = BV_GUMBEL_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  END DO

  UPP_INIT = LOW_INIT + STEP
  LOW_INIT = MAX(LOW_INIT - STEP, 1.001D0)

END SUBROUTINE BV_GUMBEL_INIT



SUBROUTINE BV_GUMBEL_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  REAL(KIND=8)           :: EPS = 1.5D-08, TOL = 1D-3, LOW, UPP, X
  REAL(KIND=8), EXTERNAL :: BV_GUMBEL_LOGLIK_F, LOCAL_MIN

  CALL BV_GUMBEL_INIT(U1,U2,NNUM,LOW,UPP)

  LL = -LOCAL_MIN(LOW,UPP,EPS,TOL,BV_GUMBEL_LOGLIK_F,U1,U2,NNUM,X)

  RES(1) = X
  RES(2) = 0D0

! CONTAINS
  
!   FUNCTION TARGET_FUN(DELTA)
!     IMPLICIT NONE
!     REAL(KIND=8) :: DELTA, TARGET_FUN
!     TARGET_FUN = -BV_GUMBEL_LOGLIK_F(U1,U2,DELTA,NNUM)
!   END FUNCTION TARGET_FUN

END SUBROUTINE BV_GUMBEL_MLE



SUBROUTINE DBV_FRANK_F(U1,U2,RES,DELTA,N)
  REAL(KIND=8):: U1(N),U2(N),RES(N)
  REAL(KIND=8):: DELTA
  INTEGER::N

  IF(DELTA.EQ.0)THEN
     RES = 1
     RETURN
  ENDIF

  IF(DELTA.LT.0)THEN
     CALL DBV_FRANK_F_N(U1,U2,RES,DELTA,N)
  ELSE
     CALL DBV_FRANK_F_N(1-U1,U2,RES,-DELTA,N)
  ENDIF

CONTAINS

  SUBROUTINE DBV_FRANK_F_N(U1,U2,RES,DELTA,N)
    REAL(KIND=8):: U1(N),U2(N),RES(N),DENOMINATOR(N)
    REAL(KIND=8):: DELTA
    INTEGER::N

    DENOMINATOR = &
         EXP(DELTA*(U1+U2-1))&
         +EXP(DELTA*(1-U1-U2))&
         +EXP(DELTA*(U2+1-U1))&
         +EXP(DELTA*(U1+1-U2))&
         -2*EXP(DELTA*(1-U1))&
         -2*EXP(DELTA*(1-U2))&
         +2&
         +2*EXP(DELTA)&
         -2*EXP(DELTA*U2)&
         -2*EXP(DELTA*U1)

    RES = -DELTA*(1-EXP(DELTA)) / DENOMINATOR

  END SUBROUTINE DBV_FRANK_F_N

END SUBROUTINE DBV_FRANK_F





FUNCTION BV_FRANK_LOGLIK_F(U1,U2,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA,BV_FRANK_LOGLIK_F
  INTEGER :: N

  CALL DBV_FRANK_F(U1,U2,RES,DELTA,N)
  BV_FRANK_LOGLIK_F = SUM(LOG(RES))
  

END FUNCTION BV_FRANK_LOGLIK_F



SUBROUTINE BV_FRANK_INIT(U1,U2,NNUM,LOW_INIT,UPP_INIT)
  IMPLICIT NONE

  INTEGER               :: NNUM
  REAL(KIND=8)          :: U1(NNUM),U2(NNUM),LOW_INIT,UPP_INIT
  REAL(KIND=8),EXTERNAL :: BV_FRANK_LOGLIK_F
  REAL(KIND=8)          :: LOGLIK_L,LOGLIK_R,STEP
  REAL(KIND=8)          :: M1,M2,UM1(NNUM),UM2(NNUM),R

  M1 = SUM(U1) / NNUM
  M2 = SUM(U2) / NNUM
  UM1 = U1 - M1
  UM2 = U2 - M2

  R = SUM( UM1 * UM2 ) / SQRT( SUM(UM1**2) * SUM(UM2**2) )

  STEP = 5D0

  IF(R.GE.0D0)THEN
     
     LOW_INIT = 1D-3
     
     LOGLIK_L = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
     LOGLIK_R = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
     
     DO WHILE(LOGLIK_L.LT.LOGLIK_R)
        LOW_INIT = LOW_INIT + STEP
        
        LOGLIK_L = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
        LOGLIK_R = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
     END DO

     UPP_INIT = LOW_INIT + STEP
     LOW_INIT = MAX(LOW_INIT - STEP, 1D-3)

  ELSE
     
     LOW_INIT = -1D-3
     
     LOGLIK_L = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
     LOGLIK_R = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT-STEP,NNUM)
     
     DO WHILE(LOGLIK_L.LT.LOGLIK_R)
        LOW_INIT = LOW_INIT - STEP
        
        LOGLIK_L = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
        LOGLIK_R = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT-STEP,NNUM)
     END DO

     LOW_INIT = LOW_INIT - STEP
     UPP_INIT = MIN(LOW_INIT + 2D0 * STEP, -1D-3)

  END IF


END SUBROUTINE BV_FRANK_INIT


SUBROUTINE BV_FRANK_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  ! EPS IS USED FOR COMPUTING G, WHICH IS GRADIENT OF F, THIS NUMBER
  ! COMES FROM THE _EPSILON USED IN SCIPY.OPTIMIZE.APPROX_FPRIME, AND
  ! _EPSILON = SQRT(NUMPY.FINFO(FLOAT).EPS)
  REAL(KIND=8)           :: EPS = 1.5D-8, TOL = 1D-3, LOW, UPP, X
  REAL(KIND=8), EXTERNAL :: BV_FRANK_LOGLIK_F, LOCAL_MIN
  
  CALL BV_FRANK_INIT(U1,U2,NNUM,LOW,UPP)

  LL = -LOCAL_MIN(LOW,UPP,EPS,TOL,BV_FRANK_LOGLIK_F,U1,U2,NNUM,X)

  RES(1) = X
  RES(2) = 0D0

! CONTAINS
  
!   FUNCTION TARGET_FUN(DELTA)
!     IMPLICIT NONE
!     REAL(KIND=8) :: DELTA, TARGET_FUN
!     TARGET_FUN = -BV_FRANK_LOGLIK_F(U1,U2,DELTA,NNUM)
!   END FUNCTION TARGET_FUN

END SUBROUTINE BV_FRANK_MLE




SUBROUTINE DBV_JOE_F(U1,U2,RES,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA
  INTEGER :: N

  ! LOCAL VARIABLES
  REAL(KIND=8) :: Y,A,B,R,F_DELTA,G_DELTA,POSINF=HUGE(0.0D00)
  INTEGER :: I


  IF(DELTA.EQ.1.0D00)THEN
     RES = 1.0D00
     RETURN     
  END IF

  F_DELTA = DELTA / (2*DELTA-1)
  G_DELTA = DELTA**2 / (2*DELTA-1)
  

  DO I = 1,N

     A = 1-U1(I)
     B = 1-U2(I)
     Y = (1-A**DELTA)*(1-B**DELTA)
     R = A / B

     IF(R.GE.1)THEN
        IF(R**G_DELTA.GT.POSINF)THEN
           RES(I) = (DELTA-Y) / (R**DELTA * B)
        ELSE
           RES(I) = (DELTA-Y) &
                / (R**G_DELTA * B**F_DELTA + (1/R)**G_DELTA * A**F_DELTA -A**G_DELTA * B**G_DELTA)**(1/F_DELTA)
        END IF
     ELSE
        IF((1/R)**G_DELTA.GT.POSINF)THEN
           RES(I) = (DELTA-Y) / ((1/R)**DELTA * A)
        ELSE
           RES(I) = (DELTA-Y) &
                / (R**G_DELTA * B**F_DELTA + (1/R)**G_DELTA * A**F_DELTA -A**G_DELTA * B**G_DELTA)**(1/F_DELTA)
        END IF

     END IF

  END DO

END SUBROUTINE DBV_JOE_F




FUNCTION BV_JOE_LOGLIK_F(U1,U2,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA,BV_JOE_LOGLIK_F
  INTEGER :: N

  CALL DBV_JOE_F(U1,U2,RES,DELTA,N)
  BV_JOE_LOGLIK_F = SUM(LOG(RES))
  

END FUNCTION BV_JOE_LOGLIK_F



SUBROUTINE BV_JOE_INIT(U1,U2,NNUM,LOW_INIT,UPP_INIT)
  IMPLICIT NONE

  INTEGER               :: NNUM
  REAL(KIND=8)          :: U1(NNUM),U2(NNUM),LOW_INIT,UPP_INIT
  ! LOCAL
  REAL(KIND=8),EXTERNAL :: BV_JOE_LOGLIK_F
  REAL(KIND=8)          :: LOGLIK_L,LOGLIK_R,STEP

  LOW_INIT = 1.001D0
  
  STEP = 5D0
  
  LOGLIK_L = BV_JOE_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
  LOGLIK_R = BV_JOE_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  DO WHILE(LOGLIK_L.LT.LOGLIK_R)
     LOW_INIT = LOW_INIT + STEP
     
     LOGLIK_L = BV_JOE_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
     LOGLIK_R = BV_JOE_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  END DO

  UPP_INIT = LOW_INIT + STEP
  LOW_INIT = MAX(LOW_INIT - STEP,1.001D0)

END SUBROUTINE BV_JOE_INIT


SUBROUTINE BV_JOE_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  ! LOCAL
  REAL(KIND=8)           :: EPS = 1.5D-8, TOL = 1D-3, LOW, UPP, X
  REAL(KIND=8), EXTERNAL :: BV_JOE_LOGLIK_F, LOCAL_MIN

  CALL BV_JOE_INIT(U1,U2,NNUM,LOW,UPP)

  LL = -LOCAL_MIN(LOW,UPP,EPS,TOL,BV_JOE_LOGLIK_F,U1,U2,NNUM,X)

  RES(1) = X
  RES(2) = 0D0

! CONTAINS
  
!   FUNCTION TARGET_FUN(DELTA)
!     IMPLICIT NONE
!     REAL(KIND=8) :: DELTA, TARGET_FUN
!     TARGET_FUN = -BV_JOE_LOGLIK_F(U1,U2,DELTA,NNUM)
!   END FUNCTION TARGET_FUN

END SUBROUTINE BV_JOE_MLE



SUBROUTINE MULTI_PROCESS_MLE(DF,NR,NC,FML_LST,NF,RES,FML_VEC,LL_VEC,PNUM,THREADS_NUM)
  !
  ! SUBROUTINE FOR THE PARALLEL MODEL SELECTION WORK FOR EVERY EDGE IN
  ! A PARTICULAR VINE TREE.
  !
  ! PARAMETER:
  !
  ! DF           : INPUT. DATA COLLECTED FROM EACH EDGE OF A PARTICULAR VINE
  !                TREE. DF(:,2I-1) AND DF(:,2I) COMES FROM THE SAME EDGE.
  !              
  ! NR           : INPUT. ROW NUMBER OF DF.
  !              
  ! NC           : INPUT. COLUMN NUMBER OF DF, MUST BE EVEN.
  !              
  ! FML_LST      : INPUT. AN INTEGER ARRAY WITH LENGTH NF, INDICATING WHICH
  !                FAMILIES OF BIVARIATE COPULA TO BE CHOSEN FROM.
  !              
  ! NF           : INPUT. LENGTH OF FML_LST.
  !              
  ! RES          : OUTPUT. 2 X PNUM MATRIX, EACH COLUMN CONTAINS THE PARAMETER 
  !                LIST OF THE CORRESPONDING EDGE
  !              
  ! FML_VEC      : OUTPUT. ARRAY WITH PNUM LENGTH, INDICATING WHICH BV COPULA FAMILY IS 
  !                BEST SUITABLE FOR THE CORRESPONDING EDGES.
  !              
  ! LL_VEC       : OUTPUT. ARRAY WITH PNUM LENGTH, THE LOGLIKELIHOOD VALUE FOR 
  !                CORRESPONDING EDGES.
  !              
  ! PNUM         : INPUT. NUMBER OF PAIRED DATA ARRAY.
  !
  ! THREADS_NUM  : INTEGER. THE VALUE INDICATES THE NUMBER OF THREADS
  !                THAT WILL BE USED IN THE PARALLEL MLE WORK. IF THIS
  !                NUMBER IS SETTED MORE THAN THE MAX NUMBER OF CPU 
  !                CORES, THE LATTER, WHICH IS RETURNED VIA OMP_GET_NUM_PROCS().
  !                WILL BE USED. DEFAULT VALUE IS 1.

  USE OMP_LIB
  IMPLICIT NONE
  
  INTEGER      :: NF,NC,NR,PNUM
  INTEGER      :: FML_LST(NF)
  REAL(KIND=8) :: DF(NR,NC)
  REAL(KIND=8) :: RES(2,PNUM),LL_VEC(PNUM)
  INTEGER      :: FML_VEC(PNUM)
  ! LOCAL
  INTEGER      :: I,THREADS_NUM,PROCS_NUM

  PROCS_NUM = OMP_GET_NUM_PROCS()

  
  IF(PROCS_NUM.LT.THREADS_NUM)THEN
     CALL OMP_SET_NUM_THREADS(PROCS_NUM)
  ELSE
     CALL OMP_SET_NUM_THREADS(THREADS_NUM)
  END IF


  !$OMP PARALLEL DO PRIVATE(I) FIRSTPRIVATE(PNUM) SHARED(DF,NR,FML_LST,NF,RES,LL_VEC)
  DO I = 1,PNUM
     CALL BV_COP_MODEL_SELECTION(DF(:,2*I-1),DF(:,2*I),NR,FML_LST,NF,RES(:,I),FML_VEC(I),LL_VEC(I))
  END DO
  !$OMP END PARALLEL DO

END SUBROUTINE MULTI_PROCESS_MLE



SUBROUTINE BV_COP_MODEL_SELECTION(U1,U2,NR,FML_LST,NF,RES,FML,LL)
  !
  ! MODIFY DATE : TUE NOV 20
  ! 
  ! SUBROUTINE FOR THE BV COPULA MODEL SELECTION FOR TWO ARRAYS U1 AND
  ! U2.
  !
  ! PARAMETER:
  !
  ! U1,U2   : INPUT. DATA ARRAY
  ! 
  ! NR      : INPUT. ROW NUMBER OF DF.
  ! 
  ! FML_LST : INPUT. AN INTEGER ARRAY WITH LENGTH NF, INDICATING WHICH
  !           FAMILIES OF BIVARIATE COPULA TO BE CHOSEN FROM.
  !
  ! NF      : INPUT. LENGTH OF FML_LST.
  !
  ! RES     : OUTPUT. ARRAY WITH LENGTH 2. MLE RESULT OF THE CHOSEN BV
  !           COPULA FAMILY.
  !
  ! FML     : OUTPUT. INTEGER INDICATING THE CHOSEN FAMILY FOR DATA ARRAY
  !           U1 AND U2.
  !
  ! LL      : OUTPUT. LOGLIKELIHOOD VALUE.

  IMPLICIT NONE

  INTEGER      :: NR,NF,FML_LST(NF),FML
  REAL(KIND=8) :: U1(NR),U2(NR),RES(2),LL
  ! LOCAL
  INTEGER      :: I,PNT(1)
  REAL(KIND=8) :: TMP_RES(2,NF),TMP_LL(NF)

  DO I = 1, NF
     CALL BV_COP_MLE(U1,U2,NR,FML_LST(I),TMP_RES(:,I),TMP_LL(I))
  END DO

  ! MAXLOC RETURNS 1X1 ARRAY
  PNT = MAXLOC(TMP_LL)
  FML = FML_LST(PNT(1))
  RES = TMP_RES(:,PNT(1))
  LL = TMP_LL(PNT(1))

END SUBROUTINE BV_COP_MODEL_SELECTION



SUBROUTINE BV_COP_MLE(U1,U2,NR,FML,RES,LL)
  ! THIS SUBROUTINE WRAPS THE MLE PROCEDURE FOR TEN POPULAR BV COPULA.
  IMPLICIT NONE

  REAL(KIND=8) :: U1(NR),U2(NR),RES(2),LL
  INTEGER      :: FML,NR

  IF(FML.EQ.1)THEN
     CALL BV_NORM_MLE(U1,U2,NR,RES,LL)
  ELSE IF(FML.EQ.2)THEN
     CALL BV_T_MLE(U1,U2,NR,RES,LL)
  ELSE IF(FML.EQ.3)THEN
     CALL BV_CLAYTON_MLE(U1,U2,NR,RES,LL)
  ELSE IF(FML.EQ.4)THEN
     CALL BV_GUMBEL_MLE(U1,U2,NR,RES,LL)
  ELSE IF(FML.EQ.5)THEN
     CALL BV_FRANK_MLE(U1,U2,NR,RES,LL)
  ELSE IF(FML.EQ.6)THEN
     CALL BV_JOE_MLE(U1,U2,NR,RES,LL)
  END IF
  
END SUBROUTINE BV_COP_MLE


SUBROUTINE BV_COP_LOGLIK(U1,U2,NR,PAR,FML,LL)
  
  IMPLICIT NONE
  
  REAL(KIND=8) :: U1(NR),U2(NR),PAR(2),LL
  INTEGER      :: FML,NR

  REAL(KIND=8),EXTERNAL :: BV_NORM_LOGLIK_F
  REAL(KIND=8),EXTERNAL :: BV_T_LOGLIK_F
  REAL(KIND=8),EXTERNAL :: BV_CLAYTON_LOGLIK_F
  REAL(KIND=8),EXTERNAL :: BV_GUMBEL_LOGLIK_F
  REAL(KIND=8),EXTERNAL :: BV_FRANK_LOGLIK_F
  REAL(KIND=8),EXTERNAL :: BV_JOE_LOGLIK_F  

  IF(FML.EQ.1)THEN
     LL = BV_NORM_LOGLIK_F(U1,U2,PAR(1),NR)
  ELSE IF(FML.EQ.2)THEN
     LL = BV_T_LOGLIK_F(U1,U2,PAR(1),PAR(2),NR)
  ELSE IF(FML.EQ.3)THEN
     LL = BV_CLAYTON_LOGLIK_F(U1,U2,PAR(1),NR)
  ELSE IF(FML.EQ.4)THEN
     LL = BV_GUMBEL_LOGLIK_F(U1,U2,PAR(1),NR)
  ELSE IF(FML.EQ.5)THEN
     LL = BV_FRANK_LOGLIK_F(U1,U2,PAR(1),NR)
  ELSE IF(FML.EQ.6)THEN
     LL = BV_JOE_LOGLIK_F(U1,U2,PAR(1),NR)
  END IF
  
END SUBROUTINE BV_COP_LOGLIK


SUBROUTINE MULTI_PROCESS_LOGLIK(DF,NR,NC,PAR_VEC,FML_VEC,PNUM,THREADS_NUM,LL)
  ! 'MULTI_PROCESS_LOGLIK' COMPUTES SUMMATION OF EDGE LOGLIKELIHOOD
  ! VALUES OF A GIVEN VINE TREE.
  !
  !   PARAMETER
  !
  !     DF          :  INPUT. DOUBLE WITH SHAPE NR X NC.
  !     NR          :  INPUT,HIDE. INTEGER. ROW NUMBER OF DF.
  !     NC          :  INPUT,HIDE. INTEGER. COL NUMBER OF DF.
  !     PAR_VEC     :  INPUT. DOUBLE. PARAMETER VECTOR OF EACH EDGE.
  !     FML_VEC     :  INPUT. DOUBLE. FAMILY VECTOR OF EACH EDGE
  !     PNUM        :  INPUT. INTEGER. EDGE NUMBER.
  !     THREADS_NUM :  INPUT. INTEGER. NUMBER OF THREADS FOR FOPENMP.
  !     LL          :  OUTPUT. DOUBLE. LOGLIKELIHOOD VALUE OF THE VINE TREE.
  !
  USE OMP_LIB  

  IMPLICIT NONE

  INTEGER      :: NR,NC,PNUM,THREADS_NUM
  REAL(KIND=8) :: DF(NR,NC),PAR_VEC(2,PNUM),LL
  INTEGER      :: FML_VEC(PNUM)

  ! LOCAL
  INTEGER      :: I,PROCS_NUM
  REAL(KIND=8) :: LL_VEC(PNUM)


  PROCS_NUM = OMP_GET_NUM_PROCS()
  
  IF(PROCS_NUM.LT.THREADS_NUM)THEN
     CALL OMP_SET_NUM_THREADS(PROCS_NUM)
  ELSE
     CALL OMP_SET_NUM_THREADS(THREADS_NUM)
  END IF

  !$OMP PARALLEL DO PRIVATE(I) FIRSTPRIVATE(PNUM) SHARED(DF,NR,FML_VEC,PAR_VEC,LL_VEC)
  DO I = 1, PNUM
     CALL BV_COP_LOGLIK(DF(:,2*I-1),DF(:,2*I),NR,PAR_VEC(:,I),FML_VEC(I),LL_VEC(I))
  END DO
  !$OMP END PARALLEL DO

  LL = SUM(LL_VEC)

END SUBROUTINE MULTI_PROCESS_LOGLIK



  

SUBROUTINE ADINF(Z,RES,N)
  ! COMPUTES THE LIMITED ANDERSON-DARLING DISTRIBUTION ACCORDING TO
  ! THE EXPRESSION IN PP.2 OF "EVALUATING THE ANDERSON-DARLING
  ! DISTRIBUTION", GEORGE MARSAGLIA, JOHN C. W. MARSAGLIA
  !
  !   PARAMETER
  !
  !     Z    : INPUT. DOUBLE WITH LENGTH N
  !     RES  : OUTPUT. DOUBLE WITH LENGTH N
  !     N    : INPUT. HIDE. INTEGER OF LENGTH OF Z.
  IMPLICIT NONE

  REAL(KIND=8) :: Z(N),RES(N),ZZ
  INTEGER      :: N,I
  
  DO I = 1,N
     ZZ = Z(I)
     IF(ZZ.LE.2D0)THEN
        RES(I) = ZZ ** (-5D-1) * EXP(-1.2337141D0 / ZZ) * &
             (2.00012D0 + (2.47105D-1 - (6.49821D-2 - (3.47962D-2 - (1.1672D-2 - 1.68691D-3 * ZZ) *ZZ) *ZZ) *ZZ) * ZZ)
     ELSE
        RES(I) = EXP(-EXP(1.0776D0-(2.30695D0-(4.3424D-1-(8.2433D-2-(8.056D-3-3.146D-4 * ZZ) * ZZ) *ZZ) *ZZ) *ZZ))
     END IF
  END DO
  

END SUBROUTINE ADINF

! MODULE FOR QUICK SORT BELOW COMES FROM 
! 'WWW.FORTRAN.COM/QSORT_C.F95'
MODULE QSORT_C_MODULE

  IMPLICIT NONE
  PUBLIC :: QSORTC
  PRIVATE :: PARTITION

CONTAINS

  RECURSIVE SUBROUTINE QSORTC(A)
    REAL(KIND=8), INTENT(IN OUT), DIMENSION(:) :: A
    INTEGER :: IQ

    IF(SIZE(A) > 1) THEN
       CALL PARTITION(A, IQ)
       CALL QSORTC(A(:IQ-1))
       CALL QSORTC(A(IQ:))
    ENDIF
  END SUBROUTINE QSORTC

  SUBROUTINE PARTITION(A, MARKER)
    REAL(KIND=8), INTENT(IN OUT), DIMENSION(:) :: A
    INTEGER, INTENT(OUT) :: MARKER
    INTEGER :: I, J
    REAL(KIND=8) :: TEMP
    REAL(KIND=8) :: X      ! PIVOT POINT
    X = A(1)
    I= 0
    J= SIZE(A) + 1

    DO
       J = J-1
       DO
          IF (A(J) <= X) EXIT
          J = J-1
       END DO
       I = I+1
       DO
          IF (A(I) >= X) EXIT
          I = I+1
       END DO
       IF (I < J) THEN
          ! EXCHANGE A(I) AND A(J)
          TEMP = A(I)
          A(I) = A(J)
          A(J) = TEMP
       ELSEIF (I == J) THEN
          MARKER = I+1
          RETURN
       ELSE
          MARKER = I
          RETURN
       ENDIF
    END DO

  END SUBROUTINE PARTITION


  RECURSIVE SUBROUTINE QSORTC_2(A,B)
    REAL(KIND=8), INTENT(IN OUT), DIMENSION(:) :: A,B
    INTEGER :: IQ

    IF(SIZE(A) > 1) THEN
       CALL PARTITION_2(A,B, IQ)
       CALL QSORTC_2(A(:IQ-1),B(:IQ-1))
       CALL QSORTC_2(A(IQ:),B(IQ:))
    ENDIF
  END SUBROUTINE QSORTC_2

  SUBROUTINE PARTITION_2(A,B, MARKER)
    REAL(KIND=8), INTENT(IN OUT), DIMENSION(:) :: A,B
    INTEGER, INTENT(OUT) :: MARKER
    INTEGER :: I, J
    REAL(KIND=8) :: TEMP
    REAL(KIND=8) :: X      ! PIVOT POINT
    X = A(1)
    I= 0
    J= SIZE(A) + 1

    DO
       J = J-1
       DO
          IF (A(J) <= X) EXIT
          J = J-1
       END DO
       I = I+1
       DO
          IF (A(I) >= X) EXIT
          I = I+1
       END DO
       IF (I < J) THEN
          ! EXCHANGE A(I) AND A(J), B(I) AND B(J)
          TEMP = A(I)
          A(I) = A(J)
          A(J) = TEMP
          TEMP = B(I)
          B(I) = B(J)
          B(J) = TEMP
       ELSEIF (I == J) THEN
          MARKER = I+1
          RETURN
       ELSE
          MARKER = I
          RETURN
       ENDIF
    END DO

  END SUBROUTINE PARTITION_2

END MODULE QSORT_C_MODULE


FUNCTION AD_STAT(U,N)
  ! ROUTINE FOR COMPUTING THE ANDERSON-DARLING STATISTIC GIVEN ORDERED
  ! UNIFORMED DATA.
  !
  ! PARAMETER: 
  !
  !   U : INPUT. DOUBLE PRECISION ARRAY WITH LENGTH N. ELEMENTS OF 'U'
  !       HAS BOUNDS (0,1).
  !
  !   N : INPUT,HIDE. INTEGER. LENGTH OF U
  IMPLICIT NONE

  REAL(KIND=8) :: U(N)
  INTEGER ::N
  ! LOCAL
  REAL(KIND=8) :: AD_STAT
  INTEGER      :: I

  AD_STAT = 0D0
  DO I = 1,N
     AD_STAT = AD_STAT + &
          (2*I-1) * ( LOG(U(I)) + LOG(1-U(N+1-I)))
  END DO
  AD_STAT = -DBLE(N) - AD_STAT / DBLE(N)

END FUNCTION AD_STAT


SUBROUTINE KENDALLTAU(U1,U2,N,TAU)
  ! CALCULATE KENDALL TAU CORRELATION OF TWO ARRAYS.
  !
  ! PARAMETER
  !
  !   U1, U2 : INPUT. DOUBLE PRECISION ARRAYS.
  !   N      : INPUT,HIDE. INTEGER OF LENGTH OF U1 AND U2.
  !   TAU    : OUTPUT. DOUBLE PRECISION.
  USE QSORT_C_MODULE
  IMPLICIT NONE
  REAL(KIND=8) :: U1(N),U2(N),TAU
  INTEGER      :: N
  ! LOCAL
  REAL(KIND=8) :: X(N), Y(N)

  X = U1
  Y = U2

  CALL QSORTC_2(X,Y)
  CALL KTAU(N,X,Y,TAU)

END SUBROUTINE KENDALLTAU
