#!/usr/bin/env python
"""

    syntax:
        comma seperated list of args

        args:
            <short name>(<long name>)[<type>]=<default>

        all but short name are optional
        if long name is missing, short name will be used
        if long name is missing, and short name has len > 1
            long name will be short name and short name will be the
            first character of short name

    examples:
        f(foo)[int]=10
        f[int]=10

"""


import argparse


type_lookup = {'int': int, 'str': str, 'bool': bool, 'float': float, \
        'long': long, 'complex': complex, 'unicode': unicode, \
        'list': list, 'tuple': tuple, 'bytearray': bytearray, \
        'buffer': buffer, 'set': set, 'frozenset': frozenset, \
        'dict': dict, 'memoryview': memoryview}


def lookup_type(typestring):
    if typestring in type_lookup:
        return type_lookup[typestring]
    raise TypeError("Unknown type string: %s" % typestring)


def parse_token(token):
    """
    Return args and kwargs for ArgumentParser.add_argument
    """
    if '=' in token:
        token, default = token.split('=')
    else:
        default = None

    if '[' in token:
        token, dtype = token.split('[')
        dtype = dtype[:-1]  # remove ']'
        dtype = lookup_type(dtype)
    else:
        dtype = str

    if '(' in token:
        sname, lname = token.split('(')
        sname = sname[0]
        lname = lname[:-1]
    else:
        sname = token[0]
        lname = token
    if default is not None:
        default = dtype(default)

    args = ('-%s' % sname, '--%s' % lname)
    return args, dict(type=dtype, default=default)


def parse(arg_string, description=None):
    """
    make an argparser from arg_string
    """
    if description is None:
        description = "Autogenerated parser from string: %s" % arg_string
    parser = argparse.ArgumentParser(description=description)
    for t in arg_string.split(','):
        args, kwargs = parse_token(t)
        parser.add_argument(*args, **kwargs)
    return parser


def get(arg_string, args=None, namespace=None, \
        partial=False, return_leftover=False):
    p = parse(arg_string)
    if partial:
        ns, leftover = p.parse_known_args(args, namespace)
        if return_leftover:
            return ns, leftover
        return ns
    return p.parse_args(args, namespace)


def test():
    ns = get('f', args=['-f', 'foo'])
    assert hasattr(ns, 'f')
    assert type(ns.f) == str
    assert ns.f == 'foo'

    ns = get('f,b', args=['-f', 'foo', '-b', 'bar'])
    assert hasattr(ns, 'f')
    assert hasattr(ns, 'b')
    assert ns.f == 'foo'
    assert ns.b == 'bar'

    ns = get('f(foo)', args=['-f', 'foo'])
    assert hasattr(ns, 'foo')
    assert type(ns.foo) == str
    assert ns.foo == 'foo'

    ns = get('f(foo)[int]', args=['-f', '1'])
    assert type(ns.foo) == int
    assert ns.foo == 1

    ns = get('f(foo)[int]=1', args=['-f', '2'])
    assert ns.foo == 2

    ns = get('f(foo)[int]=1', args=[])
    assert ns.foo == 1
