import logging
import string
import os
import inspect
import re
from time import sleep

import boto.rds
import boto.ec2.elb
import boto.ec2.autoscale
import boto.ec2
from boto.ec2.autoscale import LaunchConfiguration
from boto.ec2.autoscale import AutoScalingGroup

from tier import Tier
from qds_ops.tparty.knife import Knife
from qds_ops.entities.rds import RDS


class DefaultDJ(Tier):
    def __init__(self, subparser):
        self.tier = "defaultDJ"
        self.ami_id = "ami-1327167a"
        self.region = "us-east-1"
        self.instance_type = "m1.xlarge"
        self.run_list = '{"run_list":["recipe[webslave::deploy_dj]"],"qubole_tier":"%s", "worker_name":"pju_worker"}'
        self.availibility_zone = "us-east-1b"
        self.modules = ["hive_scripts", "hadoop", "hustler", "hive", "sqoop", "oozie", "pig", "tapp2"]

        super(DefaultDJ, self).__init__(subparser)

        # Create
        create = self.tier_parser.add_parser("create", help="Create Cron Tier. "
                                                            "Command will return after instance is in running state")
        create.add_argument("-a", "--ami-id", help="AMI ID to use for the CRON tier",
                            default=self.ami_id, dest="ami_id")
        create.add_argument("-r", "--region", help="Region where the tier will be located",
                            default=self.region, dest="region")
        create.add_argument("-i", "--instance-type", help="Instance Type of the machine",
                            choices=["t1.micro", "m1.large", "m1.xlarge"], dest="instance_type",
                            default=self.instance_type)
        create.add_argument("-t", "--tunnel", dest="tunnel",
                            action="store_true", help="Use a tunnel through a web node")
        create.set_defaults(func=self.create)

    @staticmethod
    def get_lc_name(args):
        return "defaultdj_%s_launch_group" % args.environment

    @staticmethod
    def get_as_name(default_as_group, tier):
        return "%s-%s" % (default_as_group, tier)

    def create(self, args):
        as_lc_name = DefaultDJ.get_lc_name(args)

        logging.info("Getting credentials from Chef server and RDS")
        knife = Knife()
        creds = knife.get_credentials(args.environment)
        rows = RDS.run_query(args, "select value from rstore.settings where var='hadoop.keyname'")
        key_name = rows["value"]

        rows = RDS.run_query(args, "select value from rstore.settings where var='aws.as_group_name'")
        default_as_group = rows["value"]
        as_group_name = DefaultDJ.get_as_name(default_as_group, self.tier)

        logging.info("Connecting to AWS using boto")
        autoscale_conn = boto.ec2.autoscale.connect_to_region(
            args.region,
            aws_access_key_id=creds["access"],
            aws_secret_access_key=creds["secret"])

        for lc in autoscale_conn.get_all_launch_configurations():
            if re.match(str(as_lc_name), lc.name):
                raise Exception("AS-Launch-Config '%s' already exists!" % (lc.name))

        for group in autoscale_conn.get_all_groups():
            if re.match(as_group_name, group.name):
                raise Exception("AS-Group '%s' already exists!" % (group.name))

        logging.info("Creating new AutoScaling Launch Config named '%s' ..." % (as_lc_name))

        #open the UDF
        base_path = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
        filein = open(base_path + "/../scripts/udf.sh")
        src = string.Template(filein.read())
        resolved_runlist = self.run_list % self.tier
        d = {'runlist': resolved_runlist, 'environment': args.environment}
        user_data = src.safe_substitute(d)

        new_launch_config = LaunchConfiguration(
            name=as_lc_name,
            image_id=args.ami_id,
            key_name=key_name,
            security_groups=['default'],
            user_data=user_data,
            instance_type=self.instance_type)

        autoscale_conn.create_launch_configuration(new_launch_config)

        logging.info("Created new Autoscaling launch config named '%s' successfully\n\n" % (as_lc_name))

        logging.info("Creating new AS Group named '%s' ..." % (as_group_name))

        as_group = AutoScalingGroup(
            name=as_group_name,
            availability_zones=[self.availibility_zone],
            launch_config=new_launch_config,
            min_size=1,
            max_size=1,
            desired_capacity=1,
            connection=autoscale_conn)

        autoscale_conn.create_auto_scaling_group(as_group)

        sleep(10)  # wait for auto scaling activity to get registered

        logging.info("AS Group '%s' successfully created. Current activities:\n '%s'\n\n" \
                     % (as_group.name, as_group.get_activities()))