import json
from subprocess import Popen, PIPE
import inspect
import os
import signal

from qds_ops.tparty.knife import Knife


class Tier(object):
    def __init__(self, global_subparser):
        parser = global_subparser.add_parser(self.tier,
                                             help="Commands to monitor and control %s tier" % self.tier)
        self.tier_parser = parser.add_subparsers()

        #List
        list_parser = self.tier_parser.add_parser("list",
                                                  help="List all nodes in %s tier" % self.tier)
        list_parser.set_defaults(func=self.list)

        #Package
        package_parser = self.tier_parser.add_parser("package",
                                                     help="List all packages in %s tier" % self.tier)
        package_parser.set_defaults(func=self.packages)

        #SSH
        ssh_parser = self.tier_parser.add_parser("ssh",
                                                 help="SSH and run any bash command on machines in %s tier" % self.tier)
        ssh_parser.add_argument("command", help="Bash command to run")
        ssh_parser.add_argument("-a", "--any", dest="any",
                                action="store_true", help="Login into any node in %s" % (self.tier))
        ssh_parser.set_defaults(func=self.ssh)

        #Screen
        screen_parser = self.tier_parser.add_parser("screen",
                                                    help="Use GNU screen and open terminals on ALL machines in %s tier" % self.tier)
        screen_parser.set_defaults(func=self.screen)

        #Deploy
        module_choices = self.modules
        module_choices.extend(["all"])
        deploy_parser = self.tier_parser.add_parser("deploy",
                                                    help="Run deploy in %s tier" % self.tier)
        deploy_parser.add_argument("-m", "--module", dest="module", choices=module_choices,
                                   default="all", help="Deploy module to %s" % (self.tier))
        deploy_parser.add_argument("-r", "--no-rolling", dest="rolling",
                                   default=True, action="store_false",
                                   help="Don't do rolling deploy to %s" % (self.tier))
        deploy_parser.add_argument("-t", "--stop-djs-at-start", dest="stopdjs",
                                   default=False, action="store_true",
                                   help="At start of rolling deploy, stop DJs on current nodes")
        deploy_parser.add_argument("-c", "--run-common-stuff-recipe", dest="commstuff",
                                   default=False, action="store_true", help="Run common stuff at start of deploy")
        deploy_parser.add_argument("-d", "--no-djrestart", dest="djrestart",
                                   default=True, action="store_false", help="Don't do DJ restart on %s" % (self.tier))
        deploy_parser.add_argument("-s", "--srcpath", dest="src_path", required=True,
                                   default="~/src/", help="Root src directory")
        deploy_parser.set_defaults(func=self.deploy)

    def deploy(self, args):
        knife = Knife()
        chef_command = ""
        if args.module == "all":
            chef_command = "sudo chef-client -j /etc/chef/first-boot.json -E %s" % (args.environment)
        else:
            runlist = '{"run_list":["recipe[webslave::deploy_%s]"]}' % (args.module)
            chef_command = "echo '%s' > /tmp/runlist; sudo chef-client -j /tmp/runlist" % (runlist)

        knife.reap_instances(args)
        #Entering critical region. Interruption in this region causes a lot of grief
        #and manual merging of chef version files is required
        signal.signal(signal.SIGINT, signal.SIG_IGN)
        signal.signal(signal.SIGHUP, signal.SIG_IGN)
        signal.signal(signal.SIGTERM, signal.SIG_IGN)
        signal.signal(signal.SIGQUIT, signal.SIG_IGN)

        temp = knife.get_pemfile(args.environment)
        search_string = "chef_environment:%s AND qubole_tier:%s" % (args.environment, self.tier)
        p = Popen(
            ["knife", "ssh", search_string, chef_command, "-x", "ec2-user", "-i", temp, "-a", "ec2.public_hostname",
             "--no-host-key-verify"])
        retcode = p.wait()

        signal.signal(signal.SIGINT, signal.SIG_DFL)
        signal.signal(signal.SIGHUP, signal.SIG_DFL)
        signal.signal(signal.SIGTERM, signal.SIG_DFL)
        signal.signal(signal.SIGQUIT, signal.SIG_DFL)

        if retcode != 0:
            raise Exception("Deploy to %s tier failed. Return code: %d " % (self.tier, retcode))
        return retcode

    def list(self, args):
        knife_search = Knife()
        list = knife_search.search(args.environment, self.tier)
        print json.dumps(list, indent=4, sort_keys=True)

    def packages(self, args):
        knife_search = Knife()
        list = knife_search.packages(args.environment, self.tier)
        print json.dumps(list, indent=4, sort_keys=True)

    def ssh(self, args):
        knife = Knife()
        output = ""
        if args.any:
            output = knife.ssh_one(args.command, args.environment, self.tier)
        else:
            output = knife.ssh(args.command, args.environment, self.tier)
        print output

    def screen(self, args):
        knife = Knife()
        output = knife.ssh("screen", args.environment, self.tier)
        print output
