import string
import os
import inspect
import time
import json
import boto.ec2

from tier import Tier
from qds_ops.tparty.knife import Knife
from qds_ops.entities.rds import RDS

import requests

class Scheduler(Tier):
    def __init__(self, subparser):
        self.tier = "scheduler"
        self.ami_id = "ami-0598906c"
        self.region = "us-east-1"
        self.instance_type = "m1.large"
        self.run_list = '{"run_list":["recipe[webslave::deploy_scheduler]"],"qubole_tier":"scheduler"}'
        self.modules = ["scheduler"]

        super(Scheduler, self).__init__(subparser)

        # Create
        create = self.tier_parser.add_parser("create", help="Create Cron Tier")
        create.add_argument("-d", "--dry-run", help="Dry Run: Dont really create the tier",
                            default=False, dest="dry_run", action="store_true")
        create.add_argument("-a", "--ami-id", help="AMI ID to use for the CRON tier",
                            default=self.ami_id, dest="ami_id")
        create.add_argument("-r", "--region", help="Region where the tier will be located",
                            default=self.region, dest="region")
        create.add_argument("-i", "--instance-type", help="Instance Type of the machine",
                            choices=["t1.micro", "m1.large", "m1.xlarge"], dest="instance_type",
                            default=self.instance_type)
        create.add_argument("-t", "--tunnel", dest="tunnel",
                            action="store_true", help="Use a tunnel through a web node")
        create.set_defaults(func=self.create)

        # Remove
        delete = self.tier_parser.add_parser("delete", help="Delete Cron Tier from environment")
        delete.set_defaults(func=self.delete)

        #Suspend
        suspend = self.tier_parser.add_parser("suspend", help="Suspend scheduler. "
                                                              "All internal services will be stopped.")
        suspend.set_defaults(func=self.suspend)

        #Resume
        resume = self.tier_parser.add_parser("resume", help="Resume scheduler. All internal services will be started.")
        resume.set_defaults(func=self.resume)

        #Check status
        check = self.tier_parser.add_parser("check", help="Check status of scheduler. Suspended or running ?.")
        check.set_defaults(func=self.check)
    
    def get_tag(self, args):
        return "scheduler_%s" % args.environment

    def find_scheduler(self, conn, tag):
        reservations = conn.get_all_instances()
        for res in reservations:
            for inst in res.instances:
                if 'Name' in inst.tags and inst.tags["Name"] == tag:
                    return inst

        return None

    def create(self, args):
        knife = Knife()
        creds = knife.get_credentials(args.environment)
        rows = RDS.run_query(args, "select value from rstore.settings where var='hadoop.keyname'")

        #open the UDF
        base_path = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
        filein = open(base_path + "/../scripts/udf.sh")
        src = string.Template(filein.read())
        d = {'runlist': self.run_list, 'environment': args.environment}
        user_data = src.safe_substitute(d)

        conn = boto.ec2.connect_to_region(self.region,
                                          aws_access_key_id=creds["access"],
                                          aws_secret_access_key=creds["secret"])

        instance = self.find_scheduler(conn, self.get_tag(args))
        if not instance is None and instance.state != 'terminated':
            raise Exception("There is already a scheduler (%s, %s) in %s" % (instance.id, instance.state, self.tier))

        reservation = conn.run_instances(
            args.ami_id,
            key_name=rows["value"],
            instance_type=args.instance_type,
            security_groups=["default"],
            user_data=user_data,
            dry_run=args.dry_run)

        for instance in reservation.instances:
            instance.add_tag('Name', self.get_tag(args))

            while instance.state != 'running':
                print '...instance is %s' % instance.state
                time.sleep(10)
                instance.update()


    def delete(self, args):
        knife = Knife()
        creds = knife.get_credentials(args.environment)

        conn = boto.ec2.connect_to_region(self.region,
                                          aws_access_key_id=creds["access"],
                                          aws_secret_access_key=creds["secret"])

        instance = self.find_scheduler(conn, self.get_tag(args))
        if instance is None:
            raise Exception("Scheduler instance with name %s could not found" % self.get_tag(args))
        conn.terminate_instances(instance_ids=[instance.id])

    def suspend(self, args):
        knife_search = Knife()
        mc = knife_search.search(args.environment, self.tier)
        # TODO Use cname
        hostname = mc[0]
        url = "http://%s:9090/admin/pauseAll" % hostname
        r = requests.put(url)
        self.print_response(r)

    def resume(self, args):
        knife_search = Knife()
        mc = knife_search.search(args.environment, self.tier)
        # TODO Use cname
        hostname = mc[0]
        url = "http://%s:9090/admin/resumeAll" % hostname
        r = requests.put(url)
        self.print_response(r)

    def check(self, args):
        knife_search = Knife()
        mc = knife_search.search(args.environment, self.tier)
        # TODO Use cname
        hostname = mc[0]
        url = "http://%s:9090/admin/checkAll" % hostname
        print url
        r = requests.get(url)
        self.print_response(r)

    def print_response(self, res):
        print json.dumps(json.loads(res.text), indent=4)
