import logging
import string
import os
import inspect
import re
from time import sleep

import boto.rds
import boto.ec2.elb
import boto.ec2.autoscale
import boto.ec2
from boto.ec2.autoscale import LaunchConfiguration
from boto.ec2.autoscale import AutoScalingGroup
import boto.ec2.cloudwatch
from boto.ec2.cloudwatch import MetricAlarm
from qds_ops.entities.tier import Tier
from qds_ops.entities.web import Web
from qds_ops.tparty.knife import Knife
from qds_ops.entities.rds import RDS


class DefaultDJ(Web):
    tier = "defaultDJ"
    worker = "pju_worker"
    ami_id = "ami-1327167a"
    region = "us-east-1"
    instance_type = "m1.xlarge"
    run_list = '{"run_list":["recipe[webslave::deploy_dj]"],"qubole_tier":"%s", "worker_name":"%s"}'
    availibility_zone = "us-east-1b"

    def __init__(self, subparser):
        super(DefaultDJ, self).__init__(subparser)

        # sanitize-autoscaling-group
        san = self.tier_parser.add_parser("sanitize-as-group", help="Create and associate an upscaling policy with autoscaling group. ")
        san.add_argument("-r", "--region", help="Region where the tier will be located",
                            default=self.region, dest="region")
        san.set_defaults(func=self.sanitize)

        # policy-create
        pol = self.tier_parser.add_parser("as-policy", help="Create and associate an upscaling policy with autoscaling group. ")
        pol.add_argument("-r", "--region", help="Region where the tier will be located",
                            default=self.region, dest="region")
        pol.set_defaults(func=self.policy)

        # Create
        create = self.tier_parser.add_parser("create", help="Create Cron Tier. "
                                                            "Command will return after instance is in running state")
        create.add_argument("-a", "--ami-id", help="AMI ID to use for the CRON tier",
                            default=self.ami_id, dest="ami_id")
        create.add_argument("-z", "--zone", help="AWS Availibility Zone",
                            default=self.availibility_zone, dest="zone")
        create.add_argument("-r", "--region", help="Region where the tier will be located",
                            default=self.region, dest="region")
        create.add_argument("-i", "--instance-type", help="Instance Type of the machine",
                            choices=["t1.micro", "m1.large", "m1.xlarge"], dest="instance_type",
                            default=self.instance_type)
        create.add_argument("-t", "--tunnel", dest="tunnel",
                            action="store_true", help="Use a tunnel through a web node")
        create.set_defaults(func=self.create)

    def as_name(self, env):
        rows = RDS.run_query(env, False, "select value from rstore.settings where var='aws.as_group_name'")
        default_as_group = rows["value"]
        return "%s-%s" % (default_as_group, self.tier)

    def policy(self, args):
        knife = Knife()
        creds = knife.get_credentials(args.environment)

        logging.info("Connecting to AWS using boto")
        autoscale_conn = boto.ec2.autoscale.connect_to_region(
            args.region,
            aws_access_key_id=creds["access"],
            aws_secret_access_key=creds["secret"])
        as_group_name = self.as_name(args.environment)
        policies = autoscale_conn.get_all_policies(as_group=as_group_name)
        if len(policies > 0):
            raise Exception("Policy already exist for as group: " + str(as_group_name))
        policy_name = "%s-%s-upscale-policy" % (args.environment, self.tier)  
        policy = boto.ec2.autoscale.policy.ScalingPolicy(autoscale_conn, name=policy_name, adjustment_type="ChangeInCapacity", as_name=as_group_name,scaling_adjustment=1,cooldown=1200)
        autoscale_conn.create_scaling_policy(policy)
        policies = autoscale_conn.get_all_policies(as_group=as_group_name)
        policy = policies[0]
        cw_conn = boto.ec2.cloudwatch.connect_to_region(
            args.region,
            aws_access_key_id=creds["access"],
            aws_secret_access_key=creds["secret"])
        aname = "%s-%s-wait-time-policy" % (args.environment, self.tier) 
        anspace = "Qubole/%s/%s" % (args.environment, self.camelize(self.tier) + "Tier")
        scale_up_alarm = MetricAlarm(
                name=aname, namespace=anspace,
                metric='DJWaitTime', statistic='Average',
                comparison='>', threshold='150',
                period='300', evaluation_periods=1,
                alarm_actions=[policy.policy_arn],
                dimensions=None)
        cw_conn.create_alarm(scale_up_alarm)
    
    def camelize(self,a):
        return re.sub(r"(?:^|_)(.)", lambda x: x.group(0)[-1].upper(), string)

    def create(self, args):
        knife = Knife()
        creds = knife.get_credentials(args.environment)
        rows = RDS.run_query(args.environment, args.tunnel, "select value from rstore.settings where var='hadoop.keyname'")
        key_name = rows["value"]
        as_group_name = self.as_name(args.environment)

        as_lc_name = "%s_%s_launch_group" % (self.tier, args.environment)
        logging.info("Connecting to AWS using boto")
        autoscale_conn = boto.ec2.autoscale.connect_to_region(
            args.region,
            aws_access_key_id=creds["access"],
            aws_secret_access_key=creds["secret"])

        for lc in autoscale_conn.get_all_launch_configurations():
            if re.match(str(as_lc_name), lc.name):
                raise Exception("AS-Launch-Config '%s' already exists!" % (lc.name))

        for group in autoscale_conn.get_all_groups():
            if re.match(as_group_name, group.name):
                raise Exception("AS-Group '%s' already exists!" % (group.name))

        logging.info("Creating new AutoScaling Launch Config named '%s' ..." % (as_lc_name))

        #open the UDF
        base_path = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
        filein = open(base_path + "/../scripts/udf.sh")
        src = string.Template(filein.read())
        resolved_runlist = self.run_list % (self.tier, self.worker)
        d = {'runlist': resolved_runlist, 'environment': args.environment}
        user_data = src.safe_substitute(d)

        new_launch_config = LaunchConfiguration(
            name=as_lc_name,
            image_id=args.ami_id,
            key_name=key_name,
            security_groups=['default'],
            user_data=user_data,
            instance_type=self.instance_type)

        autoscale_conn.create_launch_configuration(new_launch_config)

        logging.info("Created new Autoscaling launch config named '%s' successfully\n\n" % (as_lc_name))

        logging.info("Creating new AS Group named '%s' ..." % (as_group_name))

        as_group = AutoScalingGroup(
            name=as_group_name,
            availability_zones=[args.zone],
            launch_config=new_launch_config,
            min_size=1,
            max_size=1,
            desired_capacity=1,
            connection=autoscale_conn, default_cooldown=1200,
            health_check_period=600 )

        autoscale_conn.create_auto_scaling_group(as_group)

        sleep(10)  # wait for auto scaling activity to get registered

        logging.info("AS Group '%s' successfully created. Current activities:\n '%s'\n\n" \
                     % (as_group.name, as_group.get_activities()))
        logging("Creating Autoscaling policy to upscale by 1 node if wait time > 150 secs")
        self.policy(args)
        as_obj.suspend_processes(["ReplaceUnhealthy"])

    def sanitize(self, args):
        logging.info("Getting credentials from Chef server and RDS")
        knife = Knife()
        creds = knife.get_credentials(args.environment)
        as_group_name = self.as_name(args.environment)

        logging.info("Connecting to AWS using boto")
        autoscale_conn = boto.ec2.autoscale.connect_to_region(
            args.region,
            aws_access_key_id=creds["access"],
            aws_secret_access_key=creds["secret"])

        as_objs = autoscale_conn.get_all_groups([as_group_name])
        if len(as_objs) < 1:
            raise Exception("As Group: %s does not exist. First create this group "
                            "and then call sanitize" % as_group_name)
        as_obj = as_objs[0]    
        as_obj.cooldown = 1200
        as_obj.health_check_period = 600
        as_obj.update()
        as_obj.suspend_processes(["ReplaceUnhealthy"])

