import json
import subprocess
import signal

from qds_ops.tparty.knife import Knife
from qds_ops.utils.deploy_utils import DeployUtils


class Tier(object):
    def __init__(self, global_subparser):
        parser = global_subparser.add_parser(self.tier,
                                             help="Commands to monitor and control %s tier" % self.tier)
        self.tier_parser = parser.add_subparsers()

        # List
        self.list_parser = self.tier_parser.add_parser("list",
                                                       help="List all nodes in %s tier" % self.tier)
        self.list_parser.set_defaults(func=self.list)

        #Package
        self.package_parser = self.tier_parser.add_parser("package",
                                                          help="List all packages in %s tier" % self.tier)
        self.package_parser.set_defaults(func=self.packages)

        #SSH
        self.ssh_parser = self.tier_parser.add_parser("ssh",
                                                      help="SSH and run any bash command on machines in %s tier" % self.tier)
        self.ssh_parser.add_argument("command", help="Bash command to run")
        self.ssh_parser.add_argument("-a", "--any", dest="any",
                                     action="store_true", help="Login into any node in %s" % (self.tier))
        self.ssh_parser.set_defaults(func=self.ssh)

        #Screen
        self.screen_parser = self.tier_parser.add_parser("screen",
                                                         help="Use GNU screen and open terminals on ALL machines in %s tier" % self.tier)
        self.screen_parser.set_defaults(func=self.screen)

        #Login
        self.login_parser = self.tier_parser.add_parser("login",
                                                        help="Login into one of the machines in a tier %s tier" % self.tier)
        self.login_parser.add_argument("-i", "--host", dest="host",
                                       help="Specify a public IP or hostname")
        self.login_parser.set_defaults(func=self.login)

        #Deploy
        module_choices = self.modules
        module_choices.extend(["all"])
        self.deploy_parser = self.tier_parser.add_parser("deploy",
                                                         help="Run deploy in %s tier" % self.tier)
        self.deploy_parser.add_argument("-m", "--module", dest="module", choices=module_choices,
                                        default="all", help="Deploy module to %s" % (self.tier))
        self.deploy_parser.set_defaults(func=self.deploy)

    def deploy(self, args):
        knife = Knife()
        chef_command = DeployUtils.get_chef_command_for_module(args.module, args.environment)

        knife.reap_instances(args)
        # Entering critical region. Interruption in this region causes a lot of grief
        #and manual merging of chef version files is required
        signal.signal(signal.SIGINT, signal.SIG_IGN)
        signal.signal(signal.SIGHUP, signal.SIG_IGN)
        signal.signal(signal.SIGTERM, signal.SIG_IGN)
        signal.signal(signal.SIGQUIT, signal.SIG_IGN)

        temp = knife.get_pemfile(args.environment)
        retcode = DeployUtils.run_chef_command(chef_command, args.environment, self.tier, temp, "")

        signal.signal(signal.SIGINT, signal.SIG_DFL)
        signal.signal(signal.SIGHUP, signal.SIG_DFL)
        signal.signal(signal.SIGTERM, signal.SIG_DFL)
        signal.signal(signal.SIGQUIT, signal.SIG_DFL)

        if retcode != 0:
            raise Exception("Deploy to %s tier failed. Return code: %d " % (self.tier, retcode))
        return retcode

    def list(self, args):
        knife_search = Knife()
        list = knife_search.search(args.environment, self.tier)
        print json.dumps(list, indent=4, sort_keys=True)

    def packages(self, args):
        knife_search = Knife()
        list = knife_search.packages(args.environment, self.tier)
        print json.dumps(list, indent=4, sort_keys=True)

    def ssh(self, args):
        knife = Knife()
        output = ""
        if args.any:
            output = knife.ssh_one(args.command, args.environment, self.tier)
        else:
            output = knife.ssh(args.command, args.environment, self.tier)
        print output

    def screen(self, args):
        knife = Knife()
        output = knife.ssh("screen", args.environment, self.tier)
        print output

    def login(self, args):
        knife = Knife()
        pemfile = knife.get_pemfile(args.environment)
        host = args.host
        if host is None:
            list = knife.search(args.environment, self.tier)
            host = list[0]

        subprocess.call(["ssh", "-i", pemfile, "ec2-user@%s" % host])