import logging
import string
import os
import inspect
import re
from time import sleep

import boto.rds
import boto.ec2.elb
import boto.ec2.autoscale
import boto.ec2
from boto.ec2.autoscale import LaunchConfiguration
from boto.ec2.autoscale import AutoScalingGroup
import boto.ec2.cloudwatch
from boto.ec2.cloudwatch import MetricAlarm
from qds_ops.entities.tier import Tier
from qds_ops.entities.web import Web
from qds_ops.tparty.knife import Knife
from qds_ops.entities.rds import RDS


class DefaultDJ(Web):
    tier = "defaultDJ"
    worker = "pju_worker"
    ami_id = "ami-1327167a"
    instance_type = "r3.xlarge"
    run_list = '{"run_list":["recipe[webslave::deploy_dj]"],"qubole_tier":"%s", "worker_name":"%s"}'
    availibility_zone = "us-east-1b"

    def __init__(self, subparser):
        super(DefaultDJ, self).__init__(subparser)

        # Create
        create = self.tier_parser.add_parser("create", help="Create Cron Tier. "
                                                            "Command will return after instance is in running state")
        create.add_argument("-a", "--ami-id", help="AMI ID to use for the CRON tier",
                            default=self.ami_id, dest="ami_id")
        create.add_argument("-z", "--zone", help="AWS Availibility Zone",
                            default=self.availibility_zone, dest="zone")
        create.add_argument("-r", "--region", help="Region where the tier will be located",
                            default=self.region, dest="region")
        create.add_argument("-i", "--instance-type", help="Instance Type of the machine",
                            choices=["m3.large", "m3.xlarge", "r3.large", "r3.xlarge", "r3.2xlarge"], dest="instance_type",
                            default=self.instance_type)
        create.add_argument("-t", "--tunnel", dest="tunnel",
                            action="store_true", help="Use a tunnel through a web node")
        create.set_defaults(func=self.create)
    
    def get_elb(self, env):
        return None

    def as_name(self, env):
        rows = RDS.run_query(env, False, "select value from rstore.settings where var='aws.as_group_name'")
        default_as_group = rows["value"]
        return "%s-%s" % (default_as_group, self.tier)

    def create(self, args):
        knife = Knife()
        creds = knife.get_credentials(args.environment)
        rows = RDS.run_query(args.environment, args.tunnel, "select value from rstore.settings where var='hadoop.keyname'")
        key_name = rows["value"]
        as_group_name = self.as_name(args.environment)

        as_lc_name = "%s_%s_launch_group" % (self.tier, args.environment)
        logging.info("Connecting to AWS using boto")
        autoscale_conn = boto.ec2.autoscale.connect_to_region(
            args.region,
            aws_access_key_id=creds["access"],
            aws_secret_access_key=creds["secret"])

        for lc in autoscale_conn.get_all_launch_configurations():
            if re.match(str(as_lc_name), lc.name):
                raise Exception("AS-Launch-Config '%s' already exists!" % (lc.name))

        for group in autoscale_conn.get_all_groups():
            if re.match(as_group_name, group.name):
                raise Exception("AS-Group '%s' already exists!" % (group.name))

        logging.info("Creating new AutoScaling Launch Config named '%s' ..." % (as_lc_name))

        #open the UDF
        base_path = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
        filein = open(base_path + "/../scripts/udf.sh")
        src = string.Template(filein.read())
        resolved_runlist = self.run_list % (self.tier, self.worker)
        d = {'runlist': resolved_runlist, 'environment': args.environment}
        user_data = src.safe_substitute(d)

        new_launch_config = LaunchConfiguration(
            name=as_lc_name,
            image_id=args.ami_id,
            key_name=key_name,
            security_groups=['default'],
            user_data=user_data,
            instance_type=self.instance_type)

        autoscale_conn.create_launch_configuration(new_launch_config)

        logging.info("Created new Autoscaling launch config named '%s' successfully\n\n" % (as_lc_name))

        logging.info("Creating new AS Group named '%s' ..." % (as_group_name))

        as_group = AutoScalingGroup(
            name=as_group_name,
            availability_zones=[args.zone],
            launch_config=new_launch_config,
            min_size=1,
            max_size=1,
            desired_capacity=1,
            connection=autoscale_conn, default_cooldown=1200,
            health_check_period=600 )

        autoscale_conn.create_auto_scaling_group(as_group)

        sleep(10)  # wait for auto scaling activity to get registered

        logging.info("AS Group '%s' successfully created. Current activities:\n '%s'\n\n" \
                     % (as_group.name, as_group.get_activities()))
        logging.info("Creating Autoscaling policy to upscale by 1 node if wait time > 150 secs")
        self.policy(args)
        as_objs = autoscale_conn.get_all_groups([as_group_name])
        as_obj = as_objs[0]    
        as_obj.suspend_processes(["ReplaceUnhealthy"])


