import sys, logging, re, time, os.path, subprocess, json, inspect
from tier                  import Tier
from qds_ops.entities.rds  import RDS
from qds_ops.tparty.knife  import Knife

class Oozie(Tier):

    def __init__(self):
        tier = "oozie-*"
        modules = ["hive_scripts", "hadoop", "hustler", "hive", "sqoop", "pig", "tapp2", "cloudman"]
        super(Oozie, self).__init__(tier=tier, modules=modules)
        self.FORMATTED_TIME_REGEX = "\\d\\d\\d\\d-\\d\\d-\\d\\d-\\d\\d-\\d\\d-\\d\\d"
        self.EASY_HUSTLER_PATH = "/usr/lib/hustler/bin/easy_hustler"
        self.is_autoscalable = False

    def configure_parsers(self):
        self.deploy_parser.add_argument("--ami", dest="ami", default="ami-c20ad4aa", help="nodes are launched from this ami")
        self.deploy_parser.add_argument("-r", "--no-rolling", dest="rolling", default=True, action="store_false", help="don't do rolling deploy")
        self.deploy_parser.set_defaults(func=self.oozie_deploy)

    def oozie_deploy(self, args):
        if args.rolling == False:
            super(Oozie, self).deploy(args)
            return

        logging.info("Starting rolling deploy of oozie")
        logging.info("Fetching settings from RDS")
        settings              = RDS.get_settings(args.environment, False)
        env_creds             = Knife().get_credentials(args.environment)
        ts                    = self.get_formatted_time()       
 
        hadoop_vars = [
            "mapred.fairscheduler.preemption=false",
            "mapred.membasedloadmanager.reservered.physicalmemory.mb=128",
            "mapred.job.tracker.identifier=oozie.%s" % ts,
            "mapred.hustler.nodes.max=%s" % settings["oozie.max_slave_count"],
            "mapred.tasktracker.cpus.to.maptasks=1:3\,2:6\,4:12"
        ]
        overrides = []
        overrides.append("cluster:smallcluster:s3log_access_key_id:%s" % env_creds["access"])
        overrides.append("cluster:smallcluster:s3log_secret_access_key:%s" % env_creds["secret"])
        overrides.append("cluster:smallcluster:s3log_location:%s" % os.path.join(settings["tapp.s3_bucket"], args.environment, "oozie-hadoop", ts))
        overrides.append("cluster:smallcluster:PVM_IMAGE_ID:%s" % args.ami)
        overrides.append("cluster:smallcluster:HVM_IMAGE_ID:%s" % args.ami)
        overrides.append("cluster:smallcluster:master_instance_type:%s" % settings["oozie.master_type"])
        overrides.append("cluster:smallcluster:NODE_INSTANCE_TYPE:%s" % settings["oozie.slave_type"])
        overrides.append("cluster:smallcluster:AVAILABILITY_ZONE:%s" % settings["oozie.aws_az"])
        overrides.append("cluster:smallcluster:static_security_groups:default")
        overrides.append("cluster:smallcluster:hadoop_vars:%s" % "!".join(hadoop_vars))
        overrides.append("cluster:smallcluster:cluster_size:%s" % settings["oozie.initial_nodes"])
        overrides.append("cluster:smallcluster:max_cluster_size:%s" % (int(settings["oozie.max_slave_count"]) + 1))
        overrides.append("cluster:smallcluster:hadoop_setup_timeout_seconds:1200")

        current_cluster_name = settings["oozie.cluster_name"]
        new_cluster_name     = self.new_cluster_name(current_cluster_name, ts)
        
        logging.info("starting oozie hadoop cluster with the following overrides:")
        for o in overrides:
            logging.info(o)
        exit_code = subprocess.call(
            [
                self.EASY_HUSTLER_PATH,
                "-o", ";".join(overrides),
                "start",
                "--insecure-mode",
                new_cluster_name
            ]
        )
        if exit_code == 0:
            logging.info("cluster successfully started. name = %s" % new_cluster_name) 
        else:
            logging.error("failed to start cluster. exiting")
            sys.exit(1)

        master_dns = self.get_master_dns(new_cluster_name)
        logging.info("master dns = %s" % master_dns)
  
        logging.info("updating db with info about new cluster") 
        self.update_settings(args.environment, new_cluster_name, master_dns)

        logging.info("deleting oozie.cluster_name key from memcache")
        dir_of_this_file = os.path.dirname(os.path.realpath(os.path.abspath(inspect.getfile(inspect.currentframe()))))
        memcache_script_path = os.path.join(os.path.dirname(dir_of_this_file), "scripts", "mc_del_key.rb")
        subprocess.call([memcache_script_path, args.environment, env_creds["memcache_server"]])

        logging.info("Rolling deploy of oozie cluster is complete. Please remember to shutdown the old cluster when it is free.")

    def get_formatted_time(self):
        s = time.strftime("%Y-%m-%d-%H-%M-%S", time.gmtime(int(time.time())))
        assert re.match(self.FORMATTED_TIME_REGEX, s) is not None
        return s

    def new_cluster_name(self, old_cluster_name, ts):
        match_obj = re.match("^.*(%s)$" % self.FORMATTED_TIME_REGEX, old_cluster_name)
        if match_obj is not None:
            name_without_timestamp = old_cluster_name[: -1 * len(match_obj.group(1))]
        else:
            name_without_timestamp = old_cluster_name
        return name_without_timestamp + ts

    def get_master_dns(self, cluster_name):
        cmd_out, cmd_err = subprocess.Popen(
            "%s listclusters -j %s" % (self.EASY_HUSTLER_PATH, cluster_name), 
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            shell=True
        ).communicate()
        json_obj = json.loads(cmd_out)
        cluster_info = json_obj[cluster_name]
        nodes = cluster_info["nodes"]
        for id, node_info in nodes.iteritems():
            if node_info["alias"] == "master":
                return node_info["dns_name"]
        
    def update_settings(self, env, new_cluster_name, master_dns):
        RDS.run_query(env, False, "update settings set value = \"%s\" where var = \"%s\"" % (new_cluster_name, "oozie.cluster_name"))
        RDS.run_query(env, False, "update settings set value = \"%s\" where var = \"%s\"" % (master_dns, "oozie.dns"))
        RDS.run_query(env, False, "update settings set value = \"%s\" where var = \"%s\"" % ("%s:9001" % master_dns, "oozie.jobtracker"))
        RDS.run_query(env, False, "update settings set value = \"%s\" where var = \"%s\"" % ("hdfs://%s:9000" % master_dns, "oozie.hdfs"))
