from tier import Tier
import boto
import time
import os
import inspect
import string
from qds_ops.tparty.knife import Knife
from qds_ops.entities.rds import RDS
import logging


class Log(Tier):

    def __init__(self):
        tier = "log"
        modules = ["hustler"]
        ami = "ami-c6bd64ae"
        region = "us-east-1"
        instance_type = "m1.large"
        run_list = '{"run_list":["recipe[webslave::logserver]"],"qubole_tier":"log"}'
        security_groups = ["log-server"]
        port = 7777

        super(Log, self).__init__(tier=tier, modules=modules, ami=ami,
                                  region=region, instance_type=instance_type, run_list=run_list,
                                  security_groups=security_groups, port=port
                                  )
        self.knife = Knife()
        self.is_autoscalable = False

    def configure_parsers(self):
        #create 
        create = self.tier_parser.add_parser("create", help="launches a new instance of log-server")
        create.add_argument("-d", "--dry-run", help="Dry Run: Dont really create the tier",
                            default=False, dest="dry_run", action="store_true")
        create.add_argument("-D", "--disable-termination", help="disable termination",
                            default=False, dest="disable_termination", action="store_true")
        create.add_argument("-a", "--ami-id", help="AMI ID to use for the log tier",
                            default=self.ami, dest="ami")
        create.add_argument("-r", "--region", help="Region where the tier will be located",
                            default=self.region, dest="region")
        create.add_argument("-i", "--instance-type", help="Instance Type of the machine",
                            choices=["m3.medium", "m3.large"], dest="instance_type",
                            default=self.instance_type)
        create.add_argument("-t", "--tunnel", dest="tunnel",
                            action="store_true", help="Use a tunnel through a web node")
        create.set_defaults(func=self.create)

        #delete
        delete = self.tier_parser.add_parser("delete", help="terminates all running instances of log-server")
        delete.set_defaults(func=self.delete)

    def create(self, args):
        creds = self.knife.get_credentials(args.environment)
        rows = RDS.run_query(args.environment, args.tunnel,
                             "select value from rstore.settings where var='hadoop.keyname'")

        # Open UDF
        base_path = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
        filein = open(base_path + "/../scripts/udf.sh")
        src = string.Template(filein.read())
        d = {'runlist': self.run_list, 'environment': args.environment}
        user_data = src.safe_substitute(d)

        ec2_conn = boto.ec2.connect_to_region(
            self.region,
            aws_access_key_id=creds["access"],
            aws_secret_access_key=creds["secret"]
        )
        existing = self._find_instances(ec2_conn, "Name", self._tag(args.environment))
        for instance in existing:
            if instance.state != "terminated":
                logging.info("Found non-terminated instance = %s" % instance.id)
                logging.info("Refusing to start a new instance.")
                ec2_conn.close()
                return
        reservation = ec2_conn.run_instances(
            self.ami,
            key_name = rows["value"],
            instance_type = args.instance_type,
            security_groups = self.security_groups,
            user_data = user_data,
            disable_api_termination = args.disable_termination,
            dry_run = args.dry_run)
        
        started_instance = reservation.instances[0]
        started_instance.add_tag("Name", self._tag(args.environment))

        while started_instance.state != "running":
            logging.info("waiting for instance %s to get to running state" % started_instance.id)
            time.sleep(10)
            started_instance.update()
        logging.info("instance % s is in running state" % started_instance.id)
        logging.info(" %s is public DNS name" % started_instance.dns_name)
        ec2_conn.close()

    def delete(self, args):
        creds = self.knife.get_credentials(args.environment)
        ec2_conn = boto.ec2.connect_to_region(
            self.region,
            aws_access_key_id=creds["access"],
            aws_secret_access_key=creds["secret"]
        )
        instances = self._find_instances(ec2_conn, "Name", self._tag(args.environment))
        non_terminated_instance_ids = [i.id for i in instances if
                                       i.state != "shutting-down" and i.state != "terminated"]
        if len(non_terminated_instance_ids) > 0:
            ec2_conn.terminate_instances(instance_ids=non_terminated_instance_ids)
            logging.info("terminated %s" % str(non_terminated_instance_ids))
        else:
            logging.info("no instance needs to be terminated")
        ec2_conn.close()

    def _tag(self, env):
        return "log_server_" + env

    def _find_instances(self, conn, tag, value):
        matched_instances = []
        all_instances = conn.get_only_instances()
        for i in all_instances:
            if tag in i.tags and i.tags[tag] == value:
                matched_instances.append(i)
        return matched_instances
