import json
from subprocess import Popen, PIPE
import inspect
import os
import logging

from qds_ops.tparty.knife import Knife
from qds_ops.utils.deploy_guide import DeployGuide

class Chef():

    def __init__(self, subparser):
        parser = subparser.add_parser("chef",
                                      help="Commands to get an overview of the installation")
        chef_subparser = parser.add_subparsers()

        #reap instances
        desc_parser = chef_subparser.add_parser("reap-instances",
                                                help="Remove nodes from chef server that have terminated")
        desc_parser.set_defaults(func=self.reap_instances)
        
        #describe
        desc_parser = chef_subparser.add_parser("describe",
                                                help="Describe environment attributes like cookbook version")
        desc_parser.set_defaults(func=self.describe)

        #roles
        roles_parser = chef_subparser.add_parser("roles")
        roles_parser.set_defaults(func=self.roles)

        #upload cookbook
        upload_parser = chef_subparser.add_parser("upload-cookbook",
                                                  help="Upload cookbook for specified env")
        upload_parser.add_argument("-v", "--version", dest="version",
                                   help="The cookbook version of the form x.y.z, generated by create-package-chef.sh. You can also specify a branch name in which case the tip of branch is uploaded.")
        upload_parser.add_argument("-s", "--srcpath", dest="src_path",
                                   default="~/src/", help="Root src directory")
        upload_parser.set_defaults(func=self.upload_cookbook)

         #update databag
        update_databag = chef_subparser.add_parser("update-databag",
                                                  help="update databag for specified env")
        update_databag.add_argument("-j", "--json", dest="json_obj",
            help="Valid Json with single quotes: Eg: -j '{\"key1: \"value1\",\"key2\": \"value\"}'")
        update_databag.set_defaults(func=self.update_databag)

        #Tiers
        tiers_parser = chef_subparser.add_parser("tiers",
                                                help="List all tiers in the environment")
        tiers_parser.set_defaults(func=self.tiers)

        #deploy-guide
        deploy_guide_parser = chef_subparser.add_parser("deploy-guide",
                                                        help="prints a deploy-guide for a given environment")
        deploy_guide_parser.set_defaults(func=self.deploy_guide)
        mutex_group = deploy_guide_parser.add_mutually_exclusive_group()
        mutex_group.add_argument("-o", "--only", dest="only",
                                 help="Only deploy to given comma separated list of tiers")
        mutex_group.add_argument("-s", "--skip", dest="skip",
                                 help="Skip deploy to given comma separated list of tiers")

    def reap_instances(self, args):
        knife = Knife()
        knife.reap_instances(args)
    
    def roles(self, args):
        knife = Knife()
        list = knife.roles()
        print json.dumps(list, indent=4, sort_keys=True)

    def upload_cookbook(self, args):
        base_path = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
        upload_cmd = ["/bin/bash", base_path + "/../scripts/upload_cookbook.sh", "-e", args.environment, "-v",
                      args.version, "-s", args.src_path]
        print str((upload_cmd))
        p = Popen(" ".join(upload_cmd), shell=True)
        retCode = p.wait()
        logging.info("Return Code is: " + str(retCode))

    def describe(self, args):
        knife = Knife()
        info = knife.describe(args.environment)
        print json.dumps(info, indent=4, sort_keys=True)

    def tiers(self, args):
        knife = Knife()
        tiers = knife.tiers(args.environment)
        print json.dumps(list(tiers), indent=4, sort_keys=True)

    def deploy_guide(self, args):
        knife = Knife()
        tiers = knife.tiers(args.environment)
        guide = DeployGuide(args, tiers)
        print guide.generate()

    def update_databag(self, args):
        knife = Knife()
        jsonobj=json.loads(args.json_obj)
        knife.update_data_bag(args.environment, jsonobj)

