# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
# 
#   http://www.apache.org/licenses/LICENSE-2.0
# 
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

class DeployGuide:
    """
    Generates a deploy guide


    # qds-ops -e staging chef deploy-guide
    # qds-ops -e staging chef deploy-guide --only=web,scheduler
    # qds-ops -e staging chef deploy-guide --skip=dj,oozie
    """

    def __init__(self, args, available_tiers):
        self.environment = args.environment
        self.invalid_qubole_tiers = ["redis", "log"]
        self.available_tiers = available_tiers
        self.only = args.only
        self.skip = args.skip
        self.stdout = []
        self.stdout.append("#!/bin/sh")

    def generate(self):
        if self.available_tiers is None or len(self.available_tiers) == 0:
            raise ValueError("No qubole tiers in your environment")

        tiers = filter(lambda t: t not in self.invalid_qubole_tiers, self.available_tiers)
        self.stdout.append("# available tiers on %s : %s" % (self.environment, str(tiers)))

        # remove tiers excluded by --only
        if self.only:
            tiers = filter(lambda t: t in self.only, tiers)

        # remove tiers excluded by --skip
        if self.skip:
            tiers = filter(lambda t: t not in self.skip, tiers)

        for step in [self.predeploy, self.deploy, self.postdeploy]:
            self.stdout.append("\n### %s ###\n" % step.__name__)
            [
                step(tier) for tier in tiers
            ]
        return "\n".join(self.stdout)


    def predeploy(self, tier):
        """ Any predeploy steps to be done for the tier
        """
        if tier == "scheduler":
            self.stdout.append("qds-ops -e %s scheduler suspend" % self.environment)



    def deploy(self, tier):
        """ deployment step where actual deploy occurs
        """
        if tier == "web":
            self.stdout.append("qds-ops -e %s web deploy" % self.environment)

        if tier == "oozie-master":
            self.stdout.append("qds-ops -e %s oozie-* deploy" % self.environment)

        if tier == "defaultDJ":
            self.stdout.append("qds-ops -e %s defaultDJ deploy" % self.environment)

        if tier == "scheduler":
            self.stdout.append("qds-ops -e %s scheduler deploy" % self.environment)


    def postdeploy(self, tier):
        """ postdeploy cleanup or restart of services

        """


