import json
import re
import logging
import subprocess
import signal
import string
import os
import inspect
import datetime
from datetime import datetime

from qds_ops.tparty.knife import Knife
from qds_ops.utils.deploy_utils import DeployUtils
from qds_ops.entities.rds import RDS

from time import sleep

import boto.rds
import boto.ec2.elb
import boto.ec2.autoscale
import boto.ec2
from boto.ec2.autoscale import LaunchConfiguration
from boto.ec2.autoscale import AutoScalingGroup
import boto.ec2.cloudwatch
from boto.ec2.cloudwatch import MetricAlarm


class Tier(object):

    def set_region(self, region):
        self.region = region

    def set_run_list(self, run_list):
        self.run_list = run_list

    def set_modules(self, modules):
        self.modules = modules

    def set_az(self, az):
        self.az = az

    def set_instance_type(self, instance_type):
        self.instance_type = instance_type

    def set_ami(self, ami):
        self.ami = ami

    def set_worker(self, worker):
        self.worker = worker

    def set_tier(self, tier):
        self.tier = tier

    def set_port(self, port):
        self.port = port

    def set_security_groups(self, security_groups):
        self.security_groups = security_groups

    @property
    def is_autoscalable(self):
        """ Whether or not the tier is autoscalable
        """
        return self.autoscalable

    @is_autoscalable.setter
    def is_autoscalable(self, value):
        self.autoscalable = value

    def __init__(self, tier=None, ami=None, run_list=None,
                 modules=None, instance_type="m1.xlarge", worker=None,
                 az="us-east-1b", region="us-east-1",
                 security_groups=None, port=None):

        self.tier, self.worker, self.run_list = tier, worker, run_list
        self.ami, self.instance_type = ami, instance_type
        self.az, self.region = az, region
        self.modules = [] if modules is None else modules
        self.security_groups = security_groups if security_groups is not None else ['default']
        self.port = port

    def configure_common_parsers(self, global_subparser):
        """ Common parsers for all tiers are configured here
        """
        parser = global_subparser.add_parser(self.tier.lower(),
                                             help="Commands to monitor and control %s tier" % self.tier)
        self.tier_parser = parser.add_subparsers()

        # List
        self.list_parser = self.tier_parser.add_parser("list",
                                                       help="List all nodes in %s tier" % self.tier)
        self.list_parser.set_defaults(func=self.list)

        #Package
        self.package_parser = self.tier_parser.add_parser("package",
                                                          help="List all packages in %s tier" % self.tier)
        self.package_parser.set_defaults(func=self.packages)

        #SSH
        self.ssh_parser = self.tier_parser.add_parser("ssh",
                                                      help="SSH and run any bash command on machines in %s tier" % self.tier)
        self.ssh_parser.add_argument("command", help="Bash command to run")
        self.ssh_parser.add_argument("-a", "--any", dest="any",
                                     action="store_true", help="Login into any node in %s" % (self.tier))
        self.ssh_parser.set_defaults(func=self.ssh)

        #Screen
        self.screen_parser = self.tier_parser.add_parser("screen",
                                                         help="Use GNU screen and open terminals on ALL machines in %s tier" % self.tier)
        self.screen_parser.set_defaults(func=self.screen)

        #Login
        self.login_parser = self.tier_parser.add_parser("login",
                                                        help="Login into one of the machines in a tier %s tier" % self.tier)
        self.login_parser.add_argument("-i", "--host", dest="host",
                                       help="Specify a public IP or hostname")
        self.login_parser.set_defaults(func=self.login)

        #Deploy
        module_choices = self.modules
        module_choices.extend(["all"])
        self.deploy_parser = self.tier_parser.add_parser("deploy",
                                                         help="Run deploy in %s tier" % self.tier)
        self.deploy_parser.add_argument("-m", "--module", dest="module", choices=module_choices,
                                        default="all", help="Deploy module to %s" % (self.tier))
        self.deploy_parser.set_defaults(func=self.deploy)

        if self.is_autoscalable:
            # autoscaling group (asg) create-or-update and setup scale-up policy
            create_update = self.tier_parser.add_parser("create-update", help="Create/update autoscale Tier. "
                                                                "Command will return after instance is in running state")
            create_update.add_argument("-a", "--ami-id", help="AMI ID to use for the CRON tier",
                                default=self.ami, dest="ami")
            create_update.add_argument("-z", "--zone", help="AWS Availibility Zone",
                                default=self.az, dest="zone")
            create_update.add_argument("-c", "--cooldown", help="Default cooldown",
                                default='600', dest="cooldown")
            create_update.add_argument("-g", "--health_check_grace", help="Health check grace",
                                default='600', dest="health_check_grace")
            create_update.add_argument("-r", "--region", help="Region where the tier will be located",
                                default=self.region, dest="region")
            create_update.add_argument("-i", "--instance-type", help="Instance Type of the machine",
                                choices=["m3.large", "m3.xlarge", "r3.large", "r3.xlarge", "r3.2xlarge", "m1.large", "m1.xlarge"], dest="instance_type",
                                default=self.instance_type)
            create_update.add_argument("-t", "--tunnel", dest="tunnel",
                                action="store_true", help="Use a tunnel through a web node")
            
            create_update.add_argument("-d", "--alarm-threshold", help="As policy alarm threshold",
                                default='100', dest="threshold")
            create_update.add_argument("-l", "--alarm-period", help="As policy alarm period",
                                default='300', dest="period")
            create_update.add_argument("-p", "--policy_adjustment", help="As policy adjustment",
                                default='1', dest="adjustment")
            create_update.set_defaults(func=self.asg)

    def deploy(self, args):
        knife = Knife()
        chef_command = DeployUtils.get_chef_command_for_module(args.module, args.environment)

        knife.reap_instances(args)
        # Entering critical region. Interruption in this region causes a lot of grief
        #and manual merging of chef version files is required
        signal.signal(signal.SIGINT, signal.SIG_IGN)
        signal.signal(signal.SIGHUP, signal.SIG_IGN)
        signal.signal(signal.SIGTERM, signal.SIG_IGN)
        signal.signal(signal.SIGQUIT, signal.SIG_IGN)

        temp = knife.get_pemfile(args.environment)
        retcode = DeployUtils.run_chef_command(chef_command, args.environment, self.tier, temp, "")

        signal.signal(signal.SIGINT, signal.SIG_DFL)
        signal.signal(signal.SIGHUP, signal.SIG_DFL)
        signal.signal(signal.SIGTERM, signal.SIG_DFL)
        signal.signal(signal.SIGQUIT, signal.SIG_DFL)

        if retcode != 0:
            raise Exception("Deploy to %s tier failed. Return code: %d " % (self.tier, retcode))
        return retcode

    def list(self, args):
        knife_search = Knife()
        list = knife_search.search(args.environment, self.tier)
        print json.dumps(list, indent=4, sort_keys=True)

    def packages(self, args):
        knife_search = Knife()
        list = knife_search.packages(args.environment, self.tier)
        print json.dumps(list, indent=4, sort_keys=True)

    def ssh(self, args):
        knife = Knife()
        output = ""
        if args.any:
            output = knife.ssh_one(args.command, args.environment, self.tier)
        else:
            output = knife.ssh(args.command, args.environment, self.tier)
        print output

    def screen(self, args):
        knife = Knife()
        output = knife.ssh("screen", args.environment, self.tier)
        print output

    def login(self, args):
        knife = Knife()
        pemfile = knife.get_pemfile(args.environment)
        host = args.host
        if host is None:
            list = knife.search(args.environment, self.tier)
            host = list[0]

        subprocess.call(["ssh", "-i", pemfile, "ec2-user@%s" % host])

    def as_name(self, env):
        rows = RDS.run_query(env, False,
                             "select value from rstore.settings where var='aws.as_group_name'")
        return rows["value"]

    def camelize(self, a):
        return re.sub(r"(?:^|_)(.)", lambda x: x.group(0)[-1].upper(), a)

    def asg(self, args):
        """ Configure autoscaling groups, policies and metrics for the tier

        If the launch config exists with the same name, we do not recreate it. But the
        autoscaling group will be sanitized to conform to the same settings (cooldown, waittime, policy)
        as defined here. If the autoscaling group exists, we sync, if not we create and apply
        """
        if not self.is_autoscalable:
            raise Exception("%s tier is not autoscalable in qubole" % self.tier)
        knife = Knife()
        creds = knife.get_credentials(args.environment)
        rows = RDS.run_query(args.environment, False,
                             "select value from rstore.settings where var='hadoop.keyname'")
        key_name = rows["value"]
        as_group_name = self.as_name(args.environment)
        logging.info("%s autoscale group name: %s" % (self.tier, as_group_name))
        as_lc_name = "%s_%s_launch_group_%s" % (self.tier, args.environment, datetime.now().strftime("%Y-%m-%d-%H-%M"))
        logging.info("%s autoscale launch configuration: %s" % (self.tier, as_lc_name))

        # set the udf. udf is encoded by boto
        base_path = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
        filein = open(base_path + "/../scripts/udf.sh")
        src = string.Template(filein.read())
        cpre = DeployUtils.get_cbook_prefix(args.environment)
        resolved_runlist = self.run_list % (cpre, self.tier, self.worker)
        d = {'runlist': resolved_runlist, 'environment': args.environment}
        user_data = src.safe_substitute(d)

        autoscale_conn = boto.ec2.autoscale.connect_to_region(
            args.region,
            aws_access_key_id=creds["access"],
            aws_secret_access_key=creds["secret"])

        new_launch_config = LaunchConfiguration(
                name=as_lc_name,
                image_id=args.ami,
                key_name=key_name,
                security_groups=self.security_groups,
                user_data=user_data,
                instance_type=args.instance_type)

        for lc in autoscale_conn.get_all_launch_configurations():
            if re.match(str(as_lc_name), lc.name):
                logging.warn("%s autoscale launch configuration '%s' already exists!" % (self.tier, lc.name))
                break
        else:
            autoscale_conn.create_launch_configuration(new_launch_config)
            logging.info("%s autoscale launch configuration creation success: %s" % (self.tier, as_lc_name))

        as_group = AutoScalingGroup(
                name=as_group_name,
                availability_zones=[args.zone],
                launch_config=new_launch_config,
                min_size=1,
                max_size=4,
                desired_capacity=1,
                connection=autoscale_conn,
                default_cooldown=args.cooldown,
                health_check_period=args.health_check_grace)
        for group in autoscale_conn.get_all_groups():
            if re.match(as_group_name, group.name):
                logging.warn("%s autoscale group '%s' already exists!" % (self.tier, group.name))
                # synchronize changes in asg via update
                as_group.update()
                break
        else:
            autoscale_conn.create_auto_scaling_group(as_group)
            logging.info("%s autoscaling group %s created, activities are %s" %
                     (self.tier, as_group.name, as_group.get_activities()))

        sleep(10)  # wait for auto scaling activity to get registered
        logging.info("%s autoscale policy to upscale by 1 node if wait time > 150 secs" % self.tier)
        self.policy(args)
        as_objs = autoscale_conn.get_all_groups([as_group_name])
        as_obj = as_objs[0]
        as_obj.suspend_processes(["ReplaceUnhealthy"])

    def policy(self, args):
        knife = Knife()
        creds = knife.get_credentials(args.environment)

        logging.info("applying autoscaling policy in %s" % args.environment)
        autoscale_conn = boto.ec2.autoscale.connect_to_region(
            args.region,
            aws_access_key_id=creds["access"],
            aws_secret_access_key=creds["secret"])
        as_group_name = self.as_name(args.environment)
        logging.debug("autoscaling group for %s should be %s" % (args.environment, as_group_name))

        policy_name = "%s-%s-upscale-policy" % (args.environment, self.tier)
        policies = autoscale_conn.get_all_policies(as_group=as_group_name)
        if len(policies) > 0:
            logging.warn("autoscaling policy %s:%s already exists" % (as_group_name, policies[0]))
            return
        policy = boto.ec2.autoscale.policy.ScalingPolicy(
            autoscale_conn,
            name=policy_name,
            adjustment_type="ChangeInCapacity",
            as_name=as_group_name,
            scaling_adjustment=args.adjustment,
            cooldown=args.cooldown)
        autoscale_conn.create_scaling_policy(policy)
        policies = autoscale_conn.get_all_policies(as_group=as_group_name)
        assert len(policies) == 1, "Found multiple (%s) autoscaling policies!" % as_group_name
        logging.info("created new upscale policy %s" % policies[0])
        policy = policies[0]

        cw_conn = boto.ec2.cloudwatch.connect_to_region(
            args.region,
            aws_access_key_id=creds["access"],
            aws_secret_access_key=creds["secret"])
        aname = "%s-%s-wait-time-policy" % (args.environment, self.tier)
        t = self.camelize(self.tier)
        anspace = "Qubole/%s/%sTier" % (args.environment, t)
        scale_up_alarm = MetricAlarm(
                name=aname,
                namespace=anspace,
                metric='DJWaitTime',
                statistic='Average',
                comparison='>',
                threshold=args.threshold,
                period=args.period,
                evaluation_periods=1,
                alarm_actions=[policy.policy_arn],
                dimensions={})
        cw_conn.create_alarm(scale_up_alarm)
        logging.info("created cloudwatch alarm %s for autoscaling policy %s" % (aname, policy_name))
