import json
import os
import inspect
import logging

from qds_ops.entities.tier import Tier
from qds_ops.utils.fork import Fork
from qds_ops.utils.rolling_deploy import RollingDeploy
from qds_ops.utils.deploy_utils import DeployUtils
from qds_ops.tparty.knife import Knife



class Web(Tier):

    def __init__(self):
        tier = "web"
        modules = ["hive_scripts", "hadoop", "hadoop2", "hustler", "hive", "hive13", "hive2", "sqoop", "pig", "tapp2", "cloudman", "cluster_proxy"]
        run_list = '{"run_list":["recipe[%swebslave]"], "qubole_tier":"%s", "worker_name": "%s"}'
        worker = "qpal_worker"
        ami = "ami-0edbb766"
        region = "us-east-1"
        az = "us-east-1d"
        super(Web, self).__init__(tier=tier, run_list=run_list, ami=ami, modules=modules, worker=worker, region=region, az=az)
        self.is_autoscalable = True

    def configure_parsers(self):
        #CMD
        cli = self.tier_parser.add_parser("cli",
                                          help="Show all cli.py processes or commands running on ALL web nodes")
        cli.add_argument("-z", "--zombies", dest="zombies",
                         action="store_true", help="Login into a zombie node in %s" % (self.tier))
        cli.set_defaults(func=self.cli)

        #CMD Count
        cli_count = self.tier_parser.add_parser("cli-count",
                                                help="Count number of cli.py or commands running on ALL web nodes")
        cli_count.add_argument("-z", "--zombies", dest="zombies",
                               action="store_true", help="Login into a zombie node in %s" % (self.tier))
        cli_count.set_defaults(func=self.cli_count)

        #DJ
        dj = self.tier_parser.add_parser("dj", help="List all Delayed jobs running on web nodes")
        dj.add_argument("-z", "--zombies", dest="zombies",
                        action="store_true", help="Login into a zombie node in %s" % (self.tier))
        dj.set_defaults(func=self.dj)

        #DJ Count
        dj_count = self.tier_parser.add_parser("dj-count", help="Count all DJs on each web node")
        dj_count.add_argument("-z", "--zombies", dest="zombies",
                              action="store_true", help="Login into a zombie node in %s" % (self.tier))
        dj_count.set_defaults(func=self.dj_count)

        # List - add --new argument
        self.list_parser.add_argument("-z", "--zombies", dest="zombies",
                                      action="store_true", help="Login into a zombie node in %s" % (self.tier))
        self.list_parser.set_defaults(func=self.web_list)

        # ssh - add --new argument
        self.ssh_parser.add_argument("-z", "--zombies", dest="zombies",
                                     action="store_true", help="Login into a zombie node in %s" % (self.tier))
        self.ssh_parser.set_defaults(func=self.web_ssh)

        # screen - add --new argument
        self.screen_parser.add_argument("-z", "--zombies", dest="zombies",
                                        action="store_true", help="Login into a zombie node in %s" % (self.tier))
        self.screen_parser.set_defaults(func=self.web_screen)

        # Login - add --new argument
        self.login_parser.add_argument("-z", "--zombies", dest="zombies",
                                       action="store_true", help="Login into a zombie node in %s" % (self.tier))
        self.login_parser.set_defaults(func=self.web_login)

        #Zombie
        self.zombie_parser = self.tier_parser.add_parser("dezombie",
                                                        help="Convert Zombies to Live nodes on %s tier" % self.tier)
        self.zombie_parser.set_defaults(func=self.de_zombie)


        #add new args for deploy_parser
        self.deploy_parser.add_argument("-r", "--no-rolling", dest="rolling",
                                        default=True, action="store_false",
                                        help="Don't do rolling deploy to %s" % (self.tier))
        self.deploy_parser.add_argument("-t", "--stop-djs-at-start", dest="stopdjs",
                                        default=False, action="store_true",
                                        help="At start of rolling deploy, stop DJs on current nodes")
        self.deploy_parser.add_argument("-c", "--run-common-stuff-recipe", dest="commstuff",
                                        default=False, action="store_true", help="Run common stuff at start of deploy")
        self.deploy_parser.add_argument("-d", "--no-djrestart", dest="djrestart",
                                        default=True, action="store_false",
                                        help="Don't do DJ restart on %s" % (self.tier))
        self.deploy_parser.set_defaults(func=self.web_deploy)

        #Deploy Lock
        lock_parser = self.tier_parser.add_parser("deploy-lock",
                                                  help="Check, lock or unlock deploy lock")
        lock_parser.add_argument("-a", "--action", dest="action", choices=["check", "lock", "unlock"], default="check",
                                 help="Action to apply to the lock")
        lock_parser.add_argument("-t", "--tunnel", dest="tunnel",
                                 action="store_true", help="Setup a tunnel to RDS", default=False)
        lock_parser.set_defaults(func=self.deploy_lock)

        #update-cache
        update_cache_parser = self.tier_parser.add_parser("update-cache",
                                                          help="Update settings cache via http request")
        update_cache_parser.set_defaults(func=self.update_settings_cache)

    @staticmethod
    def get_filter(args):
        if args.zombies:
            return ["shutdown_gracefully:yes"]
        else:
            return [" NOT shutdown_gracefully:yes"]

    def cli(self, args):
        knife = Knife()
        output = knife.ssh("ps -ef | egrep \"cli.py\" | grep -v \"sh -c\" | grep -v grep ",
                           args.environment, self.tier, self.get_filter(args))
        print output

    def cli_count(self, args):
        knife = Knife()
        output = knife.ssh("ps -ef | egrep \"cli.py\" | grep -v \"sh -c\" | grep -v grep | wc -l",
                           args.environment, self.tier, self.get_filter(args))
        print output

    def dj(self, args):
        knife = Knife()
        output = knife.ssh("ps -ef | egrep \"delayed_job\" | grep -v \"sh -c\" | grep -v grep ",
                           args.environment, self.tier, self.get_filter(args))
        print output

    def dj_count(self, args):
        knife = Knife()
        output = knife.ssh("ps -ef | egrep \"delayed_job\" | grep -v \"sh -c\" | grep -v grep | wc -l",
                           args.environment, self.tier, self.get_filter(args))
        print output

    def web_list(self, args):
        knife_search = Knife()
        nodes = knife_search.search(args.environment, self.tier, self.get_filter(args))
        print json.dumps(nodes, indent=4, sort_keys=True)

    def web_ssh(self, args):
        knife = Knife()
        output = ""
        if args.any:
            output = knife.ssh_one(args.command, args.environment, self.tier, self.get_filter(args))
        else:
            output = knife.ssh(args.command, args.environment, self.tier, self.get_filter(args))
        print output

    def web_screen(self, args):
        output = Knife().ssh("screen", args.environment, self.tier, self.get_filter(args))
        print output

    def de_zombie(self, args):
        base_path = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
        cmd = ["knife", "exec", base_path + "/../scripts/set_node_attr.rb", args.environment, "shutdown_gracefully",
               "no", self.tier]

        Fork.check_output(cmd, stderr=None)

    def get_elb(self, env):
        creds = Knife().get_credentials(env)
        return creds["lb_name"]

    def web_deploy(self, args):
        if args.djrestart == False and args.module != "tapp2":
            raise Exception("-d argument can be specified only with -m tapp2")

        logging.info("Trying to take a lock for deploy on " + args.environment)
        knife = Knife()
        success = knife.deploy_lock(args.environment, False, self.tier, "lock")
        if not success:
            raise Exception("Failed to take lock for deploy. Exiting...")

        try:
            knife.reap_instances(args)
            temp = knife.get_pemfile(args.environment)
            creds = knife.get_credentials(args.environment)

            if (args.commstuff == True):
                DeployUtils.run_common_stuff(args.environment, self.tier, temp)

            djrestart = "yes"
            if not args.djrestart:
                djrestart = "no"

            elb_name = self.get_elb(args.environment)
            base_path = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
            out = Fork.check_output(
                ["knife", "exec", base_path + "/../scripts/set_node_attr.rb", args.environment, "djrestart", djrestart,
                 self.tier], stderr=None)
            print out

            if not args.djrestart:
                DeployUtils.do_master_slave_deploy(args.environment, self.tier, temp, args.module, creds["access"],
                                                   creds["secret"], elb_name)
            elif ((args.module == "all" or args.module == "tapp2" or args.module == "hive") and args.rolling == True):
                logging.info("Doing Rolling deploy")
                rd = RollingDeploy(creds["host"], creds["password"], creds["username"], creds["database"], elb_name,
                                   args.environment, temp, args.stopdjs, creds['access'], creds['secret'],
                                   self.as_name(args.environment), self.tier, self.worker).deploy()
            else:
                chef_command = DeployUtils.get_chef_command_for_module(args.module, args.environment)
                DeployUtils.run_chef_command(chef_command, args.environment, self.tier, temp,
                                             "AND NOT shutdown_gracefully:yes")
            logging.info("-------Done with deploy--------")
        finally:
            success = knife.deploy_lock(args.environment, False, self.tier, "unlock")

    def web_login(self, args):
        if args.host is None:
            knife_search = Knife()
            nodes = knife_search.search(args.environment, self.tier, self.get_filter(args))
            args.host = nodes[0]
        self.login(args)


    def deploy_lock(self, args):
        knife = Knife()
        knife.deploy_lock(args.environment, args.tunnel, self.tier, args.action)

    def update_settings_cache(self, args):
        knife = Knife()
        output = knife.ssh("curl --header 'Accepts: application/json' http://localhost/update_settings_cache", args.environment, self.tier)
        print output
