## Copyright (c) 2012, 2013 Aldebaran Robotics. All rights reserved.
## Use of this source code is governed by a BSD-style license that can be
## found in the COPYING file.

""" List the names and paths of every project, or those matching a pattern

"""

import re
import operator

from qisys import ui
import qisys.parsers
import qisrc.worktree


def configure_parser(parser):
    """ Configure parser for this action """
    qisys.parsers.worktree_parser(parser)
    qisys.parsers.project_parser(parser)
    parser.add_argument("--names", action="store_true", dest="names",
                        help="sort by names")
    parser.add_argument("--paths", action="store_false", dest="names",
                        help="sort by path")
    parser.add_argument("pattern", metavar="PATTERN", nargs="?",
                        help="pattern to be matched")
    parser.set_defaults(names=True)

def do(args):
    """ Main method """
    worktree = qisrc.parsers.get_git_worktree(args)
    projects = qisrc.parsers.get_git_projects(worktree, args, default_all=True)
    if not projects:
        qisrc.worktree.on_no_matching_projects(worktree, groups=args.groups)
        return
    ui.info(ui.green, "qisrc projects in:", ui.blue, worktree.root)
    max_name = max(len(x.name) for x in projects)
    max_src  = max(len(x.src)  for x in projects)
    regex = args.pattern
    if args.pattern:
        regex = re.compile(regex)
    if args.names:
        projects.sort(key=operator.attrgetter("name"))
    else:
        projects.sort(key=operator.attrgetter("src"))
    for project in projects:
        if args.names:
            items = (project.name.ljust(max_name + 2), project.path)
        else:
            items = (project.src.ljust(max_src + 2), project.name)
        if not regex or regex.search(items[0]) or regex.search(items[1]):
            ui.info(ui.green, " * ", ui.blue, items[0], ui.reset, items[1])
