import abc
import os

import qisys.sh

class DocProject(object):

    __metaclass__ = abc.ABCMeta
    doc_type = None

    def __init__(self, doc_worktree, project, name, depends=None, dest=None):
        self.doc_worktree = doc_worktree
        self.name = name
        self.src = project.src
        self.path = project.path
        if not depends:
            depends = list()
        if not dest:
            dest = self.name
        self.depends = list()
        self.dest = dest
        self.prebuild_script = None

    @abc.abstractmethod
    def configure(self, **kwargs):
        pass

    @abc.abstractmethod
    def build(self, **kwargs):
        pass

    @abc.abstractmethod
    def install(self, destdir):
        pass

    @property
    def build_dir(self):
        build_dir = os.path.join(self.path, "build-doc")
        qisys.sh.mkdir(build_dir)
        return build_dir

    @property
    def html_dir(self):
        html_dir = os.path.join(self.build_dir, "html")
        qisys.sh.mkdir(html_dir)
        return html_dir

    @property
    def index_html(self):
        return os.path.join(self.build_dir, "html", "index.html")

    @property
    def template_project(self):
        return self.doc_worktree.template_project

    @property
    def doxydeps(self):
        res = list()
        for dep_name in self.depends:
            doc_project = self.doc_worktree.get_doc_project(dep_name, raises=False)
            if doc_project and doc_project.doc_type == "doxygen":
                res.append(doc_project)
        return res

    def append_doxy_xml_path(self, pathes):
        for doxydep in self.doxydeps:
            pathes.append(os.path.join(doxydep.build_dir, 'xml'))
            doxydep.append_doxy_xml_path(pathes)

    def __repr__(self):
        return "<%s %s in %s>" % (self.doc_type.capitalize() + "Project",
                                  self.name, self.src)

    def __eq__(self, other):
        return self.doc_type == other.doc_type and \
                self.src == other.src and \
                self.name == other.name

    def __ne__(self, other):
        return not self.__eq__(other)
