'''
Copyright 2012-2014 Ministerie van Sociale Zaken en Werkgelegenheid

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
'''

import unittest
from qualitylib import metric, domain


class FakeSonar(object):
    ''' Provide for a fake Sonar object so that the unit test don't need 
        access to an actual Sonar instance. '''
    # pylint: disable=unused-argument
    def __init__(self, critical_violations=0, major_violations=0):
        self.__critical_violations = critical_violations
        self.__major_violations = major_violations

    def critical_violations(self, *args):
        ''' Return the number of critical violations. '''
        return self.__critical_violations

    def major_violations(self, *args):
        ''' Return the number of major violations. '''
        return self.__major_violations

    @staticmethod
    def dashboard_url(*args):  
        ''' Return a fake dashboard url. '''
        return 'http://sonar'


class FakeSubject(object):
    ''' Provide for a fake subject. '''

    def __repr__(self):
        return 'FakeSubject'

    @staticmethod
    def sonar_id():
        ''' Return the Sonar id of the subject. '''
        return ''

    @staticmethod
    def is_art():
        ''' Return whether the subject is an ART. '''
        return False


class ViolationsTestMixin(object):
    # pylint: disable=too-many-public-methods
    ''' Unit tests for the Violations metric classes. '''

    def setUp(self):  # pylint: disable=invalid-name,missing-docstring
        self.__nr_violations = 51
        self.__subject = FakeSubject()
        sonar = FakeSonar(critical_violations=self.__nr_violations, 
                          major_violations=self.__nr_violations)
        project = domain.Project(sonar=sonar)
        self._metric = self.metric_class(subject=self.__subject, 
                                         project=project)

    def test_numerical_value(self):
        ''' Test that the numerical value of the metric equals the the
            number of critical violation as provided by Sonar. '''
        self.assertEqual(self.__nr_violations, 
                         self._metric.numerical_value())

    def test_value(self):
        ''' Test that the value is equal to the number of critical 
            violations as reported by Sonar. '''
        self.assertEqual(self.__nr_violations, self._metric.value())

    def test_status(self):
        ''' Test that the status is red when there are too many violations. '''
        self.assertEqual('red', self._metric.status())

    def test_status_with_technical_debt(self):
        ''' Test that the status is grey when the subject has accepted
            technical debt. '''
        # pylint: disable=W0201
        self.__subject.technical_debt_target = lambda *args: \
            domain.TechnicalDebtTarget(51, 'Comment')
        self.assertEqual('grey', self._metric.status())

    def test_report(self):
        ''' Test that the report is correct. '''
        self.assertEqual('FakeSubject heeft %d %s violations.' % \
                         (self.__nr_violations, self.violation_type), 
                         self._metric.report())

    def test_is_perfect(self):
        ''' Test that the metric is perfect when both the number of major and 
            the number of critical violations is zero. '''
        sonar = FakeSonar(critical_violations=0, major_violations=0)
        project = domain.Project(sonar=sonar)
        violations = metric.CriticalViolations(subject=self.__subject, 
                                               project=project)
        self.assertEqual('perfect', violations.status())

    def test_url(self):
        ''' Test that the url is correct. '''
        self.assertEqual(dict(Sonar=FakeSonar().dashboard_url()), 
                         self._metric.url())

    def test_norm_template_default_values(self):
        ''' Test that the right values are returned to fill in the norm 
            template. '''
        self.failUnless(self.metric_class.norm_template % \
                        self.metric_class.norm_template_default_values())


class CriticalViolationsTest(ViolationsTestMixin, unittest.TestCase):
    # pylint: disable=too-many-public-methods
    ''' Unit tests for the CriticalViolations metric class. '''

    metric_class = metric.CriticalViolations
    violation_type = 'critical'


class MajorViolationsTest(ViolationsTestMixin, unittest.TestCase):
    # pylint: disable=too-many-public-methods
    ''' Unit tests for the MajorViolations metric class. '''

    metric_class = metric.MajorViolations
    violation_type = 'major'
