""" Module providing a single class (QualysConnectConfig) that parses a config
file and provides the information required to build QualysGuard sessions.
"""
import os
import stat
import sys
import getpass
import logging

from ConfigParser import *

import qualysapi.settings as qcs

__author__ = "Parag Baxi <parag.baxi@gmail.com> & Colin Bell <colin.bell@uwaterloo.ca>"
__copyright__ = "Copyright 2011-2013, Parag Baxi & University of Waterloo"
__license__ = "BSD-new"

# Setup module level logging.
logger = logging.getLogger(__name__)

class QualysConnectConfig:
    """ Class to create a ConfigParser and read user/password details
    from an ini file.
    """
    def __init__(self, filename=qcs.default_filename, remember_me=False, remember_me_always=False):

        self._cfgfile = None

        # Set home path for file.
        home_filename = os.path.join(os.getenv('HOME'),filename)
        # Check for file existence.
        if os.path.exists(filename):
            self._cfgfile = filename
        elif os.path.exists(home_filename):
            self._cfgfile = home_filename
        
        # create ConfigParser to combine defaults and input from config file.
        self._cfgparse = ConfigParser(qcs.defaults)

        if self._cfgfile:
            self._cfgfile = os.path.realpath(self._cfgfile)
            
            mode = stat.S_IMODE(os.stat(self._cfgfile)[stat.ST_MODE])
            
            # apply bitmask to current mode to check ONLY user access permissions.
            if (mode & ( stat.S_IRWXG | stat.S_IRWXO )) != 0:
                logging.warning('%s permissions allows more than user access.'%(filename,))

            self._cfgparse.read(self._cfgfile)

        # if 'info' doesn't exist, create the section.
        if not self._cfgparse.has_section('info'):
            self._cfgparse.add_section('info')

        # use default hostname (if one isn't provided)
        if not self._cfgparse.has_option('info','hostname'):
            if self._cfgparse.has_option('DEFAULT','hostname'):
                hostname = self._cfgparse.get('DEFAULT','hostname')
                self._cfgparse.set('info', 'hostname', hostname)
            else:
                raise Exception("No 'hostname' set. QualysConnect does not know who to connect to.")

        # Proxy support
        proxy_config = proxy_url = proxy_protocol = proxy_port = proxy_username = proxy_password = None
        # User requires proxy?
        if self._cfgparse.has_option('proxy','proxy_url'):
            proxy_url = self._cfgparse.get('proxy','proxy_url')
            # Remove protocol prefix from url if included.
            for prefix in ('http://', 'https://'):
                if proxy_url.startswith(prefix):
                    proxy_protocol = prefix
                    proxy_url = proxy_url[len(prefix):]
            # Default proxy protocol is http.
            if not proxy_protocol:
                proxy_protocol = 'https://'
            # Check for proxy port request.
            if ':' in proxy_url:
                # Proxy port already specified in url.
                # Set proxy port.
                proxy_port = proxy_url[proxy_url.index(':')+1:]
                # Remove proxy port from proxy url.
                proxy_url = proxy_url[:proxy_url.index(':')]
            if self._cfgparse.has_option('proxy','proxy_port'):
                # Proxy requires specific port.
                if proxy_port:
                    # Warn that a proxy port was already specified in the url.
                    proxy_port_url = proxy_port
                    proxy_port = self._cfgparse.get('proxy','proxy_port')
                    logger.warning('Proxy port from url overwritten by specified proxy_port from config:')
                    logger.warning('%s --> %s' % (proxy_port_url, proxy_port))
                else:
                    proxy_port = self._cfgparse.get('proxy','proxy_port')
            if not proxy_port:
                # No proxy port specified.
                if proxy_protocol == 'http://':
                    # Use default HTTP Proxy port.
                    proxy_port = '8080'
                else:
                    # Use default HTTPS Proxy port.
                    proxy_port = '443'

            # Check for proxy authentication request.
            if self._cfgparse.has_option('proxy','proxy_username'):
                # Proxy requires username & password.
                proxy_username = self._cfgparse.get('proxy','proxy_username')
                proxy_password = self._cfgparse.get('proxy','proxy_password')
                # Not sure if this use case below is valid.
                # # Support proxy with username and empty password.
                # try:
                #     proxy_password = self._cfgparse.get('proxy','proxy_password')
                # except NoOptionError, e:
                #     # Set empty password.
                #     proxy_password = ''
        # Sample proxy config:
        # 'http://user:pass@10.10.1.10:3128'
        if proxy_url:
            # Proxy requested.
            proxy_config = proxy_url
            if proxy_port:
                # Proxy port requested.
                proxy_config += ':' + proxy_port
            if proxy_username:
                # Proxy authentication requested.
                proxy_config = proxy_username + ':' + proxy_password + '@' + proxy_config
            # Prefix by proxy protocol.
            proxy_config = proxy_protocol + proxy_config
        # Set up proxy if applicable.
        if proxy_config:
            self.proxies = {'https': proxy_config}
        else:
            self.proxies = None

        # cURL support
        self.curl_path = False
        if self._cfgparse.has_option('curl','use_curl'):
            use_curl = self._cfgparse.get('curl','use_curl')
            if use_curl.lower() not in ('no', '0', 'false'):
                # Use curl.
                self.curl_path = True
                # FUTURE
                # self.curl_path = 'curl'
                # # Check for specific curl path.
                # if self._cfgparse.has_option('curl','path'):
                #     self.curl_path = self._cfgparse.get('curl','path')

        # ask username (if one doesn't exist)
        if not self._cfgparse.has_option('info','username'):
            username = raw_input('QualysGuard Username: ')
            self._cfgparse.set('info', 'username', username)
        
        # ask password (if one doesn't exist)
        if not self._cfgparse.has_option('info', 'password'):
            password = getpass.getpass('QualysGuard Password: ')
            self._cfgparse.set('info', 'password', password)
        
        logging.debug(self._cfgparse.items('info'))

        if remember_me or remember_me_always:
            # Let's create that config file for next time...
            # Where to store this?
            if remember_me:
                # Store in current working directory.
                config_path = filename
            if remember_me_always:
                # Store in home directory.
                config_path = home_filename
            if not os.path.exists(config_path):
                # Write file only if it doesn't already exists.
                # http://stackoverflow.com/questions/5624359/write-file-with-specific-permissions-in-python
                mode = stat.S_IRUSR | stat.S_IWUSR  # This is 0o600 in octal and 384 in decimal.
                umask_original = os.umask(0)
                try:
                    config_file = os.fdopen(os.open(config_path, os.O_WRONLY | os.O_CREAT, mode), 'w')
                finally:
                    os.umask(umask_original)
                # Add the settings to the structure of the file, and lets write it out...
                self._cfgparse.write(config_file)
                config_file.close()

            
    def get_config_filename(self):
        return self._cfgfile
    

    def get_config(self):
        return self._cfgparse
        

    def get_username(self):
        ''' Returns username from the configfile. '''
        return self._cfgparse.get('info', 'username')


    def get_password(self):
        ''' Returns password from the configfile OR as provided. '''
        return self._cfgparse.get('info', 'password')


    def get_hostname(self):
        ''' Returns hostname. '''
        return self._cfgparse.get('info', 'hostname')