from unittest import TestCase
from quantdsl.priceprocess.schwartzsmith import calibrate, simulatePrices, plotSimulatedPrices
import json
import datetime
import numpy as np


class TestSchwartzSmithPriceProcess(TestCase):

    def testCalibrateAndSimulateAndPlot(self):
        allData = self.allObservedData[:]
        allOptimizedSchwartzParams, allOptimizedSeasonalParams, allRhos, correlationMatrix, simCorrelations, simulatedPrices = calibrate(allData, niter=100, pathCount=50000)

        print "allOptimizedSchwartzParams:", repr(allOptimizedSchwartzParams), "allOptimizedSeasonalParams:", repr(allOptimizedSeasonalParams), "allRhos:", repr(allRhos), "correlationMatrix:", repr(correlationMatrix), "simCorrelations:", repr(simCorrelations)

        plotSimulatedPrices(allData, simulatedPrices)

    def testCalibrate(self):
        allOptimizedParams, allOptimizedSeasonalFactors, allRhos, correlationMatrix, simCorrelations, simulatedPrices = calibrate(self.allObservedData, niter=3)
        # Todo: More assertions...

    def testSimulate(self):
        months = self.allObservedData[0]['months']
        observationDate = self.allObservedData[0]['observationDate']

        simulatedPrices = simulatePrices(observationDate, months, self.allOptimizedParams,
                                         self.allOptimizedSeasonalFactors, self.allRhos, pathCount=100000)

        plotSimulatedPrices(self.allObservedData, simulatedPrices)

    @property
    def allObservedData(self):
        if not hasattr(self, '_allObservedData'):
            alldata = json.loads(exampleData)
            for i, commodityData in enumerate(alldata):
                observationDate = datetime.datetime(*[int(x) for x in commodityData['observationDate'].split('-')])
                commodityData['observationDate'] = observationDate
                months = np.array([datetime.datetime(*[int(x) for x in m.split('-')]) for m in commodityData['months']])
                commodityData['months'] = months
                futures = np.array(commodityData['futures'])
                commodityData['futures'] = futures
                impvols = np.array(commodityData['impliedAtmVols'])
                commodityData['impliedAtmVols'] = impvols
                alldata[i] = commodityData
            self._allObservedData = alldata
        return self._allObservedData

    @property
    def allOptimizedParams(self):
        return [
            {'kappa': 1.2638916531154194, 'sigmae': 0.14734516836401912, 'lambdae': 0.027771544884619619, 'mue': 0.053956757418720531, 'lambdax': -0.38068685060909119, 'sigmax': 0.37788634539857169, 'x0': 0.31473878159096852, 'e0': 3.5234145313102969, 'pxe': 0.92411109909993516},
            {'kappa': 0.57715534625392007, 'sigmae': 0.10887723177092901, 'lambdae': -0.019285073785667982, 'mue': 0.0011320459844574805, 'lambdax': -0.61927428333859191, 'sigmax': 0.30200640241924076, 'x0': 1.0545125321082249, 'e0': 2.6147271447074214, 'pxe': 0.99575689314604499},
            {'kappa': 0.22412720578496656, 'sigmae': 0.0, 'lambdae': 0.044745554489620816, 'mue': 0.05264568084973386, 'lambdax': -0.18589203406147856, 'sigmax': 0.35431191721351657, 'x0': 0.60673647390729191, 'e0': 2.8391451263673511, 'pxe': 0.34674867917684954},
            {'kappa': 0.71269338830064555, 'sigmae': 0.18277823295767753, 'lambdae': 0.16091066685389732, 'mue': 0.18183521244793757, 'lambdax': -0.47577330791415523, 'sigmax': 0.31470493509078834, 'x0': 0.54187739996761075, 'e0': 0.97087034742915801, 'pxe': 0.32484063875303243}]

    @property
    def allOptimizedSeasonalFactors(self):
        return [
            np.array([ 1.0504042 ,  1.04688969,  0.94945908,  0.93742324,  0.92342429,
        1.00579111,  1.18815119,  1.18477495,  0.96726959,  0.89044758,
        0.89134138,  0.96462366]),
            np.array([ 1.07682888,  1.07363859,  0.97399972,  0.95726028,  0.95005403,
        1.03507967,  1.12945642,  1.09732927,  0.9259156 ,  0.89194462,
        0.89361007,  0.99488288]),
            np.array([ 1.23613611,  1.23360067,  1.05497264,  0.93844332,  0.8520782 ,
        0.85190983,  0.95006018,  0.98313315,  0.90873277,  0.9062526 ,
        0.97342678,  1.11125375]),
            np.array([ 1.23515626,  1.20103727,  1.0091085 ,  0.9293931 ,  0.91802625,
        0.91691301,  0.92937769,  0.93158597,  0.91763184,  0.92718699,
        0.97045402,  1.11412907])
        ]

    @property
    def allRhos(self):
        return np.array([
               [ 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ],
               [ 0.96418127,  0.96495962,  0.96593903,  0.96688393,  0.96801058,
                 0.96905276,  0.97011957,  0.97133446,  0.9724159 ,  0.9736226 ,
                 0.97467798,  0.97571055,  0.97671527,  0.97780737,  0.97874097,
                 0.97963756,  0.98060058,  0.98141555,  0.98228646,  0.98302055,
                 0.98371807,  0.98446084,  0.98508562,  0.98567892,  0.98624267,
                 0.98684407,  0.9873517 ,  0.98783605,  0.98835558,  0.98879694,
                 0.98927281,  0.98967936,  0.99007211,  0.99049935,  0.99086767,
                 0.99122658,  0.99157723,  0.99196319,  0.99229977,  0.99263112,
                 0.99299862,  0.99332135,  0.99368078,  0.9939976 ,  0.99431239,
                 0.99466456,  0.99497623,  0.99528688,  0.99559674,  0.99594462,
                 0.99625336,  0.99656176,  0.99690841,  0.99721631,  0.99756249,
                 0.99787003,  0.99817737,  0.99852288,  0.99882972,  0.99913625,
                 0.99948066,  0.99978633,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ,  1.        ,
                 1.        ,  1.        ,  1.        ,  1.        ],
               [ 0.59696328,  0.59763032,  0.59850763,  0.59938136,  0.60044872,
                 0.60145538,  0.60250148,  0.60370895,  0.60479614,  0.6060209 ,
                 0.60710063,  0.60816338,  0.60920219,  0.61033496,  0.61130474,
                 0.61223567,  0.61323307,  0.61407313,  0.61496432,  0.61570801,
                 0.61640613,  0.61713771,  0.61774122,  0.61830204,  0.61882173,
                 0.61935945,  0.61979785,  0.62020117,  0.62061567,  0.62095169,
                 0.6212961 ,  0.62157484,  0.62183009,  0.62209183,  0.62230423,
                 0.62249968,  0.62268013,  0.62286756,  0.62302228,  0.6231676 ,
                 0.62332184,  0.62345242,  0.62359382,  0.62371611,  0.6238365 ,
                 0.62397111,  0.62409124,  0.62421287,  0.62433694,  0.62448049,
                 0.62461253,  0.6247495 ,  0.62491032,  0.62505995,  0.62523652,
                 0.62540135,  0.62557408,  0.62577838,  0.62596925,  0.62616922,
                 0.62640542,  0.62662567,  0.62685586,  0.62712695,  0.62737892,
                 0.62767495,  0.62794941,  0.62823464,  0.62856852,  0.62887693,
                 0.62923705,  0.62952681,  0.62991182,  0.63026584,  0.63063095,
                 0.63105495,  0.63144359,  0.63189401,  0.63230605,  0.63272904,
                 0.63321792,  0.63366398,  0.63417865,  0.6345883 ],
               [ 0.64112709,  0.64434789,  0.64806092,  0.65143148,  0.65529161,
                 0.65877507,  0.66230041,  0.66630866,  0.66990288,  0.67397531,
                 0.6776163 ,  0.68127338,  0.68494357,  0.68908512,  0.69277547,
                 0.69647261,  0.70063849,  0.7043463 ,  0.7085221 ,  0.7122374 ,
                 0.71595559,  0.72014171,  0.72386532,  0.72759126,  0.73131944,
                 0.73551618,  0.73924875,  0.74298319,  0.74718647,  0.75092431,
                 0.75513083,  0.75887095,  0.76261165,  0.76682018,  0.77056086,
                 0.77430085,  0.77803968,  0.78224385,  0.78597852,  0.7897104 ,
                 0.7939047 ,  0.79762874,  0.80181277,  0.80552636,  0.8092341 ,
                 0.81339761,  0.81709101,  0.82077676,  0.82445429,  0.828581  ,
                 0.83223925,  0.83588761,  0.83997954,  0.84360519,  0.84767036,
                 0.85127118,  0.85485963,  0.85888133,  0.86244214,  0.86598935,
                 0.86996326,  0.87348042,  0.87698295,  0.88090543,  0.8843759 ,
                 0.88826164,  0.89169891,  0.8951202 ,  0.89894981,  0.9023365 ,
                 0.90612675,  0.9090601 ,  0.9128126 ,  0.91613013,  0.91943057,
                 0.92312304,  0.92638689,  0.93003799,  0.93326493,  0.93647441,
                 0.94006419,  0.94323649,  0.94678438,  0.94952851]])


exampleData = """
[
    {
        "name": "5x16",
        "impliedAtmVols": [
            0.558348, 
            0.4499949999999999, 
            0.47476599999999997, 
            0.542777, 
            0.542777, 
            0.394565, 
            0.38154699999999997, 
            0.352567, 
            0.352567, 
            0.3349375, 
            0.3349375, 
            0.3317405, 
            0.317308, 
            0.314525, 
            0.312202, 
            0.32533499999999993, 
            0.32533499999999993, 
            0.29711399999999993, 
            0.30157599999999996, 
            0.2908925, 
            0.2888225, 
            0.306935, 
            0.306935, 
            0.270687, 
            0.262706, 
            0.25352899999999995, 
            0.251896, 
            0.294078, 
            0.294078, 
            0.25406949999999995, 
            0.259417, 
            0.23431249999999998, 
            0.228183, 
            0.263764, 
            0.2593135, 
            0.25772649999999997, 
            0.25665699999999997, 
            0.25620849999999995, 
            0.2557715, 
            0.25167749999999994, 
            0.24745699999999998, 
            0.24644499999999997, 
            0.2448695, 
            0.245272, 
            0.24606549999999996, 
            0.24448999999999999, 
            0.2429605, 
            0.2414885, 
            0.2400395, 
            0.238648, 
            0.23729099999999997, 
            0.2359685, 
            0.23469199999999998, 
            0.2334385, 
            0.23221949999999997, 
            0.231035, 
            0.2298735, 
            0.2287465, 
            0.22764249999999997, 
            0.22656149999999997, 
            0.22543449999999998, 
            0.22426149999999997, 
            0.2230425, 
            0.2217775, 
            0.22046649999999998, 
            0.219098, 
            0.2177065, 
            0.21625749999999996, 
            0.214774, 
            0.2132445, 
            0.2116805, 
            0.210082, 
            0.20850649999999998, 
            0.206977, 
            0.205482, 
            0.20401, 
            0.2025725, 
            0.20116949999999997, 
            0.19978949999999998, 
            0.19844399999999998, 
            0.1971215, 
            0.19582199999999997, 
            0.19454549999999998, 
            0.19330349999999996
        ], 
        "months": [
            "2011-04-01", 
            "2011-05-01", 
            "2011-06-01", 
            "2011-07-01", 
            "2011-08-01", 
            "2011-09-01", 
            "2011-10-01", 
            "2011-11-01", 
            "2011-12-01", 
            "2012-01-01", 
            "2012-02-01", 
            "2012-03-01", 
            "2012-04-01", 
            "2012-05-01", 
            "2012-06-01", 
            "2012-07-01", 
            "2012-08-01", 
            "2012-09-01", 
            "2012-10-01", 
            "2012-11-01", 
            "2012-12-01", 
            "2013-01-01", 
            "2013-02-01", 
            "2013-03-01", 
            "2013-04-01", 
            "2013-05-01", 
            "2013-06-01", 
            "2013-07-01", 
            "2013-08-01", 
            "2013-09-01", 
            "2013-10-01", 
            "2013-11-01", 
            "2013-12-01", 
            "2014-01-01", 
            "2014-02-01", 
            "2014-03-01", 
            "2014-04-01", 
            "2014-05-01", 
            "2014-06-01", 
            "2014-07-01", 
            "2014-08-01", 
            "2014-09-01", 
            "2014-10-01", 
            "2014-11-01", 
            "2014-12-01", 
            "2015-01-01", 
            "2015-02-01", 
            "2015-03-01", 
            "2015-04-01", 
            "2015-05-01", 
            "2015-06-01", 
            "2015-07-01", 
            "2015-08-01", 
            "2015-09-01", 
            "2015-10-01", 
            "2015-11-01", 
            "2015-12-01", 
            "2016-01-01", 
            "2016-02-01", 
            "2016-03-01", 
            "2016-04-01", 
            "2016-05-01", 
            "2016-06-01", 
            "2016-07-01", 
            "2016-08-01", 
            "2016-09-01", 
            "2016-10-01", 
            "2016-11-01", 
            "2016-12-01", 
            "2017-01-01", 
            "2017-02-01", 
            "2017-03-01", 
            "2017-04-01", 
            "2017-05-01", 
            "2017-06-01", 
            "2017-07-01", 
            "2017-08-01", 
            "2017-09-01", 
            "2017-10-01", 
            "2017-11-01", 
            "2017-12-01", 
            "2018-01-01", 
            "2018-02-01", 
            "2018-03-01"
        ], 
        "observationDate": "2011-01-21", 
        "futures": [
            45.0, 
            44.75, 
            48.9, 
            58.0, 
            58.0, 
            47.85, 
            44.3, 
            44.55, 
            48.85, 
            52.6, 
            52.6, 
            48.1, 
            47.0, 
            46.55, 
            50.85, 
            59.9, 
            59.9, 
            49.5, 
            45.85, 
            46.2, 
            50.1, 
            54.7245, 
            54.7245, 
            50.1, 
            48.9567, 
            48.4362, 
            52.9055, 
            62.312, 
            62.312, 
            51.4496, 
            47.4447, 
            47.8085, 
            51.8621, 
            57.8534, 
            57.8534, 
            52.7104, 
            51.5075, 
            50.9599, 
            55.6621, 
            65.5586, 
            65.5586, 
            54.1303, 
            49.9167, 
            50.2995, 
            54.5643, 
            61.2394, 
            61.2394, 
            55.7953, 
            54.522, 
            53.9424, 
            58.9198, 
            69.3956, 
            69.3956, 
            57.2983, 
            52.8382, 
            53.2433, 
            57.7578, 
            64.3649, 
            64.3649, 
            58.643, 
            57.3047, 
            56.6955, 
            61.9269, 
            72.9374, 
            72.9374, 
            60.2227, 
            55.5349, 
            55.9608, 
            60.7056, 
            64.02762447144308, 
            64.02762447144308, 
            58.3357075343679, 
            57.0044202981548, 
            56.39841253883252, 
            61.602399722218294, 
            72.55520411161102, 
            72.55520411161102, 
            59.90712982163221, 
            55.24389414508752, 
            55.66756240624209, 
            60.38749939972927, 
            63.69211628474737, 
            63.69211628474737, 
            58.03002529773899
        ]
    }, 
    {
        "name": "2x16",
        "impliedAtmVols": [
            0.55276452, 
            0.4454950499999999, 
            0.47001834, 
            0.5373492299999999, 
            0.5373492299999999, 
            0.39061935, 
            0.37773153, 
            0.34904133000000004, 
            0.34904133000000004, 
            0.331588125, 
            0.331588125, 
            0.328423095, 
            0.31413492, 
            0.31137975, 
            0.30907998, 
            0.32208164999999994, 
            0.32208164999999994, 
            0.29414285999999995, 
            0.29856024, 
            0.287983575, 
            0.28593427499999996, 
            0.30386565, 
            0.30386565, 
            0.26798013, 
            0.26007894, 
            0.2509937099999999, 
            0.24937704, 
            0.29113722000000003, 
            0.29113722000000003, 
            0.25152880499999997, 
            0.25682283, 
            0.23196937499999998, 
            0.22590116999999998, 
            0.26112636, 
            0.25672036499999995, 
            0.255149235, 
            0.25409043, 
            0.25364641499999996, 
            0.25321378499999997, 
            0.24916072499999994, 
            0.24498242999999997, 
            0.24398054999999996, 
            0.242420805, 
            0.24281928, 
            0.24360484499999996, 
            0.24204509999999999, 
            0.240530895, 
            0.239073615, 
            0.237639105, 
            0.23626152, 
            0.23491808999999997, 
            0.233608815, 
            0.23234507999999998, 
            0.231104115, 
            0.22989730499999997, 
            0.22872465, 
            0.227574765, 
            0.226459035, 
            0.22536607499999997, 
            0.22429588499999997, 
            0.223180155, 
            0.22201888499999997, 
            0.220812075, 
            0.21955972499999998, 
            0.218261835, 
            0.21690701999999998, 
            0.215529435, 
            0.21409492499999996, 
            0.21262625999999998, 
            0.211112055, 
            0.209563695, 
            0.20798118, 
            0.206421435, 
            0.20490723, 
            0.20342717999999999, 
            0.2019699, 
            0.20054677499999998, 
            0.19915780499999997, 
            0.19779160499999998, 
            0.19645955999999998, 
            0.195150285, 
            0.19386377999999996, 
            0.19260004499999997, 
            0.19137046499999996
        ], 
        "months": [
            "2011-04-01", 
            "2011-05-01", 
            "2011-06-01", 
            "2011-07-01", 
            "2011-08-01", 
            "2011-09-01", 
            "2011-10-01", 
            "2011-11-01", 
            "2011-12-01", 
            "2012-01-01", 
            "2012-02-01", 
            "2012-03-01", 
            "2012-04-01", 
            "2012-05-01", 
            "2012-06-01", 
            "2012-07-01", 
            "2012-08-01", 
            "2012-09-01", 
            "2012-10-01", 
            "2012-11-01", 
            "2012-12-01", 
            "2013-01-01", 
            "2013-02-01", 
            "2013-03-01", 
            "2013-04-01", 
            "2013-05-01", 
            "2013-06-01", 
            "2013-07-01", 
            "2013-08-01", 
            "2013-09-01", 
            "2013-10-01", 
            "2013-11-01", 
            "2013-12-01", 
            "2014-01-01", 
            "2014-02-01", 
            "2014-03-01", 
            "2014-04-01", 
            "2014-05-01", 
            "2014-06-01", 
            "2014-07-01", 
            "2014-08-01", 
            "2014-09-01", 
            "2014-10-01", 
            "2014-11-01", 
            "2014-12-01", 
            "2015-01-01", 
            "2015-02-01", 
            "2015-03-01", 
            "2015-04-01", 
            "2015-05-01", 
            "2015-06-01", 
            "2015-07-01", 
            "2015-08-01", 
            "2015-09-01", 
            "2015-10-01", 
            "2015-11-01", 
            "2015-12-01", 
            "2016-01-01", 
            "2016-02-01", 
            "2016-03-01", 
            "2016-04-01", 
            "2016-05-01", 
            "2016-06-01", 
            "2016-07-01", 
            "2016-08-01", 
            "2016-09-01", 
            "2016-10-01", 
            "2016-11-01", 
            "2016-12-01", 
            "2017-01-01", 
            "2017-02-01", 
            "2017-03-01", 
            "2017-04-01", 
            "2017-05-01", 
            "2017-06-01", 
            "2017-07-01", 
            "2017-08-01", 
            "2017-09-01", 
            "2017-10-01", 
            "2017-11-01", 
            "2017-12-01", 
            "2018-01-01", 
            "2018-02-01", 
            "2018-03-01"
        ], 
        "observationDate": "2011-01-21", 
        "futures": [
            39.1015, 
            39.15625, 
            42.7875, 
            46.87119, 
            45.65369, 
            38.89996, 
            37.6733, 
            37.9059, 
            42.74375, 
            46.025, 
            46.025, 
            42.0875, 
            40.83934, 
            40.73125, 
            44.49375, 
            48.40662, 
            47.14924, 
            40.24134, 
            38.99144, 
            39.30982, 
            43.8375, 
            47.88394, 
            47.88394, 
            43.8375, 
            42.53956, 
            42.38168, 
            46.29231, 
            50.35582, 
            49.0478, 
            41.82628, 
            40.3476, 
            40.67843, 
            45.37934, 
            50.62173, 
            50.62173, 
            46.1216, 
            44.75601, 
            44.58991, 
            48.70434, 
            52.97947, 
            51.60331, 
            44.00557, 
            42.44982, 
            42.79793, 
            47.74376, 
            53.58448, 
            53.58448, 
            48.82089, 
            47.37538, 
            47.1996, 
            51.55483, 
            56.08024, 
            54.62353, 
            46.58101, 
            44.9343, 
            45.30269, 
            50.53808, 
            56.31929, 
            56.31929, 
            51.31263, 
            49.79333, 
            49.60856, 
            54.18604, 
            58.94246, 
            57.4114, 
            48.95842, 
            47.22761, 
            47.61491, 
            53.1174, 
            54.899192514356194, 
            54.899192514356194, 
            50.0187760319409, 
            48.53778535917032, 
            48.35767435633492, 
            52.81973266265617, 
            57.456218975944815, 
            55.963764832949934, 
            47.72392771248903, 
            46.03676028911113, 
            46.41429447430434, 
            51.77803854526688, 
            53.51490295293749, 
            53.51490295293749, 
            48.7575467430429
        ]
    }, 
    {
        "name": "7x8",
        "impliedAtmVols": [
            0.55276452, 
            0.4454950499999999, 
            0.47001834, 
            0.5373492299999999, 
            0.5373492299999999, 
            0.39061935, 
            0.37773153, 
            0.34904133000000004, 
            0.34904133000000004, 
            0.331588125, 
            0.331588125, 
            0.328423095, 
            0.31413492, 
            0.31137975, 
            0.30907998, 
            0.32208164999999994, 
            0.32208164999999994, 
            0.29414285999999995, 
            0.29856024, 
            0.287983575, 
            0.28593427499999996, 
            0.30386565, 
            0.30386565, 
            0.26798013, 
            0.26007894, 
            0.2509937099999999, 
            0.24937704, 
            0.29113722000000003, 
            0.29113722000000003, 
            0.25152880499999997, 
            0.25682283, 
            0.23196937499999998, 
            0.22590116999999998, 
            0.26112636, 
            0.25672036499999995, 
            0.255149235, 
            0.25409043, 
            0.25364641499999996, 
            0.25321378499999997, 
            0.24916072499999994, 
            0.24498242999999997, 
            0.24398054999999996, 
            0.242420805, 
            0.24281928, 
            0.24360484499999996, 
            0.24204509999999999, 
            0.240530895, 
            0.239073615, 
            0.237639105, 
            0.23626152, 
            0.23491808999999997, 
            0.233608815, 
            0.23234507999999998, 
            0.231104115, 
            0.22989730499999997, 
            0.22872465, 
            0.227574765, 
            0.226459035, 
            0.22536607499999997, 
            0.22429588499999997, 
            0.223180155, 
            0.22201888499999997, 
            0.220812075, 
            0.21955972499999998, 
            0.218261835, 
            0.21690701999999998, 
            0.215529435, 
            0.21409492499999996, 
            0.21262625999999998, 
            0.211112055, 
            0.209563695, 
            0.20798118, 
            0.206421435, 
            0.20490723, 
            0.20342717999999999, 
            0.2019699, 
            0.20054677499999998, 
            0.19915780499999997, 
            0.19779160499999998, 
            0.19645955999999998, 
            0.195150285, 
            0.19386377999999996, 
            0.19260004499999997, 
            0.19137046499999996
        ], 
        "months": [
            "2011-04-01", 
            "2011-05-01", 
            "2011-06-01", 
            "2011-07-01", 
            "2011-08-01", 
            "2011-09-01", 
            "2011-10-01", 
            "2011-11-01", 
            "2011-12-01", 
            "2012-01-01", 
            "2012-02-01", 
            "2012-03-01", 
            "2012-04-01", 
            "2012-05-01", 
            "2012-06-01", 
            "2012-07-01", 
            "2012-08-01", 
            "2012-09-01", 
            "2012-10-01", 
            "2012-11-01", 
            "2012-12-01", 
            "2013-01-01", 
            "2013-02-01", 
            "2013-03-01", 
            "2013-04-01", 
            "2013-05-01", 
            "2013-06-01", 
            "2013-07-01", 
            "2013-08-01", 
            "2013-09-01", 
            "2013-10-01", 
            "2013-11-01", 
            "2013-12-01", 
            "2014-01-01", 
            "2014-02-01", 
            "2014-03-01", 
            "2014-04-01", 
            "2014-05-01", 
            "2014-06-01", 
            "2014-07-01", 
            "2014-08-01", 
            "2014-09-01", 
            "2014-10-01", 
            "2014-11-01", 
            "2014-12-01", 
            "2015-01-01", 
            "2015-02-01", 
            "2015-03-01", 
            "2015-04-01", 
            "2015-05-01", 
            "2015-06-01", 
            "2015-07-01", 
            "2015-08-01", 
            "2015-09-01", 
            "2015-10-01", 
            "2015-11-01", 
            "2015-12-01", 
            "2016-01-01", 
            "2016-02-01", 
            "2016-03-01", 
            "2016-04-01", 
            "2016-05-01", 
            "2016-06-01", 
            "2016-07-01", 
            "2016-08-01", 
            "2016-09-01", 
            "2016-10-01", 
            "2016-11-01", 
            "2016-12-01", 
            "2017-01-01", 
            "2017-02-01", 
            "2017-03-01", 
            "2017-04-01", 
            "2017-05-01", 
            "2017-06-01", 
            "2017-07-01", 
            "2017-08-01", 
            "2017-09-01", 
            "2017-10-01", 
            "2017-11-01", 
            "2017-12-01", 
            "2018-01-01", 
            "2018-02-01", 
            "2018-03-01"
        ], 
        "observationDate": "2011-01-21", 
        "futures": [
            31.9791, 
            30.01532, 
            29.69667, 
            30.84948, 
            33.29165, 
            31.46002, 
            31.63013, 
            34.61646, 
            37.40726, 
            43.10484, 
            43.27069, 
            37.20726, 
            32.45639, 
            30.0125, 
            29.38375, 
            32.68444, 
            34.56652, 
            31.58969, 
            32.33345, 
            34.73411, 
            39.75565, 
            45.93463, 
            45.94599, 
            39.17758, 
            34.69984, 
            31.74129, 
            30.61846, 
            35.77007, 
            36.52956, 
            33.94665, 
            34.38436, 
            36.81671, 
            42.49087, 
            49.30485, 
            49.31253, 
            42.24149, 
            37.3779, 
            33.82741, 
            33.76812, 
            38.61936, 
            38.92108, 
            36.9469, 
            37.01295, 
            39.58047, 
            45.86933, 
            52.02408, 
            52.09401, 
            44.78236, 
            39.4835, 
            35.2734, 
            36.3225, 
            41.41102, 
            41.10239, 
            39.02451, 
            38.85214, 
            41.93687, 
            48.45546, 
            55.48635, 
            55.56331, 
            47.9204, 
            41.79232, 
            38.18742, 
            38.79423, 
            42.32447, 
            44.99597, 
            41.66463, 
            41.22851, 
            44.88913, 
            51.62326, 
            54.08725519389445, 
            54.16227463849158, 
            46.71208150821778, 
            40.738521762287455, 
            37.22451973749271, 
            37.81602895235739, 
            41.25725353778596, 
            43.86139135277088, 
            40.61405148057478, 
            40.188928297392586, 
            43.75724533586915, 
            50.321573460153054, 
            52.72343872699253, 
            52.79656654751829, 
            45.534230908556296
        ]
    }, 
    {
        "name": "Fuel",
        "impliedAtmVols": [
            0.40154549999999994, 
            0.3974975, 
            0.391874, 
            0.387665, 
            0.3925295, 
            0.40544399999999997, 
            0.41543749999999996, 
            0.38081099999999996, 
            0.3521645, 
            0.3401585, 
            0.3453565, 
            0.34562099999999996, 
            0.3135015, 
            0.30423249999999996, 
            0.29464149999999995, 
            0.29277849999999994, 
            0.2949175, 
            0.3055895, 
            0.315445, 
            0.30127699999999996, 
            0.28514249999999997, 
            0.27932349999999995, 
            0.28779899999999997, 
            0.289938, 
            0.2736885, 
            0.27070999999999995, 
            0.2640515, 
            0.2629245, 
            0.2629245, 
            0.270848, 
            0.2798985, 
            0.2708135, 
            0.260843, 
            0.25643849999999996, 
            0.2626715, 
            0.267076, 
            0.256266, 
            0.25667999999999996, 
            0.25210299999999997, 
            0.2496535, 
            0.24797449999999996, 
            0.25828999999999996, 
            0.264339, 
            0.2649255, 
            0.24907849999999998, 
            0.24798599999999998, 
            0.25244799999999995, 
            0.2586235, 
            0.254127, 
            0.2522525, 
            0.2491475, 
            0.2467555, 
            0.2472845, 
            0.2535175, 
            0.2597045, 
            0.2527585, 
            0.24541, 
            0.251436, 
            0.2507805, 
            0.24954999999999997, 
            0.24852649999999998, 
            0.2474915, 
            0.24656, 
            0.24567449999999996, 
            0.24459349999999996, 
            0.24284549999999996, 
            0.2404305, 
            0.23845249999999998, 
            0.236946, 
            0.2353245, 
            0.23363399999999998, 
            0.23186299999999999, 
            0.23012649999999998, 
            0.228436, 
            0.22677999999999995, 
            0.22515849999999998, 
            0.22357149999999998, 
            0.222019, 
            0.22050099999999997, 
            0.2190175, 
            0.21755699999999997, 
            0.2161195, 
            0.21471649999999998, 
            0.21334799999999998
        ], 
        "months": [
            "2011-04-01", 
            "2011-05-01", 
            "2011-06-01", 
            "2011-07-01", 
            "2011-08-01", 
            "2011-09-01", 
            "2011-10-01", 
            "2011-11-01", 
            "2011-12-01", 
            "2012-01-01", 
            "2012-02-01", 
            "2012-03-01", 
            "2012-04-01", 
            "2012-05-01", 
            "2012-06-01", 
            "2012-07-01", 
            "2012-08-01", 
            "2012-09-01", 
            "2012-10-01", 
            "2012-11-01", 
            "2012-12-01", 
            "2013-01-01", 
            "2013-02-01", 
            "2013-03-01", 
            "2013-04-01", 
            "2013-05-01", 
            "2013-06-01", 
            "2013-07-01", 
            "2013-08-01", 
            "2013-09-01", 
            "2013-10-01", 
            "2013-11-01", 
            "2013-12-01", 
            "2014-01-01", 
            "2014-02-01", 
            "2014-03-01", 
            "2014-04-01", 
            "2014-05-01", 
            "2014-06-01", 
            "2014-07-01", 
            "2014-08-01", 
            "2014-09-01", 
            "2014-10-01", 
            "2014-11-01", 
            "2014-12-01", 
            "2015-01-01", 
            "2015-02-01", 
            "2015-03-01", 
            "2015-04-01", 
            "2015-05-01", 
            "2015-06-01", 
            "2015-07-01", 
            "2015-08-01", 
            "2015-09-01", 
            "2015-10-01", 
            "2015-11-01", 
            "2015-12-01", 
            "2016-01-01", 
            "2016-02-01", 
            "2016-03-01", 
            "2016-04-01", 
            "2016-05-01", 
            "2016-06-01", 
            "2016-07-01", 
            "2016-08-01", 
            "2016-09-01", 
            "2016-10-01", 
            "2016-11-01", 
            "2016-12-01", 
            "2017-01-01", 
            "2017-02-01", 
            "2017-03-01", 
            "2017-04-01", 
            "2017-05-01", 
            "2017-06-01", 
            "2017-07-01", 
            "2017-08-01", 
            "2017-09-01", 
            "2017-10-01", 
            "2017-11-01", 
            "2017-12-01", 
            "2018-01-01", 
            "2018-02-01", 
            "2018-03-01"
        ], 
        "observationDate": "2011-01-21", 
        "futures": [
            4.2928, 
            4.2847, 
            4.352, 
            4.4602, 
            4.5055, 
            4.411, 
            4.4622, 
            4.8135, 
            6.031, 
            6.924, 
            6.7655, 
            5.282, 
            4.89325, 
            4.88025, 
            4.91525, 
            5.02438, 
            5.05725, 
            4.96663, 
            5.0165, 
            5.36865, 
            6.3799, 
            7.2058, 
            7.0345, 
            5.72815, 
            5.275, 
            5.2402, 
            5.2702, 
            5.3746, 
            5.4074, 
            5.3244, 
            5.3842, 
            5.6901, 
            6.6269, 
            7.40525, 
            7.19125, 
            6.0435, 
            5.592, 
            5.5522, 
            5.5772, 
            5.6866, 
            5.7244, 
            5.6364, 
            5.6992, 
            6.0081, 
            6.9449, 
            7.72525, 
            7.51625, 
            6.3685, 
            5.909, 
            5.8662, 
            5.8912, 
            6.0006, 
            6.0464, 
            5.9584, 
            6.0212, 
            6.3231, 
            7.2579, 
            8.03825, 
            7.82925, 
            6.6715, 
            6.207, 
            6.1572, 
            6.1772, 
            6.2816, 
            6.3194, 
            6.2414, 
            6.3162, 
            6.6201, 
            7.5499, 
            8.33525, 
            8.12625, 
            6.9685, 
            6.494, 
            6.4392, 
            6.4642, 
            6.5706, 
            6.6134, 
            6.5354, 
            6.6132, 
            6.9171, 
            7.8469, 
            8.63225, 
            8.41825, 
            7.2605
        ]
    }
]
"""

def read_xl_doc():
    # We want the data in the first columns of the various sheets.
    alldata = []
    print_data = []
    import xlrd
    with xlrd.open_workbook('SchwartzSmithMultiMarketExampleData.xls') as wb:
        for sheet_name in wb.sheet_names():
            sheet = wb.sheet_by_name(sheet_name)
            month_col_title = sheet.cell_value(1, 0)
            assert month_col_title == 'Month', month_col_title
            from xlrd import xldate_as_tuple
            months = [datetime.datetime(*xldate_as_tuple(c.value, wb.datemode)) for c in sheet.col_slice(0)[2:]]

            fo_col_title = sheet.cell_value(1, 1)
            assert fo_col_title == 'F0', fo_col_title
            futures = [c.value for c in sheet.col_slice(1)[2:]]

            iv_col_title = sheet.cell_value(1, 2)
            assert iv_col_title == "Vol", iv_col_title
            impvols = [c.value for c in sheet.col_slice(2)[2:]]

            def param(params, key, row, col):
                v = sheet.cell_value(row, col)
                if isinstance(key, basestring) and key.lower()[-4:] == 'date':
                    v = datetime.datetime(*xldate_as_tuple(v, wb.datemode))
                params[key] = v

            observation_date = datetime.datetime(*xldate_as_tuple(sheet.cell_value(21, 15), wb.datemode))

            seasonal_factors = [1] * 12
            for month_int in range(12):
                param(seasonal_factors, month_int, month_int+24, 14)

            params = {}
            param(params, 'kappa', 18, 13)
            param(params, 'mue', 18, 14)
            param(params, 'sigmax', 18, 15)
            param(params, 'sigmae', 18, 16)
            param(params, 'lambdae', 18, 17)
            param(params, 'lambdax', 18, 18)
            param(params, 'pxe', 18, 19)
            param(params, 'x0', 18, 20)
            param(params, 'e0', 18, 21)

            alldata.append([observation_date, months, futures, impvols, seasonal_factors, params])
            idata = {
                'observationDate': "%04d-%02d-%02d" % (observation_date.year, observation_date.month, observation_date.day),
                'months': ["%04d-%02d-%02d" % (m.year, m.month, m.day) for m in months],
                'futures': [i for i in futures],
                'impvols': [i for i in impvols]
            }
            print_data.append(idata)

    import json
    print "import datetime"
    print "from numpy import array"
    print
    print json.dumps(print_data, indent=4)

    return alldata

