
import os
import re
import sys
import threading

import webob

class MakoTemplates(object):
    def __init__(self, config):
        from mako.lookup import TemplateLookup
        self.lookup = TemplateLookup(
            directories = os.path.join(config["root_dir"], "templates"),
            module_directory = os.path.join(config["tmp_dir"], "templates"),
            collection_size = config.get("collection_size", 100),
            output_encoding = config["encoding"],
            encoding_errors = config["encoding_errors"]
        )

    def render(self, q, template, **kwargs):
        try:
            tmpl = self.lookup.get_template(template)
            return tmpl.render(**kwargs)
        except:
            from mako import exceptions
            q.res.status = 500
            return exceptions.html_error_template().render()

class Quark(object):
    def __init__(self, urls, config=None, template_class=None):
        self.config = {
            "log": True,

            "encoding": "utf-8",
            "encoding_errors": "replace",
            "content_type": "text/html",

            "root_dir": "./",
            "tmp_dir": "./temp/"
        }
        if config is not None:
            self.config.update(config)

        if template_class is not None:
            self.templates = template_class(self.config)
        else:
            self.templates = MakoTemplates(self.config)

        self.handlers = self.load_handlers(urls)

    def __call__(self, environ, start_response):
        req = webob.Request(environ)
        res = req.get_response(self.handle, catch_exc_info=True)
        start_response(res.status, res.headers.items())
        return [res.body]

    def load_handlers(self, urls):
        ret = []
        for url in urls:
            modname, varname = url[1].rsplit(":", 1)
            mod = __import__(modname)
            for m in modname.split(".")[1:]:
                mod = getattr(mod, m)
            handler = getattr(mod, varname)
            ret.append((re.compile(url[0]), handler()))
        return ret

    def new_q(self, environ):
        q = threading.local()
        q.req = webob.Request(environ)
        q.res = webob.Response()
        q.cfg = self.config.copy()
        def _render(*args, **kwargs):
            return self.templates.render(q, *args, **kwargs)
        q.render = _render
        return q

    def handle(self, environ, start_response):
        try:
            q = self.new_q(environ)
            for handler in self.handlers:
                match = handler[0].match(q.req.path_info)
                if not match:
                    continue
                if not hasattr(handler[1], q.req.method):
                    raise webob.exc.HTTPMethodNotAllowed()
                func = getattr(handler[1], q.req.method)
                body = func(q, *match.groups())
                if isinstance(q.res.body, unicode):
                    q.res.unicode_body = body
                elif isinstance(q.res.body, str):
                    q.res.body = body
                return q.res(environ, start_response)
            raise webob.exc.HTTPNotFound()
        except webob.exc.HTTPException, inst:
            return inst(environ, start_response)

