"""
Module to assist in the super-common operation of
wrapping values with prefix and suffix strings.
"""

import re

class NullType(object):
    """
    A 'null' type different from, but parallel to, None. Core function
    is representing emptyness in a way that doesn't overload None.
    
    Instantiate the desired Null values.
    """
    def __nonzero__(self):
        """I am always False."""
        return False
    
Default = NullType()

def stringify(v):
    """
    Return a string. If it's already a string, just return that.
    Otherwise, stringify it. Not safe for Python 3.
    """
    return v if isinstance(v, basestring) else str(v)

class Quoter(object):
    """
    A quote style. Instantiate it with the style information. Call
    it with a value to quote the value.
    """
    
    styles = {}         # remember named styles
    encoding = None     # global encoding
    
    def __init__(self, prefix=None, suffix=None, chars=None, name=None, padding=0, margin=0, encoding=Default):
        """
        Create a quoting style.
        """
        if chars:
            assert prefix is None and suffix is None
            assert len(chars) % 2 == 0
            half = len(chars) / 2
            self.prefix = unicode(chars[:half])
            self.suffix = unicode(chars[half:])
        else:
            assert chars is None
            self.prefix = unicode(prefix)
            self.suffix = unicode(suffix if suffix is not None else self.prefix)
        self.padding = padding
        self.margin  = margin
        self.encoding = encoding
        if name:
            Quoter.styles[name] = self
            
    def _whitespace(self, padding, margin):
        """
        Compute the appropriate margin and padding strings.
        """
        padding = padding if padding is not Default else self.padding
        margin  = margin  if margin  is not Default else self.margin
        pstr = ' ' * padding if isinstance(padding, int) else padding
        mstr  = ' ' * margin  if isinstance(margin, int) else margin
        return (pstr, mstr)
        
    def _output(self, *parts, **kwargs):
        """
        Given a list of string parts, concatentate them and output
        with the given encoding (if any).
        """
        outstr = ''.join(parts)
        encoding = kwargs.get('encoding', None) or self.encoding or Quoter.encoding
        return outstr.encode(encoding) if encoding else outstr
        
    def __call__(self, value, padding=Default, margin=Default, encoding=Default):
        """
        Quote the value, with the given padding, margin, and encoding.
        """
        pstr, mstr = self._whitespace(padding, margin)
        return self._output(mstr, self.prefix, pstr, stringify(value), pstr, self.suffix, mstr)

# create some default named styles (ASCII)

braces   = Quoter('{', '}', name='braces')
brackets = Quoter('[', ']', name='brackets')
angles   = Quoter('<', '>', name='angles')
parens   = Quoter('(', ')', name='parens')
single   = Quoter("'",      name='single')
double   = Quoter('"',      name='double')
triple   = Quoter('"""',    name='triple')
backticks= Quoter("`",      name='backticks')
qb, qs, qd, qt = backticks, single, double, triple # short aliases

# and some Unicode styles
anglequote  = Quoter(u'\u00ab', u'\u00bb', name='anglequote')  # guillemet!
curlysingle = Quoter(u'\u2018', u'\u2019', name='curlysingle')  
curlydouble = Quoter(u'\u201c', u'\u201d', name='curlydouble')

class LambdaQuoter(Quoter):
    """
    A Quoter that uses code to decide what quotes to use, based on the value.
    """
    def __init__(self, func, name=None, padding=0, margin=0, encoding=None):
        """
        Create a dynamic quoting style.
        """
        self.func = func
        self.padding = padding
        self.margin  = margin
        self.encoding = encoding
        if name:
            Quoter.styles[name] = self
            
    def __call__(self, value, padding=Default, margin=Default, encoding=Default):
        """
        Quote the value, based on the instance's function.
        """
        pstr, mstr = self._whitespace(padding, margin)
        prefix, value, suffix = self.func(value)
        return self._output(mstr, prefix, pstr, stringify(value), pstr, suffix, mstr, encoding=encoding)

class HTMLQuoter(Quoter):
    """
    A more sophisticated quoter that supports attributes and void elements.
    """
    
    def __init__(self, tag, attquote=single, void=False, name=None, padding=0, margin=0, encoding=None):
        """
        Create an HTMLQuoter
        """
        tagname, atts = self._parse_selector(tag)
        self.tag = unicode(tagname)
        self.atts = atts or {}
        self.attquote = attquote
        self.void = void
        self.padding = padding
        self.margin  = margin
        self.encoding = encoding
        if name:
            Quoter.styles[name] = self
            
    def _attstr(self, atts):
        """
        Format an attribute dict.
        """
        return ' '.join([''] + ["{}={}".format(k, self.attquote(v)) for k,v in atts.items()])
    
    def _parse_selector(self, selector):
        """
        Parse a selector (modeled on jQuery and CSS). Returns a (tagname, id, classes)
        tuple.
        """
        tagnames = re.findall(r'^(\w+)',      selector)
        classes  = re.findall(r'\.([\w\-]+)', selector)
        ids      = re.findall(r'\#([\w\-]+)', selector)
        assert len(tagnames) <= 1
        assert len(ids) <= 1
        atts = {}
        if ids:
            atts['id'] = ids[0]
        if classes:
            atts['class'] = ' '.join(classes)
        return (tagnames[0] if tagnames else None, atts)
    
    def __call__(self, value=None, atts=None, padding=Default, margin=Default, encoding=Default):
        """
        Quote a value in X/HTML style, with optional attributes.
        """
        pstr, mstr = self._whitespace(padding, margin)
        callatts = self._parse_selector(atts)[1] if isinstance(atts, basestring) else atts or {}
        atts = {}
        atts.update(self.atts)
        atts.update(callatts)            
        astr = self._attstr(atts) if atts else ''
        if self.void:
            assert value is None
            return self._output(mstr, '<', self.tag, astr, '>', mstr, encoding=encoding)
        else:
            return self._output(mstr, '<', self.tag, astr, '>', pstr, stringify(value),
                                pstr, '</', self.tag, '>', mstr, encoding=encoding)
        
    # could improve kwargs handling of HTMLQuoter

    
# need def_tag to define these

#comment = Quoter('<!--', '-->', name='comment')
#for t in """area base br col command embed hr img input keygen link meta
#            param source track wbr""".split():
#    def_tag(t, Quoter(t, name=t, void=True))
#for t in """a article aside audio b bdi bdo blockquote body button canvas
#            caption cite code colgroup datalist dd del details dfn div dl dt
#            em fieldset figcaption figure footer form h1 h2 h3 h4 h5 h6
#            head header hgroup html i iframe ins kbd label legend li
#            map mark menu meter nav noscript object ol optgroup option
#            output p pre progress q rp rt ruby s samp script section
#            select small span strong style sub summary sup table tbody
#            td textarea tfoot th thead time tile tr u ul var video""".split():
#    def_tag(t, Quoter(t, name=t))

# see also http://en.wikipedia.org/wiki/Quotation_mark_glyphs

def quote(value, style='double', **kwargs):
    try:
        quoter = Quoter.styles[style]
    except KeyError:
        raise ValueError("'{}' is not a known quote style".format(style))
    return quoter(value, **kwargs)
