
import re

class NullType(object):
    def __nonzero__(self):
        return False
    
Default = NullType()

def stringify(v):
    return v if isinstance(v, basestring) else str(v)

class Quoter(object):
    
    styles = {}
    encoding = None
    
    def __init__(self, prefix, suffix=None, name=None, padding=0, margin=0, encoding=Default):
        self.prefix = unicode(prefix)
        self.suffix = unicode(suffix if suffix is not None else self.prefix)
        self.padding = padding
        self.margin  = margin
        self.encoding = encoding
        if name:
            Quoter.styles[name] = self
        
    def _output(self, *parts, **kwargs):
        outstr = ''.join(parts)
        encoding = kwargs.get('encoding', None) or self.encoding or Quoter.encoding
        return outstr.encode(encoding) if encoding else outstr
        
    def __call__(self, value, padding=Default, margin=Default, encoding=Default):
        margin  = margin  or self.margin
        padding = padding or self.padding
        mstr  = ' ' * margin  if isinstance(margin, int) else margin
        pstr = ' ' * padding if isinstance(padding, int) else padding
        return self._output(mstr, self.prefix, pstr, stringify(value), pstr, self.suffix, mstr)

   
braces   = Quoter('{', '}', name='braces')
brackets = Quoter('[', ']', name='brackets')
angles   = Quoter('<', '>', name='angles')
parens   = Quoter('(', ')', name='parens')
double   = Quoter('"',      name='double')
single   = Quoter("'",      name='single')
backticks= Quoter("`",      name='backticks')
anglequote  = Quoter(u'\u00ab', u'\u00bb', name='anglequote')  # guillemet!
curlysingle = Quoter(u'\u2018', u'\u2019', name='curlysingle')  
curlydouble = Quoter(u'\u201c', u'\u201d', name='curlydouble')

class LambdaQuoter(Quoter):
    """
    A Quoter that uses code to decide what quotes to use, based on the value.
    """
    def __init__(self, func, name=None, padding=0, margin=0, encoding=None):
        self.func = func
        self.padding = padding
        self.margin  = margin
        self.encoding = encoding
        if name:
            Quoter.styles[name] = self
            
    def __call__(self, value, padding=Default, margin=Default, encoding=Default):
        margin  = margin  or self.margin
        padding = padding or self.padding
        mstr  = ' ' * margin  if isinstance(margin, int) else margin
        pstr = ' ' * padding if isinstance(padding, int) else padding
        prefix, value, suffix = self.func(value)
        return self._output(mstr, prefix, pstr, stringify(value), pstr, suffix, mstr, encoding=encoding)

class HTMLQuoter(Quoter):
    
    def __init__(self, tag, attquote=single, void=False, name=None, padding=0, margin=0, encoding=None):
        tagname, atts = self._parse_selector(tag)
        self.tag = unicode(tagname)
        self.atts = atts or {}
        self.attquote = attquote
        self.void = void
        self.padding = padding
        self.margin  = margin
        self.encoding = encoding
        if name:
            Quoter.styles[name] = self
            
    def _attstr(self, atts):
        return ' '.join([''] + ["{}={}".format(k, self.attquote(v)) for k,v in atts.items()])
    
    def _parse_selector(self, tag):
        """
        Return a selector (modeled on jQuery and CSS).
        """
        tagnames = re.findall(r'^(\w+)',      tag)
        classes  = re.findall(r'\.([\w\-]+)', tag)
        ids      = re.findall(r'\#([\w\-]+)', tag)
        assert len(tagnames) <= 1
        assert len(ids) <= 1
        atts = {}
        if ids:
            atts['id'] = ids[0]
        if classes:
            atts['class'] = ' '.join(classes)
        return (tagnames[0] if tagnames else None, atts)
    
    def __call__(self, value=None, atts=None, padding=Default, margin=Default, encoding=Default):
        
        margin  = margin  or self.margin
        padding = padding or self.padding
        mstr  = ' ' * margin  if isinstance(margin, int) else margin
        pstr = ' ' * padding if isinstance(padding, int) else padding
        
        callatts = self._parse_selector(atts)[1] if isinstance(atts, basestring) else atts or {}
        atts = {}
        atts.update(self.atts)
        atts.update(callatts)            
        astr = self._attstr(atts) if atts else ''
        if self.void:
            assert value is None
            return self._output(mstr, '<', self.tag, astr, '>', mstr, encoding=encoding)
        else:
            return self._output(mstr, '<', self.tag, astr, '>', pstr, stringify(value),
                                pstr, '</', self.tag, '>', mstr, encoding=encoding)
        
    # could improve kwargs handling of HTMLQuoter

    
# need def_tag to define these

#comment = Quoter('<!--', '-->', name='comment')
#for t in """area base br col command embed hr img input keygen link meta
#            param source track wbr""".split():
#    def_tag(t, Quoter(t, name=t, void=True))
#for t in """a article aside audio b bdi bdo blockquote body button canvas
#            caption cite code colgroup datalist dd del details dfn div dl dt
#            em fieldset figcaption figure footer form h1 h2 h3 h4 h5 h6
#            head header hgroup html i iframe ins kbd label legend li
#            map mark menu meter nav noscript object ol optgroup option
#            output p pre progress q rp rt ruby s samp script section
#            select small span strong style sub summary sup table tbody
#            td textarea tfoot th thead time tile tr u ul var video""".split():
#    def_tag(t, Quoter(t, name=t))

# see also http://en.wikipedia.org/wiki/Quotation_mark_glyphs

def quote(value, style='double', **kwargs):
    try:
        quoter = Quoter.styles[style]
    except KeyError:
        raise ValueError("'{}' is not a known quote style".format(style))
    return quoter(value, **kwargs)
