"""
Module to assist in the super-common operation of
wrapping values with prefix and suffix strings.
"""

import re
import six
from options import Options, Prohibited
    
def is_string(v):
    return isinstance(v, six.string_types)    

def stringify(v):
    """
    Return a string. If it's already a string, just return that.
    Otherwise, stringify it. Not safe for Python 3.
    """
    return v if is_string(v) else str(v)

class Quoter(object):
    """
    A quote style. Instantiate it with the style information. Call
    it with a value to quote the value.
    """
    
    styles = {}         # remember named styles
                        # NB global across all style Quoter classes
    
    options = Options(
        prefix   = '',
        suffix   = None,
        name     = None,
        padding  = 0,
        margin   = 0,
        encoding = None
    )
    
    # TBD add back chars= option as a MultiSetter
    
    def __init__(self, *args, **kwargs):
        """
        Create a quoting style.
        """
        self.options = Quoter.options.push(kwargs)
        if args:
            used = self.options.addflat(args, ['prefix', 'suffix'])
            if 'suffix' not in used:
                self.options.suffix = self.options.prefix
                # this suffix = prefix behavior appropriate for flat args only
        if self.options.name:
            Quoter.styles[self.options.name] = self
            
    def _whitespace(self, opts):
        """
        Compute the appropriate margin and padding strings.
        """   
        pstr = ' ' * opts.padding if isinstance(opts.padding, int) else opts.padding
        mstr = ' ' * opts.margin  if isinstance(opts.margin, int)  else opts.margin
        return (pstr, mstr)
        
    def _output(self, parts, opts):
        """
        Given a list of string parts, concatentate them and output
        with the given encoding (if any).
        """
        outstr = ''.join(parts)
        return outstr.encode(opts.encoding) if opts.encoding else outstr
        
    def __call__(self, value, **kwargs):
        """
        Quote the value, with the given padding, margin, and encoding.
        """
        opts = self.options.push(kwargs)
        pstr, mstr = self._whitespace(opts)
        suffix = opts.suffix if opts.suffix is not None else opts.prefix
        parts = [ mstr, opts.prefix, pstr, stringify(value), pstr, suffix, mstr ]
        return self._output(parts, opts)

# create some default named styles (ASCII)

braces   = Quoter('{', '}', name='braces')
brackets = Quoter('[', ']', name='brackets')
angles   = Quoter('<', '>', name='angles')
parens   = Quoter('(', ')', name='parens')
single   = Quoter("'",      name='single')
double   = Quoter('"',      name='double')
triple   = Quoter('"""',    name='triple')
backticks= Quoter("`",      name='backticks')
qb, qs, qd, qt = backticks, single, double, triple # short aliases

# and some Unicode styles
anglequote  = Quoter(six.u('\u00ab'), six.u('\u00bb'), name='anglequote')  # guillemet!
curlysingle = Quoter(six.u('\u2018'), six.u('\u2019'), name='curlysingle')  
curlydouble = Quoter(six.u('\u201c'), six.u('\u201d'), name='curlydouble')

class LambdaQuoter(Quoter):
    """
    A Quoter that uses code to decide what quotes to use, based on the value.
    """
    
    options = Quoter.options.add(
        func   = None,
        prefix = Prohibited,
        suffix = Prohibited,
    )
    
    def __init__(self, *args, **kwargs):
        """
        Create a dynamic quoting style.
        """
        self.options = LambdaQuoter.options.push(kwargs)
        if args:
            self.options.addflat(args, ['func'])
        if self.options.name:
            Quoter.styles[self.options.name] = self
    
    # NB not calling Quoter.__init__ to avoid its processing - should it??

    def __call__(self, value, **kwargs):
        """
        Quote the value, based on the instance's function.
        """
        opts = self.options.push(kwargs)
        pstr, mstr = self._whitespace(opts)
        prefix, value, suffix = opts.func(value)
        parts = [ mstr, prefix, pstr, stringify(value), pstr, suffix, mstr ]
        return self._output(parts, opts)

class HTMLQuoter(Quoter):
    """
    A more sophisticated quoter that supports attributes and void elements.
    """
    
    options = Quoter.options.add(
        tag      = None,
        atts     = {},
        attquote = single,
        void     = False,
        prefix   = Prohibited,
        suffix   = Prohibited,
    )
    
    def __init__(self, *args, **kwargs):
        """
        Create an HTMLQuoter
        """
        self.options = HTMLQuoter.options.push(kwargs)
        if args:
            self.options.addflat(args, ['tag', 'atts'])

        create_atts = self.options.atts
        self.options.tag, self.options.atts = self._parse_selector(self.options.tag)
        if create_atts:
            self.options.atts.update(create_atts)
        # explicit addition of atts
            
        # NB this is an explicit multi-setter
        
        if self.options.name:
            Quoter.styles[self.options.name] = self
            
    def _attstr(self, atts, opts):
        """
        Format an attribute dict.
        """
        return ' '.join([''] + ["{0}={1}".format(k, opts.attquote(v)) for k,v in atts.items()])
        
    def _parse_selector(self, selector):
        """
        Parse a selector (modeled on jQuery and CSS). Returns a (tagname, id, classes)
        tuple.
        """
        tagnames = re.findall(r'^(\w+)',      selector)
        classes  = re.findall(r'\.([\w\-]+)', selector)
        ids      = re.findall(r'\#([\w\-]+)', selector)
        assert len(tagnames) <= 1
        assert len(ids) <= 1
        atts = {}
        if ids:
            atts['id'] = ids[0]
        if classes:
            atts['class'] = ' '.join(classes)
        return (tagnames[0] if tagnames else None, atts)
    
    def __call__(self, *args, **kwargs):
        """
        Quote a value in X/HTML style, with optional attributes.
        """
        
        if 'atts' in kwargs:
            catts = kwargs['atts']
            del kwargs['atts']
        else:
            catts = {}
        
        value = ''
        if len(args) > 0:
            value = args[0]
        if len(args) > 1:                
            catts = args[1]
        if len(args) > 2:
            raise ValueError('just one or two args, please')
            
        opts = self.options.push(kwargs)
        pstr, mstr = self._whitespace(opts)

        # do we need some special processing to remove atts from the kwargs?
        # or some magic to integrate call atts to with existing atts?
        # or ..?
        # this is a good test / hard case for the magial processing
        # probably magic
        
        callatts = self._parse_selector(catts)[1] if is_string(catts) else catts
        atts = {}
        atts.update(opts.atts)
        atts.update(callatts)
        
        astr = self._attstr(atts, opts) if atts else ''
        if opts.void or not args:
            parts = [ mstr, '<', opts.tag, astr, '>', mstr ]
        else:
            parts = [ mstr, '<', opts.tag, astr, '>', pstr,
                      stringify(value),
                      pstr, '</', opts.tag, '>', mstr ] 
        return self._output(parts, opts)
        
    # could improve kwargs handling of HTMLQuoter
    
    # question is, should call attributes overwrite, or add to, object atts?
    # may not be a single answer - eg, in case of class especially
    
    # This might be case where replace is the primary option, but there's
    # an option to add (or even subtract) - say using a class Add, Plus, Subtract,
    # Minus, Relative, Rel, Delta, etc as an indicator
    
    
    

    
# need def_tag to define these

#comment = Quoter('<!--', '-->', name='comment')
#for t in """area base br col command embed hr img input keygen link meta
#            param source track wbr""".split():
#    def_tag(t, Quoter(t, name=t, void=True))
#for t in """a article aside audio b bdi bdo blockquote body button canvas
#            caption cite code colgroup datalist dd del details dfn div dl dt
#            em fieldset figcaption figure footer form h1 h2 h3 h4 h5 h6
#            head header hgroup html i iframe ins kbd label legend li
#            map mark menu meter nav noscript object ol optgroup option
#            output p pre progress q rp rt ruby s samp script section
#            select small span strong style sub summary sup table tbody
#            td textarea tfoot th thead time tile tr u ul var video""".split():
#    def_tag(t, Quoter(t, name=t))

# see also http://en.wikipedia.org/wiki/Quotation_mark_glyphs

def quote(value, style='double', **kwargs):
    try:
        quoter = Quoter.styles[style]
    except KeyError:
        raise ValueError("'{}' is not a known quote style".format(style))
    return quoter(value, **kwargs)
    # if the kwarg 'chars' is given, should that take precedence over style,
    # and/or be orthogonal choice vs style?
    
# should Quoter be memoized?
