#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2012 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
SMS Data Loader
"""

import datetime
import logging

from sqlalchemy.orm import joinedload_all

import edbob
from edbob.time import localize
from edbob.util import requires_impl

import rattail_locsms as locsms
from rattail.core import Object
from rattail.gpc import GPC
from rattail.db import Session
from rattail.db import model
from rattail.db import cache
from rattail import enum
from rattail.db.api import get_setting, save_setting


log = logging.getLogger(__name__)


class DataAlreadyLoadedError(Exception):

    def __str__(self):
        return "SMS data has already been loaded"


def restrict_by(name, query, column):
    values = edbob.config.get('rattail.sw.locsms', 'load.restrictions.%s' % name)
    if values:
        values = values.split(',')
        values = [x.strip() for x in values]
        query = query.filter(column.in_(values))
    return query


class LoadProcessor(Object):
    """
    Generic class for performing a full data load from SMS.
    """

    def load_all_data(self, progress=None):
        """
        Loads all relevant data from SMS into Rattail.
        """

        session = Session()
        if get_setting(session, 'rattail.sw.locsms.data_loaded'):
            session.close()
            raise DataAlreadyLoadedError

        sms_session = locsms.Session()

        cancel = False
        for loader in self.relevant_loaders():
            loader = loader(session=session, sms_session=sms_session)
            if not loader.perform_load(progress):
                cancel = True
                break

        if cancel:
            session.rollback()
        else:
            session.commit()

        save_setting(session, 'rattail.sw.locsms.data_loaded', 'True')
        session.close()
        sms_session.close()


    def relevant_loaders(self):
        """
        Generator which should yield all relevant data loader classes.
        Override this if you don't want all available data loaded.
        """

        yield StoreLoader
        yield DepartmentLoader
        yield SubdepartmentLoader
        yield VendorLoader
        yield ProductLoader
        yield CostLoader
        yield PriceLoader
        yield CustomerGroupLoader
        yield CustomerLoader
        yield OperatorLoader


class DataLoader(Object):
    """
    Generic class for performing a particular type of data load from SMS.
    """

    @property
    @requires_impl(is_property=True)
    def sms_class(self):
        pass

    @property
    def name(self):
        return self.sms_class.__name__

    def start(self):
        pass

    def get_sms_query(self):
        return self.sms_session.query(self.sms_class)

    def get_sms_iterable(self, sms_query):
        return sms_query

    def relevant_cachers(self):
        yield None

    def perform_cache(self, progress):
        for obj in self.relevant_cachers():
            if obj:
                cacher, attr = obj
                cacher = cacher(session=self.session)
                cache = cacher.get_cache(progress)
                if cache is None:
                    return False
                setattr(self, attr, cache)
        return True

    def perform_load(self, progress):
        """
        Performs the load processing for a particular data type.
        """

        self.start()

        if not self.perform_cache(progress):
            return False

        sms_query = self.get_sms_query()
        count = sms_query.count()
        if not count:
            return True

        prog = None
        if progress:
            prog = progress("Loading %s data from SMS" % self.name, count)

        cancel = False
        for i, sms_instance in enumerate(self.get_sms_iterable(sms_query), 1):
            self.load_sms_instance(sms_instance)
            self.session.flush()
            if prog and not prog.update(i):
                cancel = True
                break

        if prog:
            prog.destroy()
        return not cancel


class StoreLoader(DataLoader):
    """
    Loads :class:`rattail_locsms.Store` data into Rattail.
    """

    @property
    def sms_class(self):
        return locsms.Store

    def load_sms_instance(self, sms_store):
        store = model.Store(id=sms_store.F1056)
        store.name = sms_store.F1531

        if sms_store.F1536:
            store.add_phone_number(sms_store.F1536)

        if sms_store.F1537:
            store.add_phone_number(sms_store.F1537, type='Fax')

        self.session.add(store)


class DepartmentLoader(DataLoader):
    """
    Loads :class:`rattail_locsms.Department` data into Rattail.
    """

    @property
    def sms_class(self):
        return locsms.Department

    def load_sms_instance(self, sms_dept):
        dept = model.Department(number=sms_dept.F03)
        dept.name = sms_dept.F238
        self.session.add(dept)


class SubdepartmentLoader(DataLoader):
    """
    Loads :class:`rattail_locsms.Subdepartment` data into Rattail.
    """

    @property
    def sms_class(self):
        return locsms.Subdepartment

    def relevant_cachers(self):
        yield cache.DepartmentCacher, 'departments'

    def load_sms_instance(self, sms_subdept):

        dept = self.departments.get(sms_subdept.F03)
        if not dept:
            dept = model.Department(number=sms_subdept.F03)
            dept.name = "Department %u" % dept.number
            self.session.add(dept)
            self.departments[dept.number] = dept

        subdept = model.Subdepartment(number=sms_subdept.F04)
        subdept.name = sms_subdept.F1022
        subdept.department = dept
        self.session.add(subdept)


class VendorLoader(DataLoader):
    """
    Loads :class:`rattail_locsms.Vendor` data into Rattail.
    """

    @property
    def sms_class(self):
        return locsms.Vendor

    def load_sms_instance(self, sms_vendor):
        vendor = model.Vendor(id=sms_vendor.F27)
        vendor.name = sms_vendor.F334
        vendor.special_discount = sms_vendor.F347

        if sms_vendor.F341:
            vendor.add_phone_number(sms_vendor.F341)

        if sms_vendor.F1890:
            vendor.add_phone_number(sms_vendor.F1890)

        if sms_vendor.F342:
            vendor.add_phone_number(sms_vendor.F342, type='Fax')

        if sms_vendor.F1891:
            vendor.add_phone_number(sms_vendor.F1891, type='Fax')

        if sms_vendor.F335:
            person = model.Person(display_name=sms_vendor.F335)
            vendor.contacts.append(person)

        if sms_vendor.F1889:
            person = model.Person(display_name=sms_vendor.F1889)
            vendor.contacts.append(person)

        if sms_vendor.F2603:
            vendor.add_email_address(sms_vendor.F2603)

        self.session.add(vendor)


class ProductLoader(DataLoader):
    """
    Loads :class:`rattail_locsms.Product` data into Rattail.
    """

    @property
    def sms_class(self):
        return locsms.Product

    def relevant_cachers(self):
        yield cache.DepartmentCacher, 'departments'
        yield cache.SubdepartmentCacher, 'subdepartments'
        yield cache.BrandCacher, 'brands'

    def get_sms_query(self):
        q = super(ProductLoader, self).get_sms_query()
        q = restrict_by('F01', q, locsms.Product.F01)
        return q

    def get_sms_iterable(self, sms_query):
        return sms_query.options(joinedload_all(
                locsms.Product.pos, locsms.POS.subdepartment, locsms.Subdepartment.department))

    def load_sms_instance(self, sms_prod):
        upc = sms_prod.F01
        if not upc.isdigit():
            log.warning("ProductLoader: Invalid UPC: %s" % upc)
            return
        
        upc = GPC(upc, calc_check_digit='upc')

        dept = None
        if sms_prod.department:
            dept = self.departments.get(sms_prod.department.F03)
            if not dept:
                dept = model.Department()
                dept.number = sms_prod.department.F03
                dept.name = sms_prod.department.F238
                self.session.add(dept)
                self.departments[dept.number] = dept

        subdept = None
        if sms_prod.subdepartment:
            subdept = self.subdepartments.get(sms_prod.subdepartment.F04)
            if not subdept:
                subdept = model.Subdepartment()
                subdept.number = sms_prod.subdepartment.F03
                subdept.name = sms_prod.subdepartment.F1022
                self.session.add(subdept)
                self.subdepartments[subdept.number] = subdept

        brand = None
        if sms_prod.F155:
            brand = self.brands.get(sms_prod.F155)
            if not brand:
                brand = model.Brand(name=sms_prod.F155)
                self.session.add(brand)
                self.brands[brand.name] = brand

        prod = model.Product(upc=upc)
        prod.department = dept
        prod.subdepartment = subdept
        prod.brand = brand
        prod.description = sms_prod.F29
        prod.size = sms_prod.F22

        prod.unit_of_measure = enum.UNIT_OF_MEASURE_EACH
        if sms_prod.pos:
            if sms_prod.pos.F106 == '1':
                if sms_prod.subdepartment and sms_prod.subdepartment.F82 == '1':
                    prod.unit_of_measure = enum.UNIT_OF_MEASURE_POUND
            elif sms_prod.pos.F82 == '1':
                prod.unit_of_measure = enum.UNIT_OF_MEASURE_POUND

        self.session.add(prod)


class CostLoader(DataLoader):
    """
    Loads :class:`rattail_locsms.Cost` data into Rattail.
    """

    @property
    def sms_class(self):
        return locsms.Cost

    def relevant_cachers(self):
        yield cache.ProductCacher, 'products'
        yield cache.VendorCacher, 'vendors'

    def get_sms_query(self):
        q = super(CostLoader, self).get_sms_query()
        q = restrict_by('F01', q, locsms.Cost.F01)
        return q

    def get_sms_iterable(self, sms_query):
        return sms_query.order_by(locsms.Cost.F90.desc())

    def load_sms_instance(self, sms_cost):
        if sms_cost.F1000 != 'PAL':
            log.warning("CostLoader: Target (F1000) '%s' not supported: %s" %
                        (sms_cost.F1000, sms_cost.F01))
            return

        if sms_cost.F1184 != 'CASE':
            log.warning("CostLoader: Buying format (F1184) '%s' not supported: %s" %
                        (sms_cost.F1184, sms_cost.F01))
            return

        upc = sms_cost.F01
        if not upc.isdigit():
            log.warning("CostLoader: Invalid UPC: %s" % upc)
            return

        upc = GPC(upc, calc_check_digit='upc')
        product = self.products.get(upc)
        if not product:
            product = model.Product(upc=upc)
            self.session.add(product)
            self.products[product.upc] = product

        vendor = self.vendors.get(sms_cost.F27)
        if not vendor:
            vendor = model.Vendor(id=sms_cost.F27)
            self.session.add(vendor)
            self.vendors[vendor.id] = vendor

        cost = model.ProductCost()
        cost.vendor = vendor
        cost.code = sms_cost.F26
        cost.case_size = sms_cost.F19
        cost.case_cost = sms_cost.F38
        cost.unit_cost = sms_cost.F1140
        product.costs.append(cost)


class PriceLoader(DataLoader):
    """
    Loads :class:`rattail_locsms.Price` data into Rattail.
    """

    @property
    def sms_class(self):
        return locsms.Price

    def relevant_cachers(self):
        yield cache.ProductCacher, 'products'

    def start(self):
        self.now = edbob.utc_time(naive=True)

    def get_sms_query(self):
        q = super(PriceLoader, self).get_sms_query()
        q = restrict_by('F01', q, locsms.Price.F01)
        return q

    def utc_timestamp(self, date, time):
        if not date:
            return None
        if time:
            time = datetime.datetime.strptime(time, '%H%M').time()
        else:
            time = datetime.time(0) # assume midnight
        stamp = datetime.datetime.combine(date, time)
        return localize(stamp, from_='rattail.sw.locsms', to='utc', naive=True)

    def price_is_current(self, price):
        if price.starts and price.ends:
            return price.starts <= self.now <= price.ends
        if price.starts:
            return price.starts <= self.now
        if price.ends:
            return self.now <= price.ends
        return True

    def load_sms_instance(self, sms_price):
        if sms_price.F1000 != 'PAL':
            return

        upc = sms_price.F01
        if not upc.isdigit():
            log.warning("PriceLoader: Invalid UPC: %s" % upc)
            return

        upc = GPC(upc, calc_check_digit='upc')
        prod = self.products.get(upc)
        if not prod:
            prod = model.Product(upc=upc)
            self.session.add(prod)
            self.products[prod.upc] = prod

        # Regular Price
        if sms_price.F30 is not None or sms_price.F140 is not None:
            price = model.ProductPrice()
            price.type = enum.PRICE_TYPE_REGULAR
            price.level = sms_price.F126
            price.starts = self.utc_timestamp(sms_price.F35, sms_price.F36)
            price.ends = self.utc_timestamp(sms_price.F129, sms_price.F130)
            price.price = sms_price.F30
            price.multiple = sms_price.F31 or 1
            if sms_price.F140 is not None:
                price.pack_price = sms_price.F140
                price.pack_multiple = sms_price.F142 or 1
            prod.prices.append(price)
            if price.level == 1 and self.price_is_current(price):
                prod.regular_price = price

        # TPR
        tpr_price = None
        if sms_price.F181 is not None or sms_price.F1186 is not None:
            price = model.ProductPrice()
            price.type = enum.PRICE_TYPE_TPR
            price.level = sms_price.F126
            price.starts = self.utc_timestamp(sms_price.F183, None)
            price.ends = self.utc_timestamp(sms_price.F184, None)
            price.price = sms_price.F181
            price.multiple = sms_price.F182 or 1
            if sms_price.F1186 is not None:
                price.pack_price = sms_price.F1186
                price.pack_multiple = sms_price.F1187
            prod.prices.append(price)
            tpr_price = price

        # Sale
        sale_price = None
        if sms_price.F136 is not None or sms_price.F139 is not None:
            price = model.ProductPrice()
            price.type = enum.PRICE_TYPE_SALE
            price.level = sms_price.F126
            price.starts = self.utc_timestamp(sms_price.F137, sms_price.F144)
            price.ends = self.utc_timestamp(sms_price.F138, sms_price.F145)
            price.price = sms_price.F136
            price.multiple = sms_price.F135 or 1
            if sms_price.F139 is not None:
                price.pack_price = sms_price.F139
                price.pack_multiple = sms_price.F143
            prod.prices.append(price)
            sale_price = price

        # Assign "current" price (Sale/TPR/None).
        if sale_price:
            if self.price_is_current(sale_price):
                prod.current_price = sale_price
            elif tpr_price:
                if self.price_is_current(tpr_price):
                    prod.current_price = tpr_price
                else:
                    prod.current_price = None
            else:
                prod.current_price = None
        elif tpr_price:
            if self.price_is_current(tpr_price):
                prod.current_price = tpr_price
            else:
                prod.current_price = None
        else:
            prod.current_price = None


class CustomerGroupLoader(DataLoader):
    """
    Loads :class:`rattail_locsms.CustomerGroup` data into Rattail.
    """

    @property
    def sms_class(self):
        return locsms.CustomerGroup

    def load_sms_instance(self, sms_group):
        group = model.CustomerGroup()
        group.id = sms_group.F1154
        group.name = sms_group.F1268
        self.session.add(group)


class CustomerLoader(DataLoader):
    """
    Loads :class:`rattail_locsms.Customer` data into Rattail.
    """

    @property
    def sms_class(self):
        return locsms.Customer

    def relevant_cachers(self):
        yield cache.CustomerGroupCacher, 'groups'

    def get_sms_query(self):
        q = super(CustomerLoader, self).get_sms_query()
        q = restrict_by('F1148', q, locsms.Customer.F1148)
        return q

    def load_sms_instance(self, sms_customer):
        person = model.Person()
        person.first_name = sms_customer.F1149
        person.last_name = sms_customer.F1150

        customer = model.Customer(id=sms_customer.F1148)
        customer.name = sms_customer.F1155
        customer.people.append(person)

        if sms_customer.F1172:
            customer.add_phone_number(sms_customer.F1172)
        if sms_customer.F1173:
            customer.add_phone_number(sms_customer.F1173)
        if sms_customer.F1573:
            customer.add_email_address(sms_customer.F1573)

        if sms_customer.F1154:
            group = self.groups.get(sms_customer.F1154)
            if not group:
                group = model.CustomerGroup(id=sms_customer.F1154)
                self.groups[group.id] = group
            customer.groups.append(group)

        if sms_customer.F1777:
            group = self.groups.get(sms_customer.F1777)
            if not group:
                group = model.CustomerGroup(id=sms_customer.F1777)
                self.groups[group.id] = group
            customer.groups.append(group)

        self.session.add(customer)
        return customer


class OperatorLoader(DataLoader):
    """
    Loads :class:`rattail_locsms.Operator` data into Rattail.
    """

    @property
    def sms_class(self):
        return locsms.Operator

    def relevant_cachers(self):
        yield cache.CustomerCacher, 'customers'

    def load_sms_instance(self, sms_operator):

        customer = None
        if sms_operator.F1148:
            customer = self.customers.get(sms_operator.F1148)
            if not customer:
                customer = model.Customer(id=sms_operator.F1148)
                self.session.add(customer)
                self.customers[customer.id] = customer

        if customer and customer.person:
            person = customer.person
        else:
            person = model.Person()
            person.first_name = sms_operator.F1143
            person.last_name = sms_operator.F1144
        if customer and not customer.person:
            customer.people.append(person)

        employee = model.Employee(id=sms_operator.F1126)
        employee.person = person
        employee.display_name = sms_operator.F1127

        if sms_operator.F1563:
            employee.add_phone_number(sms_operator.F1563)
        if sms_operator.F1564:
            employee.add_phone_number(sms_operator.F1564)
        if sms_operator.F1571:
            employee.add_email_address(sms_operator.F1571)

        status = (sms_operator.F1552 or '').upper()
        if status == 'ACTV':
            employee.status = enum.EMPLOYEE_STATUS_CURRENT
        elif status == 'FRMR':
            employee.status = enum.EMPLOYEE_STATUS_FORMER

        self.session.add(employee)
