#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2012 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
SQLAlchemy Schema
"""

import datetime

from sqlalchemy import and_
from sqlalchemy.orm import relationship, object_session

from rattail_locsms.db import Base


__all__ = ['Store', 'Department', 'Subdepartment', 'POS', 'Price',
           'Product', 'AlternateCode', 'Operator', 'Vendor', 'Cost', 'PurchasingDocument',
           'PurchasingDetail', 'CustomerGroup', 'Customer', 'AccountLink',
           'SaleTransaction', 'SaleDetail']


class Store(Base):
    """
    Represents a store (STD_TAB).
    """

    __tablename__ = 'STD_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "Store(F1056={0})".format(repr(self.F1056))

    def __unicode__(self):
        return unicode(self.F1531)


class Subdepartment(Base):
    """
    Represents a subdepartment (SDP_TAB).
    """

    __tablename__ = 'SDP_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "Subdepartment(F04={0})".format(repr(self.F04))

    def __unicode__(self):
        return unicode(self.F1022 or '')


class Department(Base):
    """
    Represents a department (DEPT_TAB).
    """

    __tablename__ = 'DEPT_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "Department(F03={0})".format(repr(self.F03))

    def __unicode__(self):
        return unicode(self.F238)

Department.subdepartments = relationship(
    Subdepartment,
    primaryjoin=Subdepartment.F03 == Department.F03,
    foreign_keys=[Subdepartment.F03],
    backref='department')


class POS(Base):
    """
    Represents a POS record for an item (POS_TAB).
    """

    __tablename__ = 'POS_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "POS(F01={0}, F1000={1})".format(
            repr(self.F01), repr(self.F1000))

POS.subdepartment = relationship(
    Subdepartment,
    primaryjoin=Subdepartment.F04 == POS.F04,
    foreign_keys=[Subdepartment.F04],
    uselist=False)


class Vendor(Base):
    """
    Represents a vendor (VENDOR_TAB).
    """

    __tablename__ = 'VENDOR_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "Vendor(F27={0})".format(repr(self.F27))

    def __unicode__(self):
        return unicode(self.F334 or '')


class AlternateCode(Base):
    """
    Represents an alternate code for a product (ALT_TAB).
    """

    __tablename__ = 'ALT_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "AlternateCode(F154={0}, F1000={1})".format(
            repr(self.F154), repr(self.F1000))

    def __unicode__(self):
        return unicode(self.F154 or '')


class Cost(Base):
    """
    Represents a cost record for a product (COST_TAB).
    """

    __tablename__ = 'COST_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "Cost(F01={0}, F27={1}, F1184={2}, F1000={3})".format(
            repr(self.F01), repr(self.F27), repr(self.F1184), repr(self.F1000))

Cost.vendor = relationship(
    Vendor,
    primaryjoin=Vendor.F27 == Cost.F27,
    foreign_keys=[Vendor.F27],
    uselist=False)


class Price(Base):
    """
    Represents a price (PRICE_TAB).
    """

    __tablename__ = 'PRICE_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "Price(F01={0}, F126={1}, F1000={2})".format(
            repr(self.F01), repr(self.F126), repr(self.F1000))

    def get_datetime(self, date, time):
        """
        Convert a date and time field pair to a standard ``datetime`` instance.

        :param date: An internal date field value.
        :param time: An internal time field value.
        :rtype: ``datetime.datetime`` instance, or ``None``
        """
        if date is None:
            return None
        if time is None:
            time = datetime.time(0) # assume midnight
        else:
            time = datetime.datetime.strptime(time, '%H%M').time()
        return datetime.datetime.combine(date, time)

    @property
    def regular_start(self):
        """
        The regular start time as a standard ``datetime`` instance, or ``None``.
        """
        return self.get_datetime(self.F35, self.F36)

    @property
    def regular_end(self):
        """
        The regular end time as a standard ``datetime`` instance, or ``None``.
        """
        return self.get_datetime(self.F129, self.F130)

    @property
    def tpr_start(self):
        """
        The TPR start time as a standard ``datetime`` instance, or ``None``.
        """
        return self.get_datetime(self.F183, None)

    @property
    def tpr_end(self):
        """
        The TPR end time as a standard ``datetime`` instance, or ``None``.
        """
        return self.get_datetime(self.F184, None)

    @property
    def sale_start(self):
        """
        The sale start time as a standard ``datetime`` instance, or ``None``.
        """
        return self.get_datetime(self.F137, self.F144)

    @property
    def sale_end(self):
        """
        The sale end time as a standard ``datetime`` instance, or ``None``.
        """
        return self.get_datetime(self.F138, self.F145)

    def guess_active_price(self, now=None):
        """
        Calculates what "ought" to be the current active price value.

        This method was added in order to provide a means for double-checking
        the active price (F1007) value within SMS.  Under normal circumstances
        that field value should be preferred to whatever this returns.

        .. warning::
           In particular, the current implementation here is *very* basic, and
           does not even inspect the TPR or Store price values.  It is
           therefore only useful for determining if the Sale or Regular price
           should be considered active.
        """

        if now is None:
            now = datetime.datetime.now()

        # Sale
        start = self.sale_start
        end = self.sale_end
        if start and end and start <= now <= end:
            return self.F136

        # Regular
        return self.F30


class Product(Base):
    """
    Represents a product / main item record (OBJ_TAB).
    """

    __tablename__ = 'OBJ_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "Product(F01={0})".format(repr(self.F01))

    def __unicode__(self):
        return unicode(self.F02 or '')

    @property
    def subdepartment(self):
        if self.pos:
            return self.pos.subdepartment
        return None

    @property
    def department(self):
        if self.subdepartment:
            return self.subdepartment.department
        return None

    @property
    def cost(self):
        """
        The "authorized" cost for the product.

        This uses the same logic as SMS in determining which cost record is
        considered authorized: namely, the cost with ``F90`` set to either
        ``'1'`` or ``NULL``.

        However, as a convenience, it also considers a cost record as
        authorized if the product only *has* one cost record, regardless of its
        ``F90`` value.

        If no cost records exist for the product, or if multiple records exist
        and all are explicitly marked *unauthorized* (``F90 == '0'``), then
        ``None`` is returned.

        :returns: A :class:`Cost` instance, or ``None``
        """
        
        for cost in self.costs:
            if cost.F90 == '1' or cost.F90 is None:
                return cost
        if len(self.costs) == 1:
            return self.costs[0]
        return None
        
    def last_purchase_base_cost(self, vendor_id=None):
        """
        Returns the "last" ``F38`` value found in the ``REC_REG`` table for
        this product.  If ``vendor_id`` is specified, then the search will
        be restricted to purchases for that vendor only.
        """

        session = object_session(self)
        q = session.query(PurchasingDetail)
        q = q.join((PurchasingDocument,
                    PurchasingDocument.F1032 == PurchasingDetail.F1032))
        q = q.filter(PurchasingDocument.F1068.in_(('ORDER','RECV','BILL','DSD')))
        if vendor_id:
            q = q.filter(PurchasingDocument.F27 == vendor_id)
        q = q.filter(PurchasingDetail.F01 == self.F01)
        q = q.filter(PurchasingDetail.F38 != None)
        q = q.order_by(PurchasingDetail.F76.desc())
        detail = q.first()
        if detail:
            return detail.F38
        return None        

Product.alt_codes = relationship(
    AlternateCode,
    primaryjoin=AlternateCode.F01 == Product.F01,
    foreign_keys=[AlternateCode.F01],
    backref='product')

Product.main_alt_code = relationship(
    AlternateCode,
    primaryjoin=and_(
        AlternateCode.F01 == Product.F01,
        AlternateCode.F1000 == 'PAL',
        AlternateCode.F1898 == '1',
        ),
    foreign_keys=[AlternateCode.F01],
    uselist=False,
    viewonly=True)

Product.pos = relationship(
    POS,
    primaryjoin=and_(
        POS.F01 == Product.F01,
        POS.F1000 == 'PAL',
        ),
    foreign_keys=[POS.F01],
    uselist=False)

Product.costs = relationship(
    Cost,
    primaryjoin=Cost.F01 == Product.F01,
    foreign_keys=[Cost.F01],
    backref='product')

Product.prices = relationship(
    Price,
    primaryjoin=Price.F01 == Product.F01,
    foreign_keys=[Price.F01],
    backref='product')

Product.price = relationship(
    Price,
    primaryjoin=and_(
        Price.F01 == Product.F01,
        Price.F1000 == 'PAL',
        Price.F126 == 1,
        ),
    foreign_keys=[Price.F01],
    uselist=False)


class Operator(Base):
    """
    Represents an operator (user) within the system (CLK_TAB).
    """

    __tablename__ = 'CLK_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "Operator(F1185={0})".format(repr(self.F1185))

    def __unicode__(self):
        return unicode(self.F1127 or '')


class PurchasingDocument(Base):
    """
    Represents a purchasing document (REC_HDR).
    """

    __tablename__ = 'REC_HDR'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "PurchasingDocument(F1032={0})".format(repr(self.F1032))

PurchasingDocument.vendor = relationship(
    Vendor,
    primaryjoin=Vendor.F27 == PurchasingDocument.F27,
    foreign_keys=[Vendor.F27],
    uselist=False)

PurchasingDocument.order_entered_by = relationship(
    Operator,
    primaryjoin=Operator.F1126 == PurchasingDocument.F1693,
    foreign_keys=[Operator.F1126],
    uselist=False)


class PurchasingDetail(Base):
    """
    Represents a detail line within a purchasing document (REC_REG).
    """

    __tablename__ = 'REC_REG'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "PurchasingDetail(F1032={0}, F1101={1})".format(
            repr(self.F1032), repr(self.F1101))


class CustomerGroup(Base):
    """
    Represents a customer group (CLG_TAB).
    """

    __tablename__ = 'CLG_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "CustomerGroup(F1154={0})".format(repr(self.F1154))
    
    def __unicode__(self):
        return unicode(self.F1268 or '')


class Customer(Base):
    """
    Represents a customer (CLT_TAB).
    """

    __tablename__ = 'CLT_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "Customer(F1148={0})".format(repr(self.F1148))
    
    def __unicode__(self):
        first_name = self.F1149
        last_name = self.F1150
        if first_name and last_name:
            return unicode(first_name + ' ' + last_name)
        if first_name:
            return unicode(first_name)
        if last_name:
            return unicode(last_name)
        return u''


class AccountLink(Base):
    """
    Represents an account link (CLL_TAB).
    """

    __tablename__ = 'CLL_TAB'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "AccountLink(F1577={0}, F1578={1})".format(
            repr(self.F1577), repr(self.F1578))


class SaleTransaction(Base):
    """
    Represents a sale transaction (SAL_HDR).
    """

    __tablename__ = 'SAL_HDR'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "SaleTransaction(F1032={0}, F1149={1}, F1150={2})".format(
            repr(self.F1032), repr(self.F1149), repr(self.F1150))

SaleTransaction.operator = relationship(
    Operator,
    primaryjoin=Operator.F1126 == SaleTransaction.F1126,
    foreign_keys=[Operator.F1126],
    uselist=False)

SaleTransaction.customer = relationship(
    Customer,
    primaryjoin=Customer.F1148 == SaleTransaction.F1148,
    foreign_keys=[Customer.F1148],
    uselist=False)


class SaleDetail(Base):
    """
    Represents a detail line within a sale transaction (SAL_REG).
    """

    __tablename__ = 'SAL_REG'
    __table_args__ = ({'autoload': True},)

    def __repr__(self):
        return "SaleDetail(F1032={0}, F1101={1}, F1041={2})".format(
            repr(self.F1032), repr(self.F1101), repr(self.F1041))

SaleTransaction.details = relationship(
    SaleDetail,
    primaryjoin=SaleDetail.F1032 == SaleTransaction.F1032,
    foreign_keys=[SaleDetail.F1032],
    order_by=SaleDetail.F1101,
    backref='transaction')
