#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2012 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Deployment Interface
"""

import os.path
import sys
import shutil
import ConfigParser

if sys.platform == 'win32': # let the docs flow
    import win32api
    import win32con

import edbob

from rattail.core import Object


class Node(Object):
    """
    Represents a node within an SMS system.
    """

    store = None
    terminal = None
    path = None

    def __repr__(self):
        return "<Node: %s>" % self.name

    def __unicode__(self):
        return unicode(self.name)

    @property
    def name(self):
        """
        Returns the node's "name" in ``STORE:TER`` format.
        """

        return '%s:%s' % (self.store, self.terminal)

    def deploy_to_inbox(self, path):
        """
        Copies file from ``path`` to the node's inbox folder, and turns off the
        new file's archive bit.
        """

        dest = self.get_inbox_path(os.path.basename(path))
        shutil.copy(path, dest)
        flip_archive_bit(dest)

    def get_inbox_path(self, filename=None):
        """
        Returns the full path to the node's "inbox" folder.

        If ``filename`` is specified, the path returned will include it.
        """

        path = os.path.join(self.path, 'Office', 'XF%s%s' % (
                self.store, self.terminal))
        if filename:
            path = os.path.join(path, filename)
        return path

    def get_load_path(self, filename=None):
        """
        Returns the full path to the node's "load" folder.

        If ``filename`` is specified, the returned path will include it.
        """

        path = os.path.join(self.path, 'Office', 'Load')
        if filename:
            path = os.path.join(path, filename)
        return path

    def read_config(self):
        assert self.path

        ini_path = os.path.join(self.path, 'SMSStart.ini')
        ini_file = open(ini_path)
        config = ConfigParser.SafeConfigParser()
        config.readfp(ini_file)
        ini_file.close()

        self.store = config.get('SMSSTART', 'STORE')
        self.terminal = config.get('SMSSTART', 'TER')


class Environment(Object):
    """
    Represents a SMS "environment" corresponding to an arbitrary system, be it
    development, testing, production or otherwise.  This is meant to reflect
    real-world use cases and system administration workflows, and is used to
    organize deployment configuration etc.
    """

    uuid = None
    name = ''
    nodes = []

    def __init__(self, uuid=None, **kwargs):
        self.uuid = uuid
        kwargs.setdefault('nodes', list(self.nodes))
        Object.__init__(self, **kwargs)

    def __repr__(self):
        return "<Environment: %s>" % self.name

    @property
    def sorted_nodes(self):
        # TODO: Implement custom sort.
        return self.nodes


class HotfixEnvironment(Environment):
    """
    Contains settings for deploying a hotfix(es), specific to an environment.
    """

    uuid = None
    target_nodes = []

    def __init__(self, *args, **kwargs):
        kwargs.setdefault('target_nodes', list(self.target_nodes))
        # Object.__init__(self, **kwargs)
        super(HotfixEnvironment, self).__init__(*args, **kwargs)

    def __repr__(self):
        return "<HotfixEnvironment: %s>" % self.name


class HotfixFile(Object):
    """
    Represents a local file intended to be deployed as part of a hotfix.
    """

    path = ''
    target_dir = None

    def __repr__(self):
        return "<HotfixFile: %s: %s>" % (self.filename, self.target_dir)

    @property
    def filename(self):
        """
        The basename of the local file.
        """
        return os.path.basename(self.path)


class SamplesEnvironment(Environment):
    """
    Contains settings for deploying samples, specific to an environment.
    """

    uuid = None
    source_path = ''
    mako_vars = {}
    use_mercurial_times = True
    target_nodes = []

    def __init__(self, *args, **kwargs):
        kwargs.setdefault('mako_vars', dict(self.mako_vars))
        kwargs.setdefault('target_nodes', list(self.target_nodes))
        super(SamplesEnvironment, self).__init__(*args, **kwargs)

    def __repr__(self):
        return "<SamplesEnvironment: %s>" % self.name


def get_node(key=None):
    """
    .. highlight:: ini

    Constructs a :class:`Node` instance and returns it.  The path to the node's
    Storeman folder is required and must be present in config.  ``key`` is used
    to distinguish in case you have more than one node configured::

       [rattail.sw.locsms]
       node = C:\Storeman
       node.testlane = \\TESTLANE\Storeman

    .. highlight:: python

    Those two nodes could be retrieved with the following calls::

       node = get_node()
       node = get_node('testlane')
    """

    if key:
        key = 'node.%s' % key
    else:
        key = 'node'
    path = edbob.config.require('rattail.sw.locsms', key)

    node = Node(path=path)
    node.read_config()
    return node


def get_inbox_path(node, path=None):
    """
    Returns the full path to the "inbox" folder for the given
    :class:`rattail_locsms.deploy.Node` instance.  ``path`` is accepted for
    compatibility's sake but is ignored.
    """

    return os.path.join(node.path, 'Office', 'XF%s%s' % (node.store, node.terminal))


def get_root_path(node, path):
    """
    Returns a fully-qualified version of ``path``, relative to the root of the
    given :class:`rattail_locsms.deploy.Node` instance.
    """

    return os.path.join(node.path, path)


def get_office_path(node, path):
    """
    Returns a fully-qualified version of ``path``, relative to the given
    :class:`rattail_locsms.deploy.Node`'s ``Office`` folder.
    """

    return os.path.join(node.path, 'Office', path)


def get_target_path(node, target_dir):
    """
    Returns an appropriate fully-qualified path according to the mappings found
    in ``TARGET_DIRS``.
    """

    return TARGET_DIRS[target_dir](node, target_dir)


TARGET_DIRS = {
    '<Inbox>':  get_inbox_path,
    'Bitmaps':  get_root_path,
    'Cgi':      get_office_path,
    'Htm':      get_office_path,
    'Images':   get_root_path,
    'Lbz':      get_office_path,
    'Load':     get_office_path,
    'Office':   get_root_path,
    'Public':   get_office_path,
    'Script':   get_office_path,
    'Sqr':      get_office_path,
    'Ssm':      get_office_path,
    'XchDev':   get_root_path,
    'XchFile':  get_root_path,
    'XchFtp':   get_root_path,
    'XchPay':   get_root_path,
    }


def flip_archive_bit(path):
    """
    Turns off the "File is ready for archiving" bit on file specified by
    ``path``.
    """

    attrs = win32api.GetFileAttributes(path)
    win32api.SetFileAttributes(path, attrs &~ win32con.FILE_ATTRIBUTE_ARCHIVE)


def deploy_to_inbox(path, node=None):
    """
    Deploys a file (``path``) to a SMS node's inbox folder.

    This function mainly exists to allow for abstracting the process somewhat,
    since on Linux there is no way to directly flip the file's archive bit and
    therefore the file must instead be deployed to a "pseudo" inbox folder
    which is monitored by a service running under Windows somewhere, and is in
    turn responsible for final (true) deployment.
    """

    option = 'pseudo_inbox'
    if node:
        option += '.' + node
    pseudo_inbox = edbob.config.get('rattail.sw.locsms', option)

    if pseudo_inbox:
        dest = os.path.join(pseudo_inbox, os.path.basename(path))
        shutil.copy(path, dest)
    else:
        node = get_node(node)
        node.deploy_to_inbox(path)
