#!/usr/bin/env python
# -*- coding: utf-8 -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2012 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Processing Deployed Changes
"""

import os
import os.path
import re
import csv
import logging
import datetime
from decimal import Decimal

import edbob
from edbob.time import localize

from rattail.core import Object
from rattail.gpc import GPC
from rattail.db import Session
from rattail.db import model
from rattail.db.api.products import get_product_by_upc
from rattail import enum
from rattail.util import load_object
from rattail.files import overwriting_move


change_pattern_generic = re.compile(r'^(\d{8})_(.+)$')
change_pattern = re.compile(r'^\d{8}_(.+)\.csv$', re.IGNORECASE)

log = logging.getLogger(__name__)


class ChangesProcessor(Object):
    """
    Base class and default implementation for processing changes which have
    been deployed from SMS.
    """

    def process_all_changes(self, changes):
        session = Session()

        log.debug("processing changes: {0}".format(repr(changes.keys())))

        for change_name, func in self.relevant_changes():
            log.debug("looking for change file: {0}".format(repr(change_name)))

            if change_name in changes:
                csv_path = changes.pop(change_name)
                log.debug("processing change file: {0}".format(repr(csv_path)))

                csv_file = open(csv_path, 'rb')
                reader = csv.DictReader(csv_file, quotechar="'")
                for row in reader:
                    func(row, session)
                    session.flush()
                csv_file.close()

            else:
                log.debug("no change file found: {0}".format(repr(change_name)))

        for csv_path in changes.itervalues():
            log.warning("ignoring change file: {0}".format(repr(csv_path)))

        session.commit()
        session.close()

    def relevant_changes(self):

        # Process deletions.
        yield 'CLK_DEL', self.process_clk_del
        yield 'CLT_DEL', self.process_clt_del
        yield 'COST_DEL', self.process_cost_del
        yield 'PRICE_DEL', self.process_price_del
        yield 'ALT_DEL', self.process_alt_del
        yield 'POS_DEL', self.process_pos_del
        yield 'OBJ_DEL', self.process_obj_del
        yield 'VENDOR_DEL', self.process_vendor_del

        # Process creations and updates.
        yield 'VENDOR_CHG', self.process_vendor_chg
        yield 'OBJ_CHG', self.process_obj_chg
        yield 'POS_CHG', self.process_pos_chg
        yield 'ALT_CHG', self.process_alt_chg
        yield 'PRICE_CHG', self.process_price_chg
        yield 'COST_CHG', self.process_cost_chg
        yield 'CLT_CHG', self.process_clt_chg
        yield 'CLK_CHG', self.process_clk_chg

    def date(self, val):
        return datetime.datetime.strptime(val, '%m/%d/%Y').date() if val else None

    def time(self, val):
        return datetime.datetime.strptime(val, '%H:%M').time() if val else None

    def decimal(self, val):
        return Decimal(val) if val else None

    def int_(self, val):
        return int(val) if val else None

    def utc_timestamp(self, date, time):
        if not date:
            return None
        if not time:
            time = datetime.time(0) # assume midnight
        stamp = datetime.datetime.combine(date, time)
        return localize(stamp, from_='rattail.sw.locsms', to='utc', naive=True)

    def get_product(self, upc, session):
        upc = GPC(upc, calc_check_digit='upc')
        q = session.query(model.Product)
        q = q.filter(model.Product.upc == upc)
        prod = q.first()
        if not prod:
            prod = model.Product(upc=upc)
            session.add(prod)
        return prod

    def get_price_object(self, row):

        p = Object()
        p.F126 = self.int_(row['F126'])

        # Regular Price
        p.F35 = self.date(row['F35'])
        p.F36 = self.time(row['F36'])
        p.F129 = self.date(row['F129'])
        p.F130 = self.time(row['F130'])
        p.F30 = self.decimal(row['F30'])
        p.F31 = self.int_(row['F31'])
        p.F140 = self.decimal(row['F140'])
        p.F142 = self.int_(row['F142'])
        p.F63 = self.decimal(row['F63'])
        p.F62 = self.int_(row['F62'])
        p.F111 = self.decimal(row['F111'])
        p.F112 = self.decimal(row['F112'])

        # TPR
        p.F183 = self.date(row['F183'])
        p.F184 = self.date(row['F184'])
        p.F181 = self.decimal(row['F181'])
        p.F182 = self.int_(row['F182'])
        p.F1186 = self.decimal(row['F1186'])
        p.F1187 = self.int_(row['F1187'])
        p.F1188 = self.decimal(row['F1188'])
        p.F1189 = self.int_(row['F1189'])
        p.F1218 = self.decimal(row['F1218'])
        p.F1219 = self.decimal(row['F1219'])

        # Sale
        p.F137 = self.date(row['F137'])
        p.F144 = self.time(row['F144'])
        p.F138 = self.date(row['F138'])
        p.F145 = self.time(row['F145'])
        p.F136 = self.decimal(row['F136'])
        p.F135 = self.int_(row['F135'])
        p.F139 = self.decimal(row['F139'])
        p.F143 = self.int_(row['F143'])
        p.F148 = self.decimal(row['F148'])
        p.F147 = self.int_(row['F147'])
        p.F1220 = self.decimal(row['F1220'])
        p.F1221 = self.decimal(row['F1221'])
        
        return p

    def price_is_current(self, price):
        now = edbob.utc_time(naive=True)
        if price.starts and price.ends:
            return price.starts <= now <= price.ends
        if price.starts:
            return price.starts <= now
        if price.ends:
            return now <= price.ends
        return True

    def copy_reg_price(self, sms_price, price):
        price.ends = self.utc_timestamp(sms_price.F129, sms_price.F130)
        price.price = sms_price.F30
        price.multiple = sms_price.F31 or 1
        if sms_price.F140:
            price.pack_price = sms_price.F140
            price.pack_multiple = sms_price.F142 or 1
        else:
            price.pack_price = None
            price.pack_multiple = None

    def copy_tpr_price(self, sms_price, price):
        price.ends = self.utc_timestamp(sms_price.F184, None)
        price.price = sms_price.F181
        price.multiple = sms_price.F182 or 1
        if sms_price.F1186:
            price.pack_price = sms_price.F1186
            price.pack_multiple = sms_price.F1187 or 1
        else:
            price.pack_price = None
            price.pack_multiple = None

    def copy_sale_price(self, sms_price, price):
        price.ends = self.utc_timestamp(sms_price.F138, sms_price.F145)
        price.price = sms_price.F136
        price.multiple = sms_price.F135 or 1
        if sms_price.F139:
            price.pack_price = sms_price.F139
            price.pack_multiple = sms_price.F143 or 1
        else:
            price.pack_price = None
            price.pack_multiple = None

        # Apply discount if one is present.
        if sms_price.F1221:
            if price.price:
                price.price = (price.price * (100 - sms_price.F1221) / 100)\
                    .quantize(Decimal('0.01'))
            if price.pack_price:
                price.pack_price = (price.pack_price * (100 - sms_price.F1221) / 100)\
                    .quantize(Decimal('0.01'))

    def process_vendor_chg(self, row, session):
        q = session.query(model.Vendor)
        q = q.filter(model.Vendor.id == row['F27'])
        vendor = q.first()
        if vendor:
            while vendor.emails:
                vendor.emails.pop()
            while vendor.phones:
                vendor.phones.pop()
        else:
            vendor = model.Vendor(id=row['F27'])
            session.add(vendor)

        vendor.name = row['F334']
        vendor.special_discount = self.int_(row['F347'])

        if row['F341']:
            vendor.add_phone_number(row['F341'])
        if row['F1890']:
            vendor.add_phone_number(row['F1890'])
        if row['F342']:
            vendor.add_phone_number(row['F342'], type='Fax')
        if row['F1891']:
            vendor.add_phone_number(row['F1891'], type='Fax')

        def get_contact(name):
            for contact in vendor._contacts:
                if contact.person.display_name == name:
                    return contact
            return None

        def set_contact(contact, preference):
            if contact.preference > preference:
                contact = vendor._contacts.pop(contact.preference - 1)
                vendor._contacts.insert(preference - 1, contact)

        next_preference = 1

        if row['F335']:
            contact = get_contact(row['F335'])
            if not contact:
                person = model.Person(display_name=row['F335'])
                contact = model.VendorContact(person=person)
                vendor._contacts.append(contact)
            set_contact(contact, next_preference)
            next_preference += 1

        if row['F1889']:
            contact = get_contact(row['F1889'])
            if not contact:
                person = model.Person(display_name=row['F1889'])
                contact = model.VendorContact(person=person)
                vendor._contacts.append(contact)
            set_contact(contact, next_preference)
            next_preference += 1

        while len(vendor._contacts) > next_preference - 1:
            contact = vendor._contacts.pop()
            session.delete(contact)

        return vendor

    def process_vendor_del(self, row, session):
        q = session.query(model.Vendor)
        q = q.filter(model.Vendor.id == row['F27'])
        vendor = q.first()
        if not vendor:
            return

        # Remove associated ProductCost records.
        q = session.query(model.ProductCost)
        q = q.filter(model.ProductCost.vendor == vendor)
        q.delete(synchronize_session=False)

        session.delete(vendor)

    def process_obj_chg(self, row, session):
        upc = row['F01']
        if not upc.isdigit():
            log.warning("invalid upc: {0}".format(repr(upc)))
            return

        product = self.get_product(upc, session)
        product.description = row['F29']
        product.size = row['F22']

        brand = None
        if row['F155']:
            q = session.query(model.Brand)
            q = q.filter(model.Brand.name == row['F155'])
            brand = q.first()
            if not brand:
                brand = model.Brand(name=row['F155'])
                session.add(brand)
        product.brand = brand

        family = None
        if row['F16']:
            family = session.query(model.Family)\
                .filter(model.Family.code == row['F16'])\
                .first()
            if not family:
                family = model.Family(code=row['F16'])
                session.add(family)
        product.family = family

        return product

    def process_obj_del(self, row, session):
        upc = row['F01']
        if not upc.isdigit():
            log.warning("invalid upc: {0}".format(repr(upc)))
            return

        upc = GPC(upc, calc_check_digit='upc')

        q = session.query(model.Product)
        q = q.filter(model.Product.upc == upc)
        product = q.first()
        if not product:
            return

        # Remove associated ProductPrice records.
        product.regular_price = None
        product.current_price = None
        q = session.query(model.ProductPrice)
        q = q.filter(model.ProductPrice.product == product)
        q.delete(synchronize_session=False)

        # Remove associated ProductCost records.
        q = session.query(model.ProductCost)
        q = q.filter(model.ProductCost.product == product)
        q.delete(synchronize_session=False)

        session.delete(product)

    def process_pos_chg(self, row, session):
        upc = row['F01']
        if not upc.isdigit():
            log.warning("non-numeric upc not supported: {0}".format(repr(upc)))
            return

        product = self.get_product(upc, session)

        subdepartment = None
        number = self.int_(row['F04'])
        if number:
            q = session.query(model.Subdepartment)
            q = q.filter(model.Subdepartment.number == number)
            subdepartment = q.first()
            if not subdepartment:
                subdepartment = model.Subdepartment(number=number)
                session.add(subdepartment)

        product.subdepartment = subdepartment
        if subdepartment:
            product.department = subdepartment.department
        else:
            product.department = None

        return product

    def process_pos_del(self, row, session):
        upc = row['F01']
        if not upc.isdigit():
            log.warning("invalid upc: {0}".format(repr(upc)))
            return

        upc = GPC(upc, calc_check_digit='upc')

        if row['F1000'] != 'PAL':
            log.warning("target {0} not supported for upc {1}".format(
                    repr(row['F1000']), repr(upc)))
            return

        q = session.query(model.Product)
        q = q.filter(model.Product.upc == upc)
        product = q.first()
        if not product:
            return

        product.subdepartment = None

    def process_alt_chg(self, row, session):
        if row['F1000'] != 'PAL':
            log.debug("target {0} not supported for upc {1}".format(
                    repr(row['F1000']), repr(row['F01'])))
            return

        upc = row['F01']
        if not upc.isdigit():
            log.warning("invalid upc: {0}".format(repr(upc)))
            return

        product = self.get_product(upc, session)
        if not product:
            log.warning("product not found: {0}".format(repr(upc)))
            return

        sms_code = row['F154']
        is_main = row['F1898'] == '1'
        if sms_code not in product.codes:
            if is_main:
                product.codes.insert(0, sms_code)
            else:
                product.codes.append(sms_code)
        elif is_main and product.code != sms_code:
            for i, code in enumerate(product.codes):
                if code == sms_code:
                    code = product.codes.pop(i)
                    product.codes.insert(0, code)
                    break

        return product

    def process_alt_del(self, row, session):
        if row['F1000'] != 'PAL':
            log.debug("target {0} not supported for upc {1}".format(
                    repr(row['F1000']), repr(row['F01'])))
            return

        upc = row['F01']
        if not upc.isdigit():
            log.warning("invalid upc: {0}".format(repr(upc)))
            return

        product = self.get_product(upc, session)
        if not product:
            log.warning("product not found: {0}".format(repr(upc)))
            return

        code = row['F154']
        if code in product.codes:
            product.codes.remove(code)

    def process_price_chg(self, row, session):
        """
        Process a price change.
        """
        upc = row[u'F01']
        if not upc.isdigit():
            log.warning(u"invalid upc: {0}".format(repr(upc)))
            return

        if row[u'F1000'] != u'PAL':
            log.debug(u"ignoring price change for target {0} for upc: {1}".format(
                    repr(row[u'F1000']), repr(row[u'F01'])))
            return

        product = self.get_product(upc, session)
        sms_price = self.get_price_object(row)
        prices = session.query(model.ProductPrice)

        # Create / update Regular price.
        if sms_price.F30 or sms_price.F140:
            starts = self.utc_timestamp(sms_price.F35, sms_price.F36)
            q = prices.filter(model.ProductPrice.product == product)
            q = q.filter(model.ProductPrice.type == enum.PRICE_TYPE_REGULAR)
            q = q.filter(model.ProductPrice.level == sms_price.F126)
            q = q.filter(model.ProductPrice.starts == starts)
            price = q.first()
            if not price:
                price = model.ProductPrice()
                price.type = enum.PRICE_TYPE_REGULAR
                price.level = sms_price.F126
                price.starts = starts
                product.prices.append(price)
            self.copy_reg_price(sms_price, price)
            if price.level == 1:
                product.regular_price = price if self.price_is_current(price) else None

        # Create / update TPR price.
        tpr_price = None
        if sms_price.F181 or sms_price.F1186:
            starts = self.utc_timestamp(sms_price.F183, None)
            q = prices.filter(model.ProductPrice.product == product)
            q = q.filter(model.ProductPrice.type == enum.PRICE_TYPE_TPR)
            q = q.filter(model.ProductPrice.level == sms_price.F126)
            q = q.filter(model.ProductPrice.starts == starts)
            price = q.first()
            if not price:
                price = model.ProductPrice()
                price.type = enum.PRICE_TYPE_TPR
                price.level = sms_price.F126
                price.starts = starts
                product.prices.append(price)
            self.copy_tpr_price(sms_price, price)
            tpr_price = price

        # Create / update Sale price.
        sale_price = None
        sale_is_current = False
        if sms_price.F136 or sms_price.F139:
            starts = self.utc_timestamp(sms_price.F137, sms_price.F144)
            q = prices.filter(model.ProductPrice.product == product)
            q = q.filter(model.ProductPrice.type == enum.PRICE_TYPE_SALE)
            q = q.filter(model.ProductPrice.level == sms_price.F126)
            q = q.filter(model.ProductPrice.starts == starts)
            price = q.first()
            if not price:
                price = model.ProductPrice()
                price.type = enum.PRICE_TYPE_SALE
                price.level = sms_price.F126
                price.starts = starts
                product.prices.append(price)
            self.copy_sale_price(sms_price, price)
            sale_price = price

        # Assign "current" price (Sale/TPR/None).
        if sale_price:
            if self.price_is_current(sale_price):
                product.current_price = sale_price
            elif tpr_price:
                if self.price_is_current(tpr_price):
                    product.current_price = tpr_price
                else:
                    product.current_price = None
            else:
                product.current_price = None
        elif tpr_price:
            if self.price_is_current(tpr_price):
                product.current_price = tpr_price
            else:
                product.current_price = None
        else:
            product.current_price = None

        return product

    def process_price_del(self, row, session):
        """
        Process a price deletion.
        """
        upc = row[u'F01']
        if not upc.isdigit():
            log.warning(u"invalid upc: {0}".format(repr(upc)))
            return

        if row[u'F1000'] != u'PAL':
            log.debug(u"ignoring deletion on target {0} for upc: {1}".format(
                    repr(row[u'F1000']), repr(row[u'F01'])))
            return

        upc = GPC(upc, calc_check_digit=u'upc')
        product = get_product_by_upc(session, upc)
        if not product:
            return

        prices = session.query(model.ProductPrice)\
            .filter(model.ProductPrice.product == product)\
            .filter(model.ProductPrice.level == self.int_(row[u'F126']))

        for price in prices:
            log.debug(u"deleting level {0} price for upc: {1}".format(
                    price.level, repr(row[u'F01'])))
            if product.regular_price is price:
                product.regular_price = None
            if product.current_price is price:
                product.current_price = None
            session.delete(price)

    def process_cost_chg(self, row, session):
        if row['F1000'] != 'PAL':
            log.warning("target {0} not supported for upc {1}".format(
                    repr(row['F1000']), repr(row['F01'])))
            return

        if row['F1184'] != 'CASE':
            log.warning("buying format {0} not supported: {1}".format(
                    repr(row['F1184']), repr(row['F01'])))
            return

        upc = row['F01']
        if not upc.isdigit():
            log.warning("invalid upc: {0}".format(repr(upc)))
            return

        product = self.get_product(upc, session)
        upc = product.upc

        q = session.query(model.Vendor)
        q = q.filter(model.Vendor.id == row['F27'])
        vendor = q.first()
        if not vendor:
            vendor = model.Vendor(id=row['F27'])
            session.add(vendor)

        q = session.query(model.ProductCost)
        q = q.filter(model.ProductCost.product == product)
        q = q.filter(model.ProductCost.vendor == vendor)
        cost = q.first()
        if not cost:
            cost = model.ProductCost()
            cost.vendor = vendor
            product.costs.append(cost)

        cost.code = row['F26']
        try:
            cost.case_size = self.int_(row['F19'])
        except ValueError:
            log.warning("case size for product {0} is not integer, but will be "
                        "coerced: {1}".format(repr(upc), repr(row['F19'])))
            cost.case_size = int(Decimal(row['F19']))
        cost.case_cost = self.decimal(row['F38'])
        cost.unit_cost = self.decimal(row['F1140'])

        if row['F90'] == '1' and cost.preference != 1:
            cost = product.costs.pop(cost.preference - 1)
            product.costs.insert(0, cost)

        return cost

    def process_cost_del(self, row, session):
        upc = row['F01']
        if not upc.isdigit():
            log.warning("invalid upc: {0}".format(repr(upc)))
            return

        upc = GPC(upc, calc_check_digit='upc')

        if row['F1000'] != 'PAL':
            log.warning("target {0} not supported for {1}".format(
                    repr(row['F1000']), repr(upc)))
            return

        if row['F1184'] != 'CASE':
            log.warning("buying format {0} not supported for {1}".format(
                    repr(row['F1184']), repr(upc)))
            return

        q = session.query(model.Product)
        q = q.filter(model.Product.upc == upc)
        product = q.first()
        if not product:
            return

        q = session.query(model.Vendor)
        q = q.filter(model.Vendor.id == row['F27'])
        vendor = q.first()
        if not vendor:
            return

        q = session.query(model.ProductCost)
        q = q.filter(model.ProductCost.product == product)
        q = q.filter(model.ProductCost.vendor == vendor)
        cost = q.first()
        if not cost:
            return

        product.costs.remove(cost)
        session.delete(cost)

    def process_clt_chg(self, row, session):
        q = session.query(model.Customer)
        q = q.filter(model.Customer.id == row['F1148'])
        customer = q.first()
        if customer:
            while customer.emails:
                customer.emails.pop()
            while customer.phones:
                customer.phones.pop()
            while customer.groups:
                customer.groups.pop()
        else:
            person = model.Person()
            person.first_name = row['F1149']
            person.last_name = row['F1150']
            customer = model.Customer(id=row['F1148'])
            customer.people.append(person)
            session.add(customer)

        customer.name = row['F1155']

        if row['F1172']:
            customer.add_phone_number(row['F1172'])
        if row['F1173']:
            customer.add_phone_number(row['F1173'])
        if row['F1573']:
            customer.add_email_address(row['F1573'])

        F1154 = row['F1154'].strip()
        if F1154:
            group = session.query(model.CustomerGroup)\
                .filter(model.CustomerGroup.id == F1154)\
                .first()
            if not group:
                group = model.CustomerGroup(id=F1154)
                session.add(group)
            customer.groups.append(group)

        F1777 = row['F1777'].strip()
        if F1777:
            group = session.query(model.CustomerGroup)\
                .filter(model.CustomerGroup.id == F1777)\
                .first()
            if not group:
                group = model.CustomerGroup(id=F1777)
                session.add(group)
            customer.groups.append(group)

        return customer

    def process_clt_del(self, row, session):
        q = session.query(model.Customer)
        q = q.filter(model.Customer.id == row['F1148'])
        customer = q.first()
        if not customer:
            return

        session.delete(customer)

    def process_clk_chg(self, row, session):
        customer = None
        if row['F1148']:
            q = session.query(model.Customer)
            q = q.filter(model.Customer.id == row['F1148'])
            customer = q.first()
            if not customer:
                customer = model.Customer(id=row['F1148'])
                session.add(customer)

        q = session.query(model.Employee)
        q = q.filter(model.Employee.id == int(row['F1126']))
        employee = q.first()
        if employee:
            while employee.emails:
                employee.emails.pop()
            while employee.phones:
                employee.phones.pop()
        else:
            person = model.Person()
            employee = model.Employee(id=int(row['F1126']))
            employee.person = person
            session.add(employee)
        if customer and not customer.person:
            customer.people.append(employee.person)
            customer.name = employee.person.display_name

        employee.first_name = row['F1143']
        employee.last_name = row['F1144']
        employee.display_name = row['F1127']

        status = (row['F1552'] or '').upper()
        if status == 'ACTV':
            employee.status = enum.EMPLOYEE_STATUS_CURRENT
        elif status == 'FRMR':
            employee.status = enum.EMPLOYEE_STATUS_FORMER
        else:
            employee.status = None

        if row['F1563']:
            employee.add_phone_number(row['F1563'])
        if row['F1564']:
            employee.add_phone_number(row['F1564'])
        if row['F1571']:
            employee.add_email_address(row['F1571'])

        return employee

    def process_clk_del(self, row, session):
        q = session.query(model.Employee)
        q = q.filter(model.Employee.id == row['F1126'])
        employee = q.first()
        if not employee:
            return

        session.delete(employee)


def process_changes(path):
    """
    Processes "changes" data which has been deployed from an SMS server
    instance.

    .. highlight:: ini
    
    This function is designed to be called by a file monitor instance.  An
    example of such configuration would be::

       [edbob.filemon]
       monitored = sms_changes
       sms_changes.dirs = [r'C:\Storeman\XchRattail\Changes']
       sms_changes.actions = ['rattail_locsms.changes:process_changes']

    If you wish to override the default data processor class, then you should
    additionally provide configuration similar to this::

       [rattail.sw.locsms]
       changes.processor_class = myrattail.mymodule:MyChangesProcessor
    """

    m = change_pattern_generic.match(os.path.basename(path))
    if not m:
        log.warning("unexpected filename pattern: {0}".format(repr(path)))
        return

    # Ignore all incoming files until we see one indicating "batch" completion.
    if m.group(2).upper() != 'CHANGES_END':
        return

    deploy_id = m.group(1)

    changes = {}
    changes_dir = os.path.dirname(path)
    for fn in os.listdir(changes_dir):
        if fn.startswith(deploy_id):
            m = change_pattern.match(fn)
            if m:
                changes[m.group(1).upper()] = os.path.join(changes_dir, fn)

    cls = edbob.config.get('rattail.sw.locsms', 'changes.processor_class')
    if cls:
        log.debug("using custom changes processor class: {0}".format(cls))
        cls = load_object(cls)
    else:
        cls = ChangesProcessor
    proc = cls()

    # Process all changes; details of this are up to the class instance.
    proc.process_all_changes(changes)

    # Create 'Processed' folder if it doesn't already exist.
    processed_dir = os.path.join(changes_dir, 'Processed')
    if not os.path.exists(processed_dir):
        os.makedirs(processed_dir)

    # Move all files from change "batch" to 'Processed' folder.
    for fn in list(os.listdir(changes_dir)):
        if fn.startswith(deploy_id):
            overwriting_move(os.path.join(changes_dir, fn), processed_dir)
