#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2012 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Samples Interface
"""

import os
import os.path
import sys
import shutil
import subprocess
import datetime
import time
import tempfile
import zipfile
import logging

from mako.template import Template

import edbob
from edbob.win32 import capture_output


log = logging.getLogger(__name__)


def absorb_sample(zip_path, source_dir):
    """
    Pulls ("absorbs") contents of the SMS Sample (or Option) file at
    ``zip_path`` into the source working directory at ``source_dir``.
    """

    cwd = os.getcwd()
    zip_path = os.path.abspath(zip_path)
    os.chdir(os.path.abspath(source_dir))
    zip_file = zipfile.ZipFile(zip_path, 'r')
    zip_file.extractall()
    zip_file.close()
    os.chdir(cwd)


def render_sample(source_dir, output_dir, **kwargs):
    for dirpath, dirnames, filenames in os.walk(source_dir):
        for filename in filenames:
            template_path = os.path.join(dirpath, filename)
            output_path = os.path.join(output_dir, template_path[len(source_dir)+1:])
            output_parent = os.path.dirname(output_path)
            if not os.path.isdir(output_parent):
                os.makedirs(os.path.dirname(output_path))
            if filename.endswith('.mako'):
                template = Template(filename=template_path, disable_unicode=True)
                output_path = output_path[:-5] # trim .mako
                output_file = open(output_path, 'wb') # pass 'b' to avoid ^M chars
                output_file.write(template.render(**kwargs))
                output_file.close()
            else:
                shutil.copyfile(template_path, output_path)
        

def zip_sample(src, zip_path, prefix=None):
    """
    Reads all files from ``src`` and writes/zips them to ``zip_path``.

    ``prefix`` is used to structure the files within the zip file according to
    expectations on the part of SMS.  If it is not specified, it will be
    gleaned from ``src`` and ``zip_path``.
    """

    if not prefix:
        samples_dir = os.path.basename(os.path.dirname(zip_path))
        prefix = os.path.join(samples_dir, os.path.basename(src))

    zip_file = zipfile.ZipFile(zip_path, 'w')
    for dirpath, dirnames, filenames in os.walk(src):
        for fn in filenames:
            path = os.path.join(dirpath, fn)
            arcname = os.path.join(prefix, dirpath[len(src)+1:], fn)
            zip_file.write(path, arcname)
    zip_file.close()


def git_executable():
    return edbob.config.get('edbob', 'git.executable', default='git')


def get_git_timestamp(source_path):
    """
    Tries to get the "last-modified" timestamp for ``source_path``, which is
    assumed to be under Git revision control.
    """

    git = git_executable()
    output = capture_output('%s log -n 1 "%s"' % (git, source_path))
    for line in output.split('\n'):
        if line.startswith('Date:'):
            timestamp = line[6:].strip()
            words = timestamp.split()
            timestamp = ' '.join(words[:-1])
            timestamp = datetime.datetime.strptime(timestamp, '%a %b %d %H:%M:%S %Y')
            return timestamp
    return None


def get_mercurial_timestamp(source_path):
    """
    Tries to get the "last-modified" timestamp for ``source_path``, which is
    assumed to be under Mercurial revision control.
    """

    output = capture_output('hg log -l 1 "%s"' % source_path)
    for line in output.split('\n'):
        if line.startswith('date:'):
            timestamp = line[6:].strip()
            words = timestamp.split()
            timestamp = ' '.join(words[:-1])
            timestamp = datetime.datetime.strptime(timestamp, '%a %b %d %H:%M:%S %Y')
            return timestamp
    return None


def get_source_timestamp(source_path):
    """
    Tries to get the "last-modified" timestamp from source control.  This of
    course assumes that ``source_path`` *is* in fact under source control.

    Returns the timestamp if found; otherwise returns ``None``.
    """

    # There are quirks when running as a GUI executable in Windows.  If that's
    # how we're running, then we mustn't mess with ``std*`` channels.
    kwargs = {}
    devnull = None
    if hasattr(sys.stdout, 'isatty') and sys.stdout.isatty():
        devnull = open(os.devnull, 'w')
        kwargs = dict(stdout=devnull, stderr=devnull)

    # Check for git repository.
    git_cmd = '%s rev-parse --show-toplevel "%s"' % (git_executable(), source_path)
    try:
        result = subprocess.call(git_cmd, **kwargs)
    except WindowsError, error:
        if sys.platform == 'win32':
            import winerror
            if error[0] == winerror.ERROR_FILE_NOT_FOUND:
                result = 1
            else:
                raise
        else:
            raise
    if result == 0:
        if devnull:
            devnull.close()
        return get_git_timestamp(source_path)

    # Check for mercurial repository.
    cwd = os.getcwd()
    os.chdir(source_path)
    result = subprocess.call('hg root', **kwargs)
    os.chdir(cwd)
    if not result:
        if devnull:
            devnull.close()
        return get_mercurial_timestamp(source_path)

    if devnull:
        devnull.close()
    return None


def fix_mtime(zip_path, mtime):
    atime = time.mktime(time.localtime())
    mtime = time.mktime(mtime.timetuple())
    os.utime(zip_path, (atime, mtime))


def generate_sample(src, dest, render_mako=True, use_source_time=True,
                    **mako_vars):
    """
    Generates a zipped sample (or option) to folder ``dest`` based on
    standalone files found in folder ``src``.  The last segment of ``dest``
    must be either ``'Samples'`` or ``'Options'``.  The last segment of ``src``
    is assumed to be the sample's (or option's) name and will be used to name
    the resultant zip file.

    If ``render_mako`` is ``True``, then files ending in ``*.mako`` will be
    rendered using the `Mako <http://www.makotemplates.org/>`_ template engine,
    passing ``mako_vars`` to each ``render()`` call.  (Also the ``.mako``
    extensions will be removed within the zip file).
    
    If ``use_source_time`` is ``True``, the resultant zip file will have its
    last-modified timestamp overwritten with the last relevant log time from
    the source control repository.  Note that this requires ``src`` to actually
    belong to a source control repository.
    """

    src = os.path.abspath(src)
    dest = os.path.abspath(dest)
    cwd = os.getcwd()

    if render_mako:
        temp_dest = tempfile.mkdtemp(prefix='locsms-samples-')

    sample_name = os.path.basename(src)
    samples_dir = os.path.basename(dest)
    if samples_dir.lower() not in ('samples', 'options'):
        raise ValueError("Final segment of destination must be 'Samples' or 'Options'")

    orig_src = src
    zip_path = os.path.join(dest, '%s.zip' % sample_name)
    if render_mako:
        src = os.path.join(temp_dest, sample_name)
        render_sample(orig_src, src, **mako_vars)
    os.chdir(src)
    zip_sample(src, zip_path)
    if use_source_time:
        mtime = get_source_timestamp(orig_src)
        if mtime:
            fix_mtime(zip_path, mtime)
        else:
            log.warning("generate_sample: Could not determine source timestamp for path: %s" % orig_src)

    if render_mako:
        # Can't figure out why a process is hanging onto the temp folder, but
        # it's preventing the entire thing from being deleted here...
        shutil.rmtree(temp_dest, ignore_errors=True)

    os.chdir(cwd)
