# -*- coding: utf-8 -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2014 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Auxiliary Data Processing
"""

import os.path
import re
import logging
import csv

import edbob
from edbob.util import requires_impl

from rattail.core import Object
from rattail.db import Session
from rattail.db import cache
from rattail.db import model
from rattail.util import load_object
from rattail.files import overwriting_move


auxiliary_pattern = re.compile(r'^(\d{8})_(.+)_LOAD.CSV$', re.IGNORECASE)

log = logging.getLogger(__name__)


class AuxiliaryProcessor(Object):
    """
    Base class and default implementation for processing auxiliary data which
    has been deployed from SMS.
    """

    def process_load(self, data_type, csv_path, progress=None):
        found = False
        for load_type, loader in self.relevant_loaders():
            if data_type == load_type:
                found = True
                break
        if not found:
            log.warning("process_load: Ignoring load file: %s" % csv_path)
            return

        session = Session()
        loader = loader(session=session)
        if loader.perform_load(csv_path, progress):
            session.commit()
        else:
            session.rollback()
        session.close()

    def relevant_loaders(self):
        yield u'DEPT', DepartmentLoader
        yield u'SDP', SubdepartmentLoader
        yield u'FAM', FamilyLoader
        yield u'RPC', ReportCodeLoader
        yield u'CLG', CustomerGroupLoader


class DataLoader(Object):
    """
    Generic class for performing a particular type of data load from SMS.
    """

    @property
    @requires_impl(is_property=True)
    def cacher(self):
        pass

    def int_(self, val):
        return int(val) if val else None

    def perform_cache(self, progress):
        cacher = self.cacher(session=self.session)
        self.instances = cacher.get_cache(progress)
        if self.instances is None:
            return False
        return True

    def perform_load(self, csv_path, progress):
        if not self.perform_cache(progress):
            return False

        valid_keys = []

        csv_file = open(csv_path, 'rb')
        reader = csv.DictReader(csv_file, quotechar="'")
        for row in reader:
            key = self.load_instance(row)
            valid_keys.append(key)
            self.session.flush()
        csv_file.close()

        for key in self.instances.keys():
            if key not in valid_keys:
                self.remove_instance(self.instances[key])

        return True

    def load_instance(self, row):
        raise NotImplementedError

    def remove_instance(self, instance):
        self.session.delete(instance)


class DepartmentLoader(DataLoader):

    cacher = cache.DepartmentCacher

    def load_instance(self, row):
        number = self.int_(row['F03'])
        assert number is not None
        department = self.instances.get(number)
        if not department:
            department = model.Department(number=number)
            self.session.add(department)

        department.name = row['F238']
        return department.number

    def remove_instance(self, department):

        # Remove association from Subdepartment records.
        q = self.session.query(model.Subdepartment)
        q = q.filter(model.Subdepartment.department == department)
        for subdept in q:
            subdept.department = None
            self.session.flush()

        # Remove association from Product records.
        q = self.session.query(model.Product)
        q = q.filter(model.Product.department == department)
        for product in q:
            product.department = None
            self.session.flush()

        self.session.delete(department)


class SubdepartmentLoader(DataLoader):

    cacher = cache.SubdepartmentCacher

    def load_instance(self, row):
        number = self.int_(row['F04'])
        assert number is not None
        subdepartment = self.instances.get(number)
        if not subdepartment:
            subdepartment = model.Subdepartment(number=number)
            self.session.add(subdepartment)

        department = None
        number = self.int_(row['F03'])
        if number:
            q = self.session.query(model.Department)
            q = q.filter(model.Department.number == number)
            department = q.first()
            if not department:
                department = model.Department(number=number)
                self.session.add(department)

        subdepartment.name = row['F1022']
        subdepartment.department = department
        return subdepartment.number

    def remove_instance(self, subdepartment):

        # Remove association from Product records.
        q = self.session.query(model.Product)
        q = q.filter(model.Product.subdepartment == subdepartment)
        for product in q:
            product.subdepartment = None
            self.session.flush()

        self.session.delete(subdepartment)


class FamilyLoader(DataLoader):

    cacher = cache.FamilyCacher

    def load_instance(self, row):
        code = self.int_(row[u'F16'])
        assert code is not None
        family = self.instances.get(code)
        if not family:
            family = model.Family(code=code)
            self.session.add(family)

        family.name = row[u'F1040']
        return family.code


class ReportCodeLoader(DataLoader):

    cacher = cache.ReportCodeCacher

    def load_instance(self, row):
        code = self.int_(row[u'F18'])
        assert code is not None
        report_code = self.instances.get(code)
        if not report_code:
            report_code = model.ReportCode(code=code)
            self.session.add(report_code)

        report_code.name = row[u'F1024']
        return report_code.code


class CustomerGroupLoader(DataLoader):

    cacher = cache.CustomerGroupCacher

    def load_instance(self, row):
        id = row['F1154']
        assert id.strip()
        group = self.instances.get(id)
        if not group:
            group = model.CustomerGroup(id=id)
            self.session.add(group)

        group.name = row['F1268']
        return group.id
        
    def remove_instance(self, group):

        # Remove association from Customer records.
        q = self.session.query(model.CustomerGroupAssignment)
        q = q.filter(model.CustomerGroupAssignment.group == group)
        for assignment in q:
            customer = assignment.customer
            self.session.delete(assignment)
            if customer:
                customer._groups.reorder()

        self.session.delete(group)


def process_auxiliary(path):
    """
    Processes auxiliary data which has been deployed from an SMS server
    instance.

    .. highlight:: ini
    
    This function is designed to be called by a file monitor instance.  An
    example of such configuration would be::

       [edbob.filemon]
       monitored = sms_auxiliary
       sms_auxiliary.dirs = [r'C:\Storeman\XchRattail\Auxiliary']
       sms_auxiliary.actions = ['rattail_locsms.auxiliary:process_auxiliary', 'os:remove']

    If you wish to override the default data processor class, then you should
    additionally provide configuration similar to this::

       [rattail.sw.locsms]
       auxiliary.processor_class = myrattail.mymodule:MyAuxiliaryProcessor
    """

    m = auxiliary_pattern.match(os.path.basename(path))
    if not m:
        log.warning("process_auxiliary: Unexpected filename pattern: %s" % path)
        return

    batch_type = m.group(2).upper()

    cls = edbob.config.get('rattail.sw.locsms', 'auxiliary.processor_class')
    if cls:
        log.debug("using custom auxiliary processor class: {0}".format(cls))
        cls = load_object(cls)
    else:
        cls = AuxiliaryProcessor
    proc = cls()

    # Process CSV file; details are up to the class instance.
    proc.process_load(batch_type, path)

    # Move data file to 'Processed' folder.
    processed_dir = os.path.join(os.path.dirname(path), 'Processed')
    if not os.path.exists(processed_dir):
        os.makedirs(processed_dir)
    overwriting_move(path, processed_dir)
