# -*- coding: utf-8 -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2014 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Rattail Commands
"""

from __future__ import unicode_literals

import os
import re
import sys
import shutil
import zipfile

from rattail.commands import Subcommand


class AdminCommand(Subcommand):
    """
    Launches the 'SMS Admin' utility.  This tool is useful for administering
    SMS in both development and production environments.
    """

    name = 'sms-admin'
    description = "Launch 'SMS Admin' utility"

    def run(self, args):
        from edbob.wx import LaunchDialog
        from rattail_locsms.wx.Admin import AdminDialog
        LaunchDialog(AdminDialog)


class GenerateSampleCommand(Subcommand):
    """
    Generates a zipped Sample (or Option) file to an SMS install folder; called
    as ``rattail sms-gen-sample``.
    """

    name = 'sms-gen-sample'
    description = "Generate zipped SMS Sample (or Option) file"

    def add_parser_args(self, parser):
        parser.add_argument('source_dir',
                            help=r"Path to Sample source code (e.g. C:\Samples\MySample)")
        parser.add_argument('target_dir',
                            help=r"Path to Samples folder (e.g. C:\Storeman\install\Samples)")
        parser.add_argument('-t', '--fix-timestamp', action='store_true',
                            help="Set timestamp according to source control log")
        parser.add_argument('-m', '--mako-args', nargs='*', metavar='VAR=VAL',
                            help="Render *.mako files with specified Mako context vars")

    def run(self, args):
        from rattail_locsms.samples import generate_sample

        mako_arg_pattern = re.compile(r'^([A-Za-z_]+)=(.*)$')

        mako_args = {}
        if args.mako_args:
            for elem in args.mako_args:
                m = mako_arg_pattern.match(elem)
                if not m:
                    sys.stderr.write("Invalid Mako argument: %s\n" % elem)
                    return
                mako_args[m.group(1)] = eval(m.group(2))

        generate_sample(self.config, args.source_dir, args.target_dir,
                        render_mako=bool(mako_args),
                        use_source_time=args.fix_timestamp,
                        **mako_args)


class PullSampleCommand(Subcommand):
    """
    "Pulls" an SMS Sample (or Option) zip file into a source repository's
    working directory; called as ``rattail sms-pull-sample``.

    This tool will unzip the sample (or option) file and copy the individual
    files contained within it to the local working directory (which is assumed
    to be associated with a source repository).
    """

    name = 'sms-pull-sample'
    description = "Pull zipped SMS Sample (or Option) file"

    def add_parser_args(self, parser):
        parser.add_argument('sample_file',
                            help=r"Path to zipped Sample (e.g. C:\Storeman\install\Samples\MySample.zip)")
        parser.add_argument('source_dir',
                            help=r"Path to source working directory (e.g. C:\Samples\MySample)")

    def run(self, args):
        from rattail_locsms.samples import absorb_sample
        absorb_sample(args.sample_file, args.source_dir)
