# -*- coding: utf-8 -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2014 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
SIL Processing Utilities
"""

from __future__ import unicode_literals

import os
import datetime

import edbob

from rattail import sil
from rattail.sil.columns import SILColumn


class SMSWriter(sil.Writer):

    def write(self, string):
        string = string.replace(os.linesep, '\r\n')
        self.fileobj.write(string)

    def write_batch_header(self, **kwargs):
        kwargs.setdefault('H04', 'PAL')
        super(SMSWriter, self).write_batch_header(**kwargs)


# # def get_header_with_create(fields, view='TEMP_DB', **kwargs):
# #     """
# #     Returns a fuller, more specific type of SIL header.  It is meant to
# #     accommodate the bulk of SIL batch file operations.  The output will be
# #     something like this::

# #        INSERT INTO HEADER_DCT VALUES
# #        ('HC', ... );

# #        CREATE TABLE TEMP_DCT (@dbDCT( ... ));

# #        INSERT INTO HEADER_DCT VALUES
# #        ('HM', ... ):

# #        CREATE VIEW XXX AS SELECT ... FROM TEMP_DCT;

# #        INSERT INTO XXX VALUES

# #     ``fields`` is a required sequence of field names, e.g. ``['F01', 'F1000',
# #     'F1032']``.  The ``view`` parameter will be used to name the view ("XXX" in
# #     the example above) and, if provided, should be the name of a SMS macro to
# #     further direct its processing of the batch.  The remainder of keyword
# #     arguments are passed directly to :func:`get_header()`.
# #     """

# #     fields = ','.join(fields)
# #     batch_id = consume_batch_id()
# #     header = get_header('HC', batch_id=batch_id, **kwargs) + '\n'
# #     header += '\n'
# #     header += "CREATE TABLE TEMP_DCT (@dbDCT(%s));\n" % fields
# #     header += '\n'
# #     header += get_header('HM', batch_id=batch_id, **kwargs) + '\n'
# #     header += '\n'
# #     header += "CREATE VIEW %s AS SELECT %s FROM TEMP_DCT;\n" % (view, fields)
# #     header += '\n'
# #     header += "INSERT INTO %s VALUES" % view
# #     return header


def provide_columns():
    """
    Provides extra SIL columns supported by LOC SMS.
    """

    SC = SILColumn
    
    custom = [ # These columns are SMS-specific.

        SC('L1000',     'CHAR(5)',      "Target Identifier"),
        SC('L1184',     'CHAR(12)',     "Buying format"),
        ]

    columns = {}
    for column in custom:
        columns[column.name] = column
    return columns


# def write_inventory_import(path, batch, progress_factory=None, **kwargs):
#     """
#     .. highlight:: python

#     Generates a SIL file which, when processed by SMS, will create an Inventory
#     Multi-Count document available for import.  ``path`` determines where the
#     file is written, and ``batch`` must be a sequence of tuples of the form::

#        (upc, cases, units)

#     Any extra keyword arguments are passed directly to :func:`get_header()`.
#     Note that several (default) batch header values are provided by this
#     function however.  (See source for specifics.)
#     """

#     kwargs.setdefault('batch_action_type', 'ADD')
#     kwargs.setdefault('batch_description', 'INVENTORY')
#     kwargs.setdefault('batch_active_date', '0000000')
#     kwargs.setdefault('batch_active_time', '0000')
#     kwargs.setdefault('batch_ending_date', '@DJDF')
#     kwargs.setdefault('batch_purge_date', '@DJD+090')
#     kwargs.setdefault('batch_audit_file', 'ERRORS.SIL')

#     import_file = open(path, 'w')
#     print >> import_file, get_header(**kwargs)
#     print >> import_file, """
# CREATE VIEW TEMP_DB AS SELECT F1032,F01,F1003,F70 FROM TEMP_DCT;

# INSERT INTO TEMP_DB VALUES"""

#     progress = None
#     if progress_factory:
#         progress = progress_factory("Writing inventory import file", len(batch))
#     for i, record in enumerate(batch, 1):
#         # Use 1 (dummy value) for F1032; will be replaced in SQL below.
#         print >> import_file, get_record(1, *record)
#         if progress:
#             progress.update(i)
#     if progress:
#         progress.destroy()

#     print >> import_file, """
# DELETE FROM HEADER_DCT WHERE F902='@WIZGET(BATCH)' AND F903='@WIZGET(SOURCE)';
# UPDATE M@WIZGET(SOURCE)@WIZGET(BATCH) SET F1032=@WIZGET(BATCH);

# @FMT(CMP,@DBHOT(FINDTABLE,INV_BAT)=,'®CREATE(INV_BAT,HDRINV)');

# @UPDATE_BATCH(JOB=ADD,TAR=INV_BAT,KEY=F1032=:F1032,
# SRC=SELECT @WIZGET(BATCH) AS F1032,'@DSDF' AS F254,'M@WIZGET(SOURCE)@WIZGET(BATCH)' AS F1255,'MULTI' AS F1068 FROM SYS_TAB);
# """
    
#     import_file.close()


def write_receiving_import(path, order, F1068='DSD', F76=None, F1653=None,
                           F1246=None, F254=None, **header):
    """
    Writes a SIL file for a Receiving Document, to be made available for import
    within SMS.
    """

    now = edbob.local_time()

    if not F76:         # order date
        F76 = now.date().strftime('%m/%d/%Y')
    if not F1653:       # shipment date
        F1653 = F76
    if not F1246:       # delivery date
        F1246 = F76
    if not F254:        # invoice date
        F254 = F76

    if 'H02' not in header:
        header['H02'] = sil.consume_batch_id()

    header.setdefault('H12', 'ADD')
    header.setdefault('H13', 'RECEIVING IMPORT')

    writer = SMSWriter(path)
    writer.write_maintenance_header(**header)
    writer.write('CREATE VIEW REC_IMPORT AS SELECT F1032,F27,F01,F1003,F70,F38 FROM ITEM_DCT;\n\n')
    writer.write('INSERT INTO REC_IMPORT VALUES\n')

    # Use 1 (dummy value) for F1032; will be replaced in SQL below.
    rows = [(1, order.vendor_id, x.upc, x.cases, x.units, x.base_cost) for x in order.details]
    writer.write_rows(rows)

    writer.write("""
@UPDATE_BATCH(JOB=ADDRPL,TAR=REC_BAT,KEY=F1032=:F1032,
SRC=SELECT BAT.F27,'%(F76)s' AS F76,'%(F1653)s' AS F1653,'%(F1246)s' AS F1246,'%(F254)s' AS F254,MAX(VND.F334) AS F334,@WIZGET(BATCH) AS F1032,'%(F1068)s' AS F1068,'M@WIZGET(SOURCE)@WIZGET(BATCH)' AS F1255
FROM M@WIZGET(SOURCE)@WIZGET(BATCH) BAT LEFT OUTER JOIN VENDOR_TAB VND ON VND.F27=BAT.F27
GROUP BY BAT.F27);

UPDATE M@WIZGET(SOURCE)@WIZGET(BATCH) SET F1032=@WIZGET(BATCH);

DELETE FROM HEADER_DCT WHERE F902='@WIZGET(BATCH)' AND F903='@WIZGET(SOURCE)';
""" % locals())

    writer.close()
