#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2012 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

# generated by wxGlade 0.6.3 on Sun Dec 18 16:54:10 2011

"""
SMS Admin Utility
"""

import wx
import wx.grid

# begin wxGlade: extracode
# end wxGlade

import os
import os.path
import sys
import shutil

if sys.platform == 'win32':
    import pywintypes
    import win32api
    import win32con
    import winerror

import edbob
from edbob.win32 import RegDeleteTree

from locsms.deploy import flip_archive_bit

from rattail.core import get_uuid
from rattail_locsms import deploy
from rattail_locsms.samples import generate_sample
from rattail.files import resource_path


KEY_BASE                = r'Software\Rattail\SMS Admin'
KEY_ENVIRONMENTS        = KEY_BASE + r'\Environments'
KEY_HOTFIX              = KEY_BASE + r'\Hotfix'
KEY_SAMPLES             = KEY_BASE + r'\Samples'


def save_environment(env):
    """
    Saves the given :class:`rattail_locsms.deploy.Environment` instance to the
    registry.
    """

    if not env.uuid:
        env.uuid = get_uuid()

    key = win32api.RegCreateKey(win32con.HKEY_CURRENT_USER, r'%s\%s' % (KEY_ENVIRONMENTS, env.uuid))
    RegDeleteTree(key, 'Nodes')
    win32api.RegSetValueEx(key, 'Name', 0, win32con.REG_SZ, env.name)
    win32api.RegCloseKey(key)

    key = win32api.RegCreateKey(win32con.HKEY_CURRENT_USER, r'%s\%s\Nodes' % (KEY_ENVIRONMENTS, env.uuid))
    for node in env.nodes:
        win32api.RegSetValueEx(key, node.name, 0, win32con.REG_SZ, node.path)
    win32api.RegCloseKey(key)


def get_environments(sort=True):
    envs = []
    try:
        key = win32api.RegOpenKeyEx(win32con.HKEY_CURRENT_USER, KEY_ENVIRONMENTS, 0, win32con.KEY_READ)
    except pywintypes.error, error:
        if error[0] == winerror.ERROR_FILE_NOT_FOUND:
            pass
    else:
        for uuid, reserved, class_, mtime in win32api.RegEnumKeyEx(key):
            envs.append(get_environment(uuid))
        win32api.RegCloseKey(key)
    if sort:
        envs = sorted(envs, key=lambda x: x.name)
    return envs


def get_environment(uuid, env=None):
    """
    Reads an environment's data from the registry (based on ``uuid``) and
    returns a :class:`rattail_locsms.deploy.Environment` instance.
    """

    if not env:
        env = deploy.Environment(uuid)

    key = win32api.RegOpenKeyEx(win32con.HKEY_CURRENT_USER, r'%s\%s' % (KEY_ENVIRONMENTS, uuid))
    try:
        value, type_ = win32api.RegQueryValueEx(key, 'Name')
    except pywintypes.error, error:
        if error[0] != winerror.ERROR_FILE_NOT_FOUND:
            raise
    else:
        env.name = value
    win32api.RegCloseKey(key)

    try:
        key = win32api.RegOpenKeyEx(win32con.HKEY_CURRENT_USER, r'%s\%s\Nodes' % (KEY_ENVIRONMENTS, uuid))
    except pywintypes.error, error:
        if error[0] != winerror.ERROR_FILE_NOT_FOUND:
            raise
    else:
        i = 0
        while True:
            try:
                name, path, type_ = win32api.RegEnumValue(key, i)
            except pywintypes.error, error:
                if error[0] == winerror.ERROR_NO_MORE_ITEMS:
                    break
                raise
            else:
                store, terminal = name.split(':')
                env.nodes.append(deploy.Node(store=store, terminal=terminal, path=path))
            i += 1

    return env


def delete_environment(env):
    key = win32api.RegOpenKeyEx(win32con.HKEY_CURRENT_USER, KEY_ENVIRONMENTS, 0, win32con.KEY_WRITE)
    RegDeleteTree(key, env.uuid)
    win32api.RegCloseKey(key)


def get_samples_environment(uuid):
    env = deploy.SamplesEnvironment(uuid)
    get_environment(uuid, env)

    try:
        key = win32api.RegOpenKeyEx(win32con.HKEY_CURRENT_USER, r'%s\%s' % (KEY_SAMPLES, uuid))
    except pywintypes.error, error:
        if error[0] != winerror.ERROR_FILE_NOT_FOUND:
            raise
    else:

        try:
            value, type_ = win32api.RegQueryValueEx(key, 'Source Path')
        except pywintypes.error, error:
            if error[0] != winerror.ERROR_FILE_NOT_FOUND:
                raise
        else:
            env.source_path = value

        try:
            value, type_ = win32api.RegQueryValueEx(key, 'Use Mercurial Times')
        except pywintypes.error, error:
            if error[0] != winerror.ERROR_FILE_NOT_FOUND:
                raise
        else:
            env.use_mercurial_times = value == 'True'

        try:
            value, type_ = win32api.RegQueryValueEx(key, 'Target Nodes')
        except pywintypes.error, error:
            if error[0] != winerror.ERROR_FILE_NOT_FOUND:
                raise
        else:
            env.target_nodes = value.split(',')

        win32api.RegCloseKey(key)

    try:
        key = win32api.RegOpenKeyEx(win32con.HKEY_CURRENT_USER, r'%s\%s\Mako Vars' % (KEY_SAMPLES, uuid))
    except pywintypes.error, error:
        if error[0] != winerror.ERROR_FILE_NOT_FOUND:
            raise
    else:
        i = 0
        while True:
            try:
                name, value, type_ = win32api.RegEnumValue(key, i)
            except pywintypes.error, error:
                if error[0] == winerror.ERROR_NO_MORE_ITEMS:
                    break
                raise
            env.mako_vars[name] = value
            i += 1
        win32api.RegCloseKey(key)

    return env


def save_samples_environment(env):
    """
    Saves the given :class:`rattail_locsms.deploy.SamplesEnvironment` instance
    to the registry.
    """

    if not env.uuid:
        env.uuid = get_uuid()

    key = win32api.RegCreateKey(win32con.HKEY_CURRENT_USER, r'%s\%s' % (KEY_SAMPLES, env.uuid))
    win32api.RegSetValueEx(key, 'Source Path', 0, win32con.REG_SZ, env.source_path)
    win32api.RegSetValueEx(key, 'Use Mercurial Times', 0, win32con.REG_SZ, repr(env.use_mercurial_times))
    win32api.RegSetValueEx(key, 'Target Nodes', 0, win32con.REG_SZ, ','.join(env.target_nodes))
    RegDeleteTree(key, 'Mako Vars')
    win32api.RegCloseKey(key)

    key = win32api.RegCreateKey(win32con.HKEY_CURRENT_USER, r'%s\%s\Mako Vars' % (KEY_SAMPLES, env.uuid))
    for name in env.mako_vars:
        win32api.RegSetValueEx(key, name, 0, win32con.REG_SZ, env.mako_vars[name])
    win32api.RegCloseKey(key)
    

def get_hotfix_environment(uuid):
    env = deploy.HotfixEnvironment(uuid)
    get_environment(uuid, env)

    try:
        key = win32api.RegOpenKeyEx(win32con.HKEY_CURRENT_USER, r'%s\%s' % (KEY_HOTFIX, uuid))
    except pywintypes.error, error:
        if error[0] != winerror.ERROR_FILE_NOT_FOUND:
            raise
    else:
        try:
            value, type_ = win32api.RegQueryValueEx(key, 'Target Nodes')
        except pywintypes.error, error:
            if error[0] != winerror.ERROR_FILE_NOT_FOUND:
                raise
        else:
            env.target_nodes = value.split(',')

        win32api.RegCloseKey(key)

    return env


def save_hotfix_environment(env):
    """
    Saves the given :class:`rattail_locsms.deploy.HotfixEnvironment` instance
    to the registry.
    """

    if not env.uuid:
        env.uuid = get_uuid()
    key = win32api.RegCreateKey(win32con.HKEY_CURRENT_USER, r'%s\%s' % (KEY_HOTFIX, env.uuid))
    win32api.RegSetValueEx(key, 'Target Nodes', 0, win32con.REG_SZ, ','.join(env.target_nodes))
    win32api.RegCloseKey(key)


def PromptDir(parent, default=''):
    """
    Prompts the user to choose a directory, but does so via the
    ``wx.FileDialog`` class instead of using ``wx.DirDialog`` (since the latter
    is too limiting).
    """

    dlg = wx.FileDialog(parent, defaultDir=default, defaultFile='(folder selection)', style=wx.FD_OPEN)
    res = dlg.ShowModal()
    dlg.Destroy()
    if res == wx.ID_OK:
        return os.path.dirname(dlg.GetPath())
    return None


class NodeListBox(wx.CheckListBox):

    def __init__(self, *args, **kwargs):
        wx.CheckListBox.__init__(self, *args, **kwargs)
        self.nodes = []

    def Clear(self):
        wx.CheckListBox.Clear(self)
        del self.nodes[:]

    def GetSelectedNodes(self):
        nodes = []
        for i in range(self.GetCount()):
            if self.IsChecked(i):
                nodes.append(self.nodes[i])
        return nodes

    def LoadEnvironment(self, env, servers_only=False):
        self.Clear()
        for i, node in enumerate(env.sorted_nodes):
            if not servers_only or node.terminal == '901':
                self.Append(node.name)
                self.nodes.append(node)
                if node.name in env.target_nodes:
                    self.Check(i)
    
    def UpdateEnvironment(self, env):
        del env.target_nodes[:]
        for i in range(self.GetCount()):
            if self.IsChecked(i):
                env.target_nodes.append(self.nodes[i].name)


class AdminDialog(wx.Dialog):
    """
    This ``wx.Dialog``-derived class is the 'SMS Admin' utility.
    """

    def __init__(self, *args, **kwds):
        # begin wxGlade: AdminDialog.__init__
        kwds["style"] = wx.DEFAULT_DIALOG_STYLE
        wx.Dialog.__init__(self, *args, **kwds)
        self.Notebook = wx.Notebook(self, -1, style=0)
        self.NotebookPane3 = wx.Panel(self.Notebook, -1)
        self.NotebookPane2 = wx.Panel(self.Notebook, -1)
        self.NotebookPane1 = wx.Panel(self.Notebook, -1)
        self.Close = wx.Button(self, wx.ID_OK, "&Close")

        self.__set_properties()
        self.__do_layout()
        # end wxGlade

        self.DeployHotfixPanel = DeployHotfixPanel(self.NotebookPane1, -1)
        self.DeployHotfixSizer.Add(self.DeployHotfixPanel, 1, wx.EXPAND|wx.ALL, 10)
        self.DeployHotfixSizer.Layout()

        self.DeploySamplesPanel = DeploySamplesPanel(self.NotebookPane2, -1)
        self.DeploySamplesSizer.Add(self.DeploySamplesPanel, 1, wx.EXPAND|wx.ALL, 10)
        self.DeploySamplesSizer.Layout()

        self.SettingsPanel = SettingsPanel(self.NotebookPane3, -1)
        self.SettingsSizer.Add(self.SettingsPanel, 1, wx.EXPAND|wx.ALL, 10)
        self.SettingsSizer.Layout()

        # Hm, I'm not real clear if this is the best way to accomplish the
        # resize, but it does the trick...
        self.GetSizer().SetSizeHints(self)
        self.Centre()

    def __set_properties(self):
        # begin wxGlade: AdminDialog.__set_properties
        self.SetTitle("SMS Admin")
        self.NotebookPane1.SetMinSize((324, 244))
        # end wxGlade

        try:
            # If this works, then we're running under the normal Python
            # interpreter.
            icon_path = resource_path('rattail_locsms:images/sms-admin.ico')

        except NotImplementedError:
            # Must be running under a frozen executable (i.e. py2exe).  In this
            # case, the icon should have been "baked in."
            hModule = win32api.GetModuleHandle(None)
            exe_path = win32api.GetModuleFileName(hModule)
            icon_path = exe_path + ';0'

        icon = wx.Icon(icon_path, wx.BITMAP_TYPE_ICO)
        self.SetIcon(icon)

    def __do_layout(self):
        # begin wxGlade: AdminDialog.__do_layout
        sizer_3 = wx.BoxSizer(wx.VERTICAL)
        sizer_4 = wx.BoxSizer(wx.VERTICAL)
        SettingsSizer = wx.BoxSizer(wx.VERTICAL)
        DeploySamplesSizer = wx.BoxSizer(wx.VERTICAL)
        DeployHotfixSizer = wx.BoxSizer(wx.VERTICAL)
        self.NotebookPane1.SetSizer(DeployHotfixSizer)
        self.NotebookPane2.SetSizer(DeploySamplesSizer)
        self.NotebookPane3.SetSizer(SettingsSizer)
        self.Notebook.AddPage(self.NotebookPane1, "Deploy Hotfix")
        self.Notebook.AddPage(self.NotebookPane2, "Deploy Samples")
        self.Notebook.AddPage(self.NotebookPane3, "Settings")
        sizer_4.Add(self.Notebook, 1, wx.EXPAND, 0)
        sizer_4.Add(self.Close, 0, wx.TOP|wx.ALIGN_RIGHT, 10)
        sizer_3.Add(sizer_4, 1, wx.ALL|wx.EXPAND, 10)
        self.SetSizer(sizer_3)
        sizer_3.Fit(self)
        self.Layout()
        self.Centre()
        # end wxGlade

        self.DeployHotfixSizer = DeployHotfixSizer
        self.DeploySamplesSizer = DeploySamplesSizer
        self.SettingsSizer = SettingsSizer

# end of class AdminDialog


class EnvironmentalPanel(wx.Panel):

    def GetSelectedEnvironment(self):
        return self.Environment.GetClientData(self.Environment.GetSelection())

    def LoadEnvironments(self, getter=None):
        selected = None
        i = self.Environment.GetSelection()
        if i >= 0:
            selected = self.Environment.GetClientData(i).uuid
        self.Environment.Clear()
        for env in get_environments():
            name = env.name
            if getter:
                env = getter(env.uuid)
            if env:
                self.Environment.Append(name, env)
        if not selected and self.Environment.GetCount():
            selected = self.Environment.GetClientData(0).uuid
        self.LoadEnvironment(selected)

    def SelectEnvironment(self, uuid):
        for i in range(self.Environment.GetCount()):
            if self.Environment.GetClientData(i).uuid == uuid:
                self.Environment.SetSelection(i)
                return True
        return False


class DeployHotfixPanel(EnvironmentalPanel):

    def __init__(self, *args, **kwds):
        # begin wxGlade: DeployHotfixPanel.__init__
        kwds["style"] = wx.TAB_TRAVERSAL
        EnvironmentalPanel.__init__(self, *args, **kwds)
        self.label_7 = wx.StaticText(self, -1, "En&vironment:")
        self.Environment = wx.Choice(self, -1, choices=[])
        self.RefreshEnvironments = wx.Button(self, -1, "&Refresh")
        self.FilesLabel = wx.StaticText(self, -1, "&Files:")
        self.Files = wx.ListView(self, -1, style=wx.LC_REPORT|wx.LC_SINGLE_SEL|wx.SUNKEN_BORDER)
        self.AddFile = wx.Button(self, -1, "&Add...")
        self.EditFile = wx.Button(self, -1, "&Edit...")
        self.DeleteFile = wx.Button(self, -1, "&Delete")
        self.OpenFile = wx.Button(self, -1, "&Open")
        self.NodesLabel = wx.StaticText(self, -1, "Target &Nodes:")
        self.Nodes = NodeListBox(self, -1, choices=[])
        self.Deploy = wx.Button(self, -1, "De&ploy")

        self.__set_properties()
        self.__do_layout()

        self.Bind(wx.EVT_CHOICE, self.OnEnvironmentChoice, self.Environment)
        self.Bind(wx.EVT_BUTTON, self.OnRefreshEnvironmentsButton, self.RefreshEnvironments)
        self.Bind(wx.EVT_LIST_ITEM_DESELECTED, self.OnFilesItemDeselected, self.Files)
        self.Bind(wx.EVT_LIST_ITEM_SELECTED, self.OnFilesItemSelected, self.Files)
        self.Bind(wx.EVT_BUTTON, self.OnAddFileButton, self.AddFile)
        self.Bind(wx.EVT_BUTTON, self.OnEditFileButton, self.EditFile)
        self.Bind(wx.EVT_BUTTON, self.OnDeleteFileButton, self.DeleteFile)
        self.Bind(wx.EVT_BUTTON, self.OnOpenFileButton, self.OpenFile)
        self.Bind(wx.EVT_BUTTON, self.OnDeployButton, self.Deploy)
        # end wxGlade

        self.Nodes.Bind(wx.EVT_CHECKLISTBOX, self.OnNodesChecklistbox)

        self.LoadEnvironments(getter=get_hotfix_environment)
        self.hotfixes = []

    def __set_properties(self):

        self.Files.InsertColumn(0, "Filename")
        self.Files.InsertColumn(1, "Target Dir")

        # begin wxGlade: DeployHotfixPanel.__set_properties
        self.Files.SetMinSize((200, -1))
        # end wxGlade

    def __do_layout(self):
        # begin wxGlade: DeployHotfixPanel.__do_layout
        sizer_2 = wx.BoxSizer(wx.VERTICAL)
        sizer_5 = wx.BoxSizer(wx.HORIZONTAL)
        sizer_14 = wx.BoxSizer(wx.VERTICAL)
        sizer_15 = wx.BoxSizer(wx.HORIZONTAL)
        sizer_2.Add(self.label_7, 0, 0, 0)
        sizer_15.Add(self.Environment, 1, 0, 0)
        sizer_15.Add(self.RefreshEnvironments, 0, wx.LEFT, 5)
        sizer_2.Add(sizer_15, 0, wx.TOP|wx.EXPAND, 5)
        sizer_2.Add(self.FilesLabel, 0, wx.TOP, 10)
        sizer_5.Add(self.Files, 1, wx.EXPAND, 0)
        sizer_14.Add(self.AddFile, 0, 0, 0)
        sizer_14.Add(self.EditFile, 0, wx.TOP, 5)
        sizer_14.Add(self.DeleteFile, 0, wx.TOP, 5)
        sizer_14.Add(self.OpenFile, 0, wx.TOP, 5)
        sizer_5.Add(sizer_14, 0, wx.LEFT|wx.EXPAND, 5)
        sizer_2.Add(sizer_5, 1, wx.TOP|wx.EXPAND, 5)
        sizer_2.Add(self.NodesLabel, 0, wx.TOP, 10)
        sizer_2.Add(self.Nodes, 1, wx.TOP|wx.EXPAND, 5)
        sizer_2.Add(self.Deploy, 0, wx.TOP|wx.ALIGN_CENTER_HORIZONTAL, 10)
        self.SetSizer(sizer_2)
        sizer_2.Fit(self)
        # end wxGlade

    def OnEnvironmentChoice(self, event): # wxGlade: DeployHotfixPanel.<event_handler>
        self.LoadEnvironment(event.GetClientData().uuid)
        event.Skip()

    def OnRefreshEnvironmentsButton(self, event): # wxGlade: DeployHotfixPanel.<event_handler>
        self.LoadEnvironments(getter=get_hotfix_environment)
        event.Skip()

    def OnFilesItemDeselected(self, event): # wxGlade: DeployHotfixPanel.<event_handler>
        self.EnableEnvironmentControls()
        event.Skip()

    def OnFilesItemSelected(self, event): # wxGlade: DeployHotfixPanel.<event_handler>
        self.EnableEnvironmentControls()
        event.Skip()

    def OnAddFileButton(self, event): # wxGlade: DeployHotfixPanel.<event_handler>
        hotfix = deploy.HotfixFile()
        dlg = HotfixFileDialog(self, hotfix)
        res = dlg.ShowModal()
        dlg.Destroy()
        if res == wx.ID_OK:
            i = self.Files.GetItemCount()
            self.Files.InsertStringItem(i, hotfix.filename)
            self.Files.SetStringItem(i, 1, hotfix.target_dir)
            self.EnableEnvironmentControls()
            self.hotfixes.append(hotfix)
        event.Skip()

    def OnEditFileButton(self, event): # wxGlade: DeployHotfixPanel.<event_handler>
        i = self.Files.GetFirstSelected()
        hotfix = self.hotfixes[i]
        dlg = HotfixFileDialog(self, hotfix)
        res = dlg.ShowModal()
        dlg.Destroy()
        if res == wx.ID_OK:
            self.Files.SetStringItem(i, 0, hotfix.filename)
            self.Files.SetStringItem(i, 1, hotfix.target_dir)
        event.Skip()

    def OnDeleteFileButton(self, event): # wxGlade: DeployHotfixPanel.<event_handler>
        dlg = wx.MessageDialog(self, "Are you sure you want to delete this hotfix file?",
                               "Really Delete?", wx.ICON_QUESTION|wx.YES_NO|wx.NO_DEFAULT)
        res = dlg.ShowModal()
        dlg.Destroy()
        if res == wx.ID_YES:
            i = self.Files.GetFirstSelected()
            del self.hotfixes[i]
            self.Files.DeleteItem(i)
            self.EnableEnvironmentControls()
        event.Skip()

    def OnOpenFileButton(self, event): # wxGlade: DeployHotfixPanel.<event_handler>
        hotfix = self.hotfixes[self.Files.GetFirstSelected()]
        os.startfile(hotfix.path)
        event.Skip()

    def OnNodesChecklistbox(self, event):
        self.SaveEnvironment()
        self.EnableEnvironmentControls()
        event.Skip()

    def OnDeployButton(self, event): # wxGlade: DeployHotfixPanel.<event_handler>
        for node in self.Nodes.GetSelectedNodes():
            for hotfix in self.hotfixes:
                target_path = deploy.get_target_path(node, hotfix.target_dir)
                shutil.copy(hotfix.path, target_path)
                if hotfix.target_dir == '<Inbox>':
                    flip_archive_bit(os.path.join(target_path, hotfix.filename))
        dlg = wx.MessageDialog(self, "Hotfix has been deployed.", "Success", wx.ICON_INFORMATION)
        dlg.ShowModal()
        dlg.Destroy()
        event.Skip()

    def EnableEnvironmentControls(self, enable=True):
        self.FilesLabel.Enable(enable)
        self.Files.Enable(enable)
        self.AddFile.Enable(enable)
        file_selected = bool(self.Files.GetSelectedItemCount())
        self.EditFile.Enable(enable and file_selected)
        self.DeleteFile.Enable(enable and file_selected)
        self.OpenFile.Enable(enable and file_selected)
        self.NodesLabel.Enable(enable)
        self.Nodes.Enable(enable)
        self.Deploy.Enable(enable and bool(self.Files.GetItemCount())
                           and bool(self.Nodes.GetSelectedNodes()))

    def LoadEnvironment(self, uuid=None):
        self.Nodes.Clear()
        if not (uuid and self.SelectEnvironment(uuid)):
            self.EnableEnvironmentControls(False)
            return
        env = get_hotfix_environment(uuid)
        self.Nodes.LoadEnvironment(env)
        self.EnableEnvironmentControls()

    def SaveEnvironment(self):
        env = self.GetSelectedEnvironment()
        self.Nodes.UpdateEnvironment(env)
        save_hotfix_environment(env)
        
# end of class DeployHotfixPanel


class HotfixFileDialog(wx.Dialog):

    def __init__(self, parent, hotfix, *args, **kwds):

        self.hotfix_file = hotfix
        args = (parent,) + args

        # begin wxGlade: HotfixFileDialog.__init__
        kwds["style"] = wx.DEFAULT_DIALOG_STYLE
        wx.Dialog.__init__(self, *args, **kwds)
        self.label_8 = wx.StaticText(self, -1, "Local &File:")
        self.FilePath = wx.TextCtrl(self, -1, "", style=wx.TE_READONLY)
        self.Browse = wx.Button(self, -1, "&Browse...")
        self.label_9 = wx.StaticText(self, -1, "Target &Directory:")
        self.TargetDir = wx.Choice(self, -1, choices=[])
        self.OK = wx.Button(self, wx.ID_OK, "OK")
        self.Cancel = wx.Button(self, wx.ID_CANCEL, "Cancel")

        self.__set_properties()
        self.__do_layout()

        self.Bind(wx.EVT_TEXT, self.OnFilePathText, self.FilePath)
        self.Bind(wx.EVT_BUTTON, self.OnBrowseButton, self.Browse)
        self.Bind(wx.EVT_CHOICE, self.OnTargetDirChoice, self.TargetDir)
        self.Bind(wx.EVT_BUTTON, self.OnOkButton, id=wx.ID_OK)
        # end wxGlade

        self.FilePath.SetValue(hotfix.path)
        if hotfix.target_dir:
            self.TargetDir.SetStringSelection(hotfix.target_dir)

    def __set_properties(self):
        # begin wxGlade: HotfixFileDialog.__set_properties
        self.SetTitle("Hotfix File")
        self.FilePath.SetMinSize((250, -1))
        self.OK.SetDefault()
        # end wxGlade

        for target in sorted(deploy.TARGET_DIRS):
            self.TargetDir.Append(target)

    def __do_layout(self):
        # begin wxGlade: HotfixFileDialog.__do_layout
        sizer_18 = wx.BoxSizer(wx.VERTICAL)
        sizer_19 = wx.BoxSizer(wx.VERTICAL)
        sizer_21 = wx.BoxSizer(wx.HORIZONTAL)
        sizer_20 = wx.BoxSizer(wx.HORIZONTAL)
        sizer_19.Add(self.label_8, 0, 0, 0)
        sizer_20.Add(self.FilePath, 1, 0, 0)
        sizer_20.Add(self.Browse, 0, wx.LEFT, 5)
        sizer_19.Add(sizer_20, 0, wx.TOP|wx.EXPAND, 5)
        sizer_19.Add(self.label_9, 0, wx.TOP, 10)
        sizer_19.Add(self.TargetDir, 0, wx.TOP|wx.EXPAND, 5)
        sizer_21.Add(self.OK, 0, 0, 0)
        sizer_21.Add(self.Cancel, 0, wx.LEFT, 10)
        sizer_19.Add(sizer_21, 0, wx.TOP|wx.ALIGN_CENTER_HORIZONTAL, 10)
        sizer_18.Add(sizer_19, 1, wx.ALL|wx.EXPAND, 10)
        self.SetSizer(sizer_18)
        sizer_18.Fit(self)
        self.Layout()
        self.Centre()
        # end wxGlade

    def OnFilePathText(self, event): # wxGlade: HotfixFileDialog.<event_handler>
        self.EnableControls()
        event.Skip()

    def OnBrowseButton(self, event): # wxGlade: HotfixFileDialog.<event_handler>
        dlg = wx.FileDialog(self, style=wx.FD_OPEN|wx.FD_FILE_MUST_EXIST)
        res = dlg.ShowModal()
        dlg.Destroy()
        if res == wx.ID_OK:
            self.FilePath.SetValue(dlg.GetPath())
        event.Skip()

    def OnTargetDirChoice(self, event): # wxGlade: HotfixFileDialog.<event_handler>
        self.EnableControls()
        event.Skip()

    def OnOkButton(self, event): # wxGlade: HotfixFileDialog.<event_handler>
        hotfix = self.hotfix_file
        hotfix.path = self.FilePath.GetValue()
        hotfix.target_dir = self.TargetDir.GetStringSelection()
        event.Skip()

    def EnableControls(self, enable=None):
        if enable is None:
            enable = (bool(self.FilePath.GetValue())
                      and self.TargetDir.GetSelection() >= 0)
        self.OK.Enable(enable)

# end of class HotfixFileDialog


class DeploySamplesPanel(EnvironmentalPanel):

    def __init__(self, *args, **kwds):
        # begin wxGlade: DeploySamplesPanel.__init__
        kwds["style"] = wx.TAB_TRAVERSAL
        EnvironmentalPanel.__init__(self, *args, **kwds)
        self.label_4 = wx.StaticText(self, -1, "&Environment:")
        self.Environment = wx.Choice(self, -1, choices=[])
        self.RefreshEnvironments = wx.Button(self, -1, "&Refresh")
        self.SourcePathLabel = wx.StaticText(self, -1, "&Source Repository Path:")
        self.SourcePath = wx.TextCtrl(self, -1, "", style=wx.TE_READONLY)
        self.BrowseSource = wx.Button(self, -1, "&Browse...")
        self.MakoVarsLabel = wx.StaticText(self, -1, "Mako &Variables (each will be eval()ed):")
        self.AddVar = wx.Button(self, -1, "+")
        self.DeleteVar = wx.Button(self, -1, "-")
        self.MakoVars = wx.grid.Grid(self, -1, size=(1, 1))
        self.UseSourceTimes = wx.CheckBox(self, -1, "&Use timestamps from source control.")
        self.NodesLabel = wx.StaticText(self, -1, "Target &Nodes:")
        self.Nodes = NodeListBox(self, -1, choices=[])
        self.DeploySamples = wx.Button(self, -1, "&Deploy")
        self.GenerateSamples = wx.Button(self, -1, "&Generate to...")

        self.__set_properties()
        self.__do_layout()

        self.Bind(wx.EVT_CHOICE, self.OnEnvironmentChoice, self.Environment)
        self.Bind(wx.EVT_BUTTON, self.OnRefreshEnvironmentsButton, self.RefreshEnvironments)
        self.Bind(wx.EVT_TEXT, self.OnSourcePathText, self.SourcePath)
        self.Bind(wx.EVT_BUTTON, self.OnBrowseSourceButton, self.BrowseSource)
        self.Bind(wx.EVT_BUTTON, self.OnAddVarButton, self.AddVar)
        self.Bind(wx.EVT_BUTTON, self.OnDeleteVarButton, self.DeleteVar)
        self.Bind(wx.grid.EVT_GRID_CMD_CELL_CHANGE, self.OnMakoVarsCellChange, self.MakoVars)
        self.Bind(wx.EVT_CHECKBOX, self.OnUseSourceTimesCheckbox, self.UseSourceTimes)
        self.Bind(wx.EVT_BUTTON, self.OnDeploySamplesButton, self.DeploySamples)
        self.Bind(wx.EVT_BUTTON, self.OnGenerateSamplesButton, self.GenerateSamples)
        # end wxGlade

        self.Nodes.Bind(wx.EVT_CHECKLISTBOX, self.OnNodesChecklistbox)

        self.LoadEnvironments(getter=get_samples_environment)

    def __set_properties(self):
        # begin wxGlade: DeploySamplesPanel.__set_properties
        self.AddVar.SetMinSize((25, -1))
        self.DeleteVar.SetMinSize((25, -1))
        self.MakoVars.CreateGrid(0, 2)
        self.MakoVars.SetRowLabelSize(0)
        self.MakoVars.SetColLabelSize(15)
        self.MakoVars.EnableDragRowSize(0)
        self.MakoVars.EnableDragGridSize(0)
        self.MakoVars.SetColLabelValue(0, "Name")
        self.MakoVars.SetColSize(0, 100)
        self.MakoVars.SetColLabelValue(1, "Value")
        self.MakoVars.SetColSize(1, 200)
        self.MakoVars.SetMinSize((-1, 75))
        self.UseSourceTimes.SetValue(1)
        # end wxGlade

    def __do_layout(self):
        # begin wxGlade: DeploySamplesPanel.__do_layout
        DeploySamplesSizer = wx.BoxSizer(wx.VERTICAL)
        sizer_17 = wx.BoxSizer(wx.HORIZONTAL)
        sizer_16 = wx.BoxSizer(wx.HORIZONTAL)
        sizer_10_copy = wx.BoxSizer(wx.HORIZONTAL)
        sizer_11_copy = wx.BoxSizer(wx.VERTICAL)
        sizer_1 = wx.BoxSizer(wx.HORIZONTAL)
        DeploySamplesSizer.Add(self.label_4, 0, 0, 0)
        sizer_1.Add(self.Environment, 1, wx.EXPAND, 0)
        sizer_1.Add(self.RefreshEnvironments, 0, wx.LEFT, 5)
        DeploySamplesSizer.Add(sizer_1, 0, wx.TOP|wx.EXPAND, 5)
        sizer_11_copy.Add(self.SourcePathLabel, 0, 0, 0)
        sizer_11_copy.Add(self.SourcePath, 0, wx.TOP|wx.EXPAND, 5)
        sizer_10_copy.Add(sizer_11_copy, 1, wx.EXPAND, 0)
        sizer_10_copy.Add(self.BrowseSource, 0, wx.LEFT|wx.ALIGN_BOTTOM, 5)
        DeploySamplesSizer.Add(sizer_10_copy, 0, wx.TOP|wx.EXPAND, 10)
        sizer_16.Add(self.MakoVarsLabel, 1, wx.ALIGN_BOTTOM, 0)
        sizer_16.Add(self.AddVar, 0, 0, 0)
        sizer_16.Add(self.DeleteVar, 0, wx.LEFT, 5)
        DeploySamplesSizer.Add(sizer_16, 0, wx.TOP|wx.EXPAND, 10)
        DeploySamplesSizer.Add(self.MakoVars, 1, wx.TOP|wx.EXPAND, 5)
        DeploySamplesSizer.Add(self.UseSourceTimes, 0, wx.TOP, 10)
        DeploySamplesSizer.Add(self.NodesLabel, 0, wx.TOP, 10)
        DeploySamplesSizer.Add(self.Nodes, 0, wx.TOP|wx.EXPAND, 5)
        sizer_17.Add(self.DeploySamples, 0, 0, 0)
        sizer_17.Add(self.GenerateSamples, 0, wx.LEFT, 10)
        DeploySamplesSizer.Add(sizer_17, 0, wx.TOP|wx.ALIGN_CENTER_HORIZONTAL, 10)
        self.SetSizer(DeploySamplesSizer)
        DeploySamplesSizer.Fit(self)
        # end wxGlade

    def OnEnvironmentChoice(self, event): # wxGlade: DeploySamplesPanel.<event_handler>
        self.LoadEnvironment(event.GetClientData().uuid)
        event.Skip()

    def OnRefreshEnvironmentsButton(self, event): # wxGlade: DeploySamplesPanel.<event_handler>
        self.LoadEnvironments(getter=get_samples_environment)
        event.Skip()

    def OnSourcePathText(self, event): # wxGlade: DeploySamplesPanel.<event_handler>
        self.SaveEnvironment()
        self.EnableEnvironmentControls()
        event.Skip()

    def OnBrowseSourceButton(self, event): # wxGlade: DeploySamplesPanel.<event_handler>
        path = PromptDir(self, self.SourcePath.GetValue())
        if path:
            self.SourcePath.SetValue(path)
        event.Skip()

    def OnAddVarButton(self, event): # wxGlade: DeploySamplesPanel.<event_handler>
        self.MakoVars.AppendRows(1)
        self.MakoVars.SetGridCursor(self.MakoVars.GetNumberRows() - 1, 0)
        self.MakoVars.SetFocus()
        self.DeleteVar.Enable()
        event.Skip()

    def OnDeleteVarButton(self, event): # wxGlade: DeploySamplesPanel.<event_handler>
        i = self.MakoVars.GetGridCursorRow()
        self.MakoVars.DeleteRows(i)
        if not self.MakoVars.GetNumberRows():
            self.DeleteVar.Disable()
        self.SaveEnvironment()
        event.Skip()

    def OnMakoVarsCellChange(self, event): # wxGlade: DeploySamplesPanel.<event_handler>
        self.SaveEnvironment()
        event.Skip()

    def OnUseSourceTimesCheckbox(self, event): # wxGlade: DeploySamplesPanel.<event_handler>
        self.SaveEnvironment()
        event.Skip()

    def OnNodesChecklistbox(self, event):
        self.SaveEnvironment()
        event.Skip()

    def _generate_samples(self, target_path):
        source_path = os.path.join(self.SourcePath.GetValue(), 'Samples')
        use_source_times = self.UseSourceTimes.GetValue()
        mako_vars = self.GetMakoVars()
        for name in os.listdir(source_path):
            src = os.path.join(source_path, name)
            generate_sample(edbob.config, src, target_path, use_source_time=use_source_times,
                            **mako_vars)

    def OnDeploySamplesButton(self, event): # wxGlade: DeploySamplesPanel.<event_handler>
        for node in self.Nodes.GetSelectedNodes():
            self._generate_samples(os.path.join(node.path, 'install', 'Samples'))
        dlg = wx.MessageDialog(self, "Samples have been deployed.", caption="Success",
                               style=wx.ICON_INFORMATION)
        dlg.ShowModal()
        dlg.Destroy()
        event.Skip()

    def OnGenerateSamplesButton(self, event): # wxGlade: DeploySamplesPanel.<event_handler>
        target_path = PromptDir(self)
        if target_path:
            self._generate_samples(target_path)
            dlg = wx.MessageDialog(self, "Samples have been generated.", "Success",
                                   style=wx.ICON_INFORMATION)
            dlg.ShowModal()
            dlg.Destroy()
        event.Skip()

    def EnableEnvironmentControls(self, enable=True):
        self.SourcePathLabel.Enable(enable)
        self.SourcePath.Enable(enable)
        self.BrowseSource.Enable(enable)
        self.MakoVarsLabel.Enable(enable)
        self.MakoVars.Enable(enable)
        self.AddVar.Enable(enable)
        self.DeleteVar.Enable(enable and bool(self.MakoVars.GetNumberRows()))
        self.UseSourceTimes.Enable(enable)
        self.NodesLabel.Enable(enable)
        self.Nodes.Enable(enable)
        source_path = self.SourcePath.GetValue().strip()
        self.DeploySamples.Enable(enable and bool(source_path)
                                  and bool(self.Nodes.GetSelectedNodes()))
        self.GenerateSamples.Enable(enable and bool(source_path))

    def GetMakoVars(self):
        mako_vars = {}
        for i in range(self.MakoVars.GetNumberRows()):
            name = self.MakoVars.GetCellValue(i, 0)
            value = self.MakoVars.GetCellValue(i, 1)
            if name and value:
                mako_vars[name] = eval(value)
        return mako_vars

    def LoadEnvironment(self, uuid=None):
        if not (uuid and self.SelectEnvironment(uuid)):
            self.SetEnvironmentDefaults()
            self.EnableEnvironmentControls(False)
            return
        env = get_samples_environment(uuid)
        self.SourcePath.ChangeValue(env.source_path)
        rows = self.MakoVars.GetNumberRows()
        if rows:
            self.MakoVars.DeleteRows(0, rows)
        for i, name in enumerate(sorted(env.mako_vars)):
            self.MakoVars.AppendRows(1)
            self.MakoVars.SetCellValue(i, 0, name)
            self.MakoVars.SetCellValue(i, 1, env.mako_vars[name])
        self.UseSourceTimes.SetValue(env.use_mercurial_times)
        self.Nodes.LoadEnvironment(env, servers_only=True)
        self.EnableEnvironmentControls()
        
    def SaveEnvironment(self):
        env = self.GetSelectedEnvironment()
        env.source_path = self.SourcePath.GetValue()
        env.use_mercurial_times = self.UseSourceTimes.GetValue()

        env.mako_vars.clear()
        for i in range(self.MakoVars.GetNumberRows()):
            name = self.MakoVars.GetCellValue(i, 0)
            value = self.MakoVars.GetCellValue(i, 1)
            if name and value:
                env.mako_vars[name] = value

        self.Nodes.UpdateEnvironment(env)

        save_samples_environment(env)

    def SetEnvironmentDefaults(self):
        self.SourcePath.ChangeValue('')
        rows = self.MakoVars.GetNumberRows()
        if rows:
            self.MakoVars.DeleteRows(0, rows)
        self.UseSourceTimes.SetValue(True)

# end of class DeploySamplesPanel


class SettingsPanel(wx.Panel):
    def __init__(self, *args, **kwds):
        # begin wxGlade: SettingsPanel.__init__
        kwds["style"] = wx.TAB_TRAVERSAL
        wx.Panel.__init__(self, *args, **kwds)
        self.label_1 = wx.StaticText(self, -1, "E&nvironments:")
        self.Environments = wx.ListBox(self, -1, choices=[])
        self.AddEnvironment = wx.Button(self, -1, "&Add...")
        self.EditEnvironment = wx.Button(self, -1, "&Edit...")
        self.DeleteEnvironment = wx.Button(self, -1, "&Delete")

        self.__set_properties()
        self.__do_layout()

        self.Bind(wx.EVT_LISTBOX, self.OnEnvironmentsListbox, self.Environments)
        self.Bind(wx.EVT_BUTTON, self.OnAddEnvironmentButton, self.AddEnvironment)
        self.Bind(wx.EVT_BUTTON, self.OnEditEnvironmentButton, self.EditEnvironment)
        self.Bind(wx.EVT_BUTTON, self.OnDeleteEnvironmentButton, self.DeleteEnvironment)
        # end wxGlade

        self.ReloadEnvironments()

    def __set_properties(self):
        # begin wxGlade: SettingsPanel.__set_properties
        pass
        # end wxGlade

    def __do_layout(self):
        # begin wxGlade: SettingsPanel.__do_layout
        sizer_6 = wx.BoxSizer(wx.VERTICAL)
        sizer_7 = wx.BoxSizer(wx.HORIZONTAL)
        sizer_8 = wx.BoxSizer(wx.VERTICAL)
        sizer_6.Add(self.label_1, 0, 0, 0)
        sizer_7.Add(self.Environments, 1, wx.EXPAND, 0)
        sizer_8.Add(self.AddEnvironment, 0, 0, 0)
        sizer_8.Add(self.EditEnvironment, 0, wx.TOP, 5)
        sizer_8.Add(self.DeleteEnvironment, 0, wx.TOP, 5)
        sizer_7.Add(sizer_8, 0, wx.LEFT|wx.EXPAND, 5)
        sizer_6.Add(sizer_7, 1, wx.TOP|wx.EXPAND, 5)
        self.SetSizer(sizer_6)
        sizer_6.Fit(self)
        # end wxGlade

    def OnEnvironmentsListbox(self, event): # wxGlade: SettingsPanel.<event_handler>
        self.EnableEnvironmentButtons()
        event.Skip()

    def OnAddEnvironmentButton(self, event): # wxGlade: SettingsPanel.<event_handler>
        env = deploy.Environment()
        dlg = EnvironmentDialog(self, env)
        res = dlg.ShowModal()
        dlg.Destroy()
        if res == wx.ID_OK:
            save_environment(env)
            self.ReloadEnvironments()
        event.Skip()

    def OnEditEnvironmentButton(self, event): # wxGlade: SettingsPanel.<event_handler>
        env = self.Environments.GetClientData(self.Environments.GetSelection())
        dlg = EnvironmentDialog(self, env)
        res = dlg.ShowModal()
        dlg.Destroy()
        if res == wx.ID_OK:
            save_environment(env)
            self.ReloadEnvironments()
            self.SelectEnvironment(env.uuid)
        event.Skip()

    def OnDeleteEnvironmentButton(self, event): # wxGlade: SettingsPanel.<event_handler>
        dlg = wx.MessageDialog(self, "Are you sure you want to permanently delete this environment?\n\n"
                               "This action will also delete samples and hotfix deployment configuration "
                               "related to this environment.", "Really Delete?",
                               wx.ICON_WARNING|wx.YES_NO|wx.NO_DEFAULT)
        res = dlg.ShowModal()
        dlg.Destroy()
        if res == wx.ID_YES:
            env = self.Environments.GetClientData(self.Environments.GetSelection())
            delete_environment(env)
            self.ReloadEnvironments()
        event.Skip()

    def EnableEnvironmentButtons(self, enable=None):
        if enable is None:
            enable = bool(self.Environments.GetSelections())
        self.EditEnvironment.Enable(enable)
        self.DeleteEnvironment.Enable(enable)
        
    def ReloadEnvironments(self):
        self.Environments.Clear()
        self.EnableEnvironmentButtons()
        for env in get_environments():
            self.Environments.Append(env.name, env)

    def SelectEnvironment(self, uuid):
        for i in range(self.Environments.GetCount()):
            env = self.Environments.GetClientData(i)
            if env.uuid == uuid:
                self.Environments.SetSelection(i)
                self.EnableEnvironmentButtons()
                break

# end of class SettingsPanel


class EnvironmentDialog(wx.Dialog):

    def __init__(self, parent, environment, *args, **kwds):

        self.environment = environment
        args = (parent,) + args

        # begin wxGlade: EnvironmentDialog.__init__
        kwds["style"] = wx.DEFAULT_DIALOG_STYLE
        wx.Dialog.__init__(self, *args, **kwds)
        self.label_2 = wx.StaticText(self, -1, "&Name:")
        self.EnvironmentName = wx.TextCtrl(self, -1, "")
        self.label_3 = wx.StaticText(self, -1, "N&odes:")
        self.Nodes = wx.ListView(self, -1, style=wx.LC_REPORT|wx.LC_SINGLE_SEL|wx.SUNKEN_BORDER)
        self.AddNode = wx.Button(self, -1, "&Add...")
        self.DeleteNode = wx.Button(self, -1, "&Delete")
        self.OK = wx.Button(self, wx.ID_OK, "OK")
        self.Cancel = wx.Button(self, wx.ID_CANCEL, "Cancel")

        self.__set_properties()
        self.__do_layout()

        self.Bind(wx.EVT_LIST_ITEM_DESELECTED, self.OnNodesItemDeselected, self.Nodes)
        self.Bind(wx.EVT_LIST_ITEM_SELECTED, self.OnNodesItemSelected, self.Nodes)
        self.Bind(wx.EVT_BUTTON, self.OnAddNodeButton, self.AddNode)
        self.Bind(wx.EVT_BUTTON, self.OnDeleteNodeButton, self.DeleteNode)
        self.Bind(wx.EVT_BUTTON, self.OnOkButton, id=wx.ID_OK)
        # end wxGlade

        env = self.environment
        self.EnvironmentName.SetValue(env.name)

        self.nodes = {}
        for i, node in enumerate(env.sorted_nodes):
            self.Nodes.InsertStringItem(i, node.name)
            self.Nodes.SetStringItem(i, 1, node.path)
            self.nodes[node.path] = node
        self.EnableNodeButtons()

    def __set_properties(self):
        # begin wxGlade: EnvironmentDialog.__set_properties
        self.SetTitle("Environment")
        self.Nodes.SetMinSize((300, 200))
        self.OK.SetDefault()
        # end wxGlade

        self.Nodes.InsertColumn(0, "Terminal")
        self.Nodes.InsertColumn(1, "Path")

    def __do_layout(self):
        # begin wxGlade: EnvironmentDialog.__do_layout
        sizer_9 = wx.BoxSizer(wx.VERTICAL)
        sizer_10 = wx.BoxSizer(wx.VERTICAL)
        sizer_13 = wx.BoxSizer(wx.HORIZONTAL)
        sizer_11 = wx.BoxSizer(wx.HORIZONTAL)
        sizer_12 = wx.BoxSizer(wx.VERTICAL)
        sizer_10.Add(self.label_2, 0, 0, 0)
        sizer_10.Add(self.EnvironmentName, 0, wx.TOP|wx.EXPAND, 5)
        sizer_10.Add(self.label_3, 0, wx.TOP, 10)
        sizer_11.Add(self.Nodes, 1, wx.EXPAND, 0)
        sizer_12.Add(self.AddNode, 0, 0, 0)
        sizer_12.Add(self.DeleteNode, 0, wx.TOP, 5)
        sizer_11.Add(sizer_12, 0, wx.LEFT|wx.EXPAND, 5)
        sizer_10.Add(sizer_11, 1, wx.TOP|wx.EXPAND, 5)
        sizer_13.Add(self.OK, 0, 0, 0)
        sizer_13.Add(self.Cancel, 0, wx.LEFT, 10)
        sizer_10.Add(sizer_13, 0, wx.TOP|wx.ALIGN_CENTER_HORIZONTAL, 10)
        sizer_9.Add(sizer_10, 1, wx.ALL|wx.EXPAND, 10)
        self.SetSizer(sizer_9)
        sizer_9.Fit(self)
        self.Layout()
        self.Centre()
        # end wxGlade

    def OnNodesItemDeselected(self, event): # wxGlade: EnvironmentDialog.<event_handler>
        self.EnableNodeButtons()
        event.Skip()

    def OnNodesItemSelected(self, event): # wxGlade: EnvironmentDialog.<event_handler>
        self.EnableNodeButtons()
        event.Skip()

    def OnAddNodeButton(self, event): # wxGlade: EnvironmentDialog.<event_handler>
        path = PromptDir(self)
        if path:
            self.AddNodeFromPath(path)
        event.Skip()

    def OnDeleteNodeButton(self, event): # wxGlade: EnvironmentDialog.<event_handler>
        dlg = wx.MessageDialog(self, "Do you really want to delete this node?", "Really Delete?",
                               wx.ICON_QUESTION|wx.YES_NO|wx.NO_DEFAULT)
        res = dlg.ShowModal()
        dlg.Destroy()
        if res == wx.ID_YES:
            i = self.Nodes.GetFirstSelected()
            path = self.Nodes.GetItem(i, 1).GetText()
            del self.nodes[path]
            self.Nodes.DeleteItem(i)
        event.Skip()

    def OnOkButton(self, event): # wxGlade: EnvironmentDialog.<event_handler>
        env = self.environment
        env.name = self.EnvironmentName.GetValue().strip()
        env.nodes = self.nodes.values()
        event.Skip()

    def AddNodeFromPath(self, path):
        node = deploy.Node(path=path)
        node.read_config()

        i = self.Nodes.GetItemCount()
        self.Nodes.InsertStringItem(i, node.name)
        self.Nodes.SetStringItem(i, 1, node.path)
        self.nodes[node.path] = node
        return True

    def EnableNodeButtons(self, enable=None):
        if enable is None:
            enable = bool(self.Nodes.GetSelectedItemCount())
        self.DeleteNode.Enable(enable)

# end of class EnvironmentDialog


if __name__ == '__main__':
    from edbob.wx import LaunchDialog
    edbob.init('rattail')
    LaunchDialog(AdminDialog)
