# -*- coding: utf-8 -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2014 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
SMS Database
"""

from __future__ import unicode_literals

from locsms.db import Session

from rattail.db.util import get_default_engine
from rattail.exceptions import ConfigurationError


def configure_session_factory(config, session_factory=None, section='rattail_locsms'):
    """
    Configure the bind engine for a database session factory.

    While this function does accept additional arguments, in most cases it
    would be called with only a config object.  In this usage, it will
    configure the ``locsms.db.Session`` class with the default database engine
    obtained from the config object.  The config is expected to contain
    something along these lines:

    .. code-block:: ini

       [rattail_locsms]
       sqlalchemy.url = mssql://username:password@STORESQL-DSN

       # This may be necessary when using pyodbc with FreeTDS; see also:
       # http://docs.sqlalchemy.org/en/rel_0_8/core/engines.html#sqlalchemy.create_engine.params.convert_unicode
       sqlalchemy.convert_unicode = True

       # This may be necessary to prevent connections from growing "stale" and
       # causing various types of "write to server failed" messages; see also:
       # http://docs.sqlalchemy.org/en/rel_0_8/core/engines.html#sqlalchemy.create_engine.params.pool_recycle
       sqlalchemy.pool_recycle = 3600
    """
    engine = get_default_engine(config, section=section)
    if not engine:
        raise ConfigurationError("Could not determine a default database engine "
                                 "within the [{0}] section.".format(section))
    if session_factory is None:
        session_factory = Session
    session_factory.configure(bind=engine)


def init(config):
    """
    Initialize the ``rattail_locsms.db`` module.

    .. note::
       This function is provided for compatibility with the ``edbob``
       initialization logic, but will be removed at some point.  It merely
       calls :func:`configure_session_factory()`.
    """
    configure_session_factory(config)
