#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2012 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
SMS Database Interface
"""

import warnings

from sqlalchemy import engine_from_config
from sqlalchemy.orm import sessionmaker
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.exc import SADeprecationWarning

import edbob

from rattail.core import Object


__all__ = ['engine', 'Session']

engine = None
Session = sessionmaker()
Base = declarative_base(cls=Object)


def init(config):
    """
    Initializes the SMS database interface.  An engine is constructed from
    settings found in the configuration file.  An example of such config::

       [rattail.sw.locsms]
       sqlalchemy.url = mssql://SMS-STORESQL

    The function ``sqlalchemy.engine_from_config()`` is used to construct the
    engine; see its documentation for more information.
    """

    global engine

    config.require('rattail.sw.locsms', 'sqlalchemy.url')
    config = config.get_dict('rattail.sw.locsms')
    engine = engine_from_config(config)
    Base.metadata.bind = engine

    import rattail_locsms as locsms
    from rattail_locsms import db
    edbob.graft(locsms, db)

    # Ignore warning caused by SQLAlchemy schema introspection.  This only
    # seems to happen on Linux with FreeTDS?
    with warnings.catch_warnings():
        warnings.filterwarnings(
            'ignore',
            r'^Additional keyword arguments passed to Float ignored\.$',
            SADeprecationWarning)

        from rattail_locsms.db import model
        edbob.graft(locsms, model)
