#!/usr/bin/env python
# -*- coding: utf-8  -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2012 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
Inter-Store Product Transfers
"""

import os
import os.path
import re
import csv
import logging
from collections import namedtuple

import edbob

import rattail_locsms as locsms
from rattail import sil
from rattail.core import Object
from rattail.files import locking_copy, temp_path, overwriting_move
from rattail_locsms.deploy import deploy_to_inbox
from rattail_locsms.sil import SMSWriter
from rattail_locsms.exceptions import InvalidTransferFileName


log = logging.getLogger(__name__)

file_pattern = re.compile(r'^T(?P<sending_store>\d{3})\d{3}-\d{8}-\d{4}-(?P<receiving_store>\d{3}).EXP$')

TransferInfo = namedtuple('TransferInfo', 'sending_store receiving_store')



def parse_filename(path):
    """
    Parses a transfer file name, and returns a ``TransferInfo`` (named tuple)
    instance containing the sending and receiving store ID values found.
    """

    fname = os.path.basename(path)
    match = file_pattern.match(fname)
    if not match:
        raise InvalidTransferFileName(path)

    data = match.groupdict()
    return TransferInfo(data['sending_store'], data['receiving_store'])


def copy_transfer(csv_path):
    """
    This function is designed to be invoked by the file monitor, on the
    "selling" end of a product transfer between stores.  It will inspect the
    contents of a store transfer and copy the file to the appropriate target
    folder, according to which store is "purchasing" the transfer.

    Note that this requires configuration to define which target folders are to
    be associated with each receiving store.

    Note also that this function is only needed when there is more than one
    possible receiving store.  If there is only one, a simple file copy
    function may be used instead.
    """

    info = parse_filename(csv_path)

    target_dir = edbob.config.require(
        'rattail.sw.locsms', 'transfers.{0}'.format(info.receiving_store))

    locking_copy(csv_path, target_dir)


def process_transfer(csv_path, node=None):
    """
    This function is designed to be invoked by the file monitor, on the
    receiving end of a product transfer between stores.  It will inspect the
    contents of a store transfer export file, and take steps to ensure that the
    products included will contain a valid (and current) cost record with the
    sending store as the vendor.

    Note that there are many caveats here...  Ask me sometime and I'll go into
    more detail. ;)
    """

    edbob.init_modules(['rattail_locsms.db'])

    log.debug("process_transfer: processing file: {0}".format(csv_path))
    info = parse_filename(csv_path)
    log.debug("process_transfer: sending store is '{0}', receiving store is '{1}'".format(
            info.sending_store, info.receiving_store))

    upcs = set()
    csv_file = open(csv_path, 'rb')
    reader = csv.DictReader(csv_file, quotechar="'")
    for row in reader:
        upcs.add(row['F01'])
    csv_file.close()

    log.debug("process_transfer: found {0} UPC(s)".format(len(upcs)))
    if not upcs:
        return

    add_costs = []
    change_costs = []
    session = locsms.Session()
    for upc in upcs:

        product = session.query(locsms.Product).get(upc)
        if not product:
            log.warning("process_transfer: product not found: {0}".format(upc))
            continue
        if not product.cost:
            log.warning("process_transfer: product exists but has no cost: {0}".format(repr(product)))
            continue

        magic_cost = None
        for cost in product.costs:
            if cost.F27 == info.sending_store:
                magic_cost = cost
                break

        if (magic_cost
            and magic_cost.F38 == product.cost.F38      # Base cost
            and magic_cost.F19 == product.cost.F19      # Case size
            and magic_cost.F1140 == product.cost.F1140  # Unit net cost
            and magic_cost.F220 == product.cost.F220    # Split item code
            and magic_cost.F1795 == product.cost.F1795  # Split item qty
            ):
            log.debug("process_transfer: product cost is already correct: {0}".format(repr(magic_cost)))
            continue

        cost = Object()
        cost.upc = upc
        cost.vendor_id = info.sending_store
        cost.case_cost = product.cost.F38
        cost.case_size = product.cost.F19
        cost.split_code = product.cost.F220
        cost.split_quantity = product.cost.F1795

        if magic_cost:
            log.debug("process_transfer: product cost will be included in CHANGE batch: {0}".format(repr(product)))
            change_costs.append(cost)
        else:
            log.debug("process_transfer: product cost will be included in ADD batch: {0}".format(repr(product)))
            add_costs.append(cost)
        
    session.close()

    if add_costs:
        deploy_costs(add_costs, 'ADD', node)
    if change_costs:
        deploy_costs(change_costs, 'CHANGE', node)


def deploy_costs(costs, H12, node):
    """
    Deploy a batch of new ``COST_TAB`` records (``costs``) to the SMS instance
    referenced by ``node``.
    """

    log.debug("deploy_costs: processing {0} batch with {1} cost(s)".format(H12, len(costs)))

    path = temp_path('.sil')
    writer = SMSWriter(path)

    batch_id = sil.consume_batch_id()
    writer.write_create_header(H02=batch_id)

    writer.write('CREATE TABLE COST_DCT (@DBDCT(F01,F1000,F27,F1184,F38,F19,F90,F220,F1795));\n\n')

    writer.write_maintenance_header(H02=batch_id,
                                    H12=H12,
                                    H13='COSTS FOR STORE TRANSFER')

    writer.write('CREATE VIEW COST_CHG AS SELECT F01,F1000,F27,F1184,F38,F19,F90,F220,F1795 FROM COST_DCT;\n\n')

    writer.write('INSERT INTO COST_CHG VALUES\n')

    def rows():
        for cost in costs:
            yield (
                cost.upc,               # F01
                'PAL',                  # F1000
                cost.vendor_id,         # F27
                'CASE',                 # F1184
                cost.case_cost,         # F38
                cost.case_size,         # F19
                0,                      # F90
                cost.split_code,        # F220
                cost.split_quantity,    # F1795
                )

    writer.write_rows(rows, len(costs))
    writer.close()
    deploy_to_inbox(path, node)

    save_path = edbob.config.get('rattail.sw.locsms', 'transfers.sil_backup_dir')
    if save_path:
        save_path = os.path.join(save_path, 'rattail.{0}.sil'.format(batch_id))
        log.debug("deploy_costs: saving copy of SIL file at: {0}".format(save_path))
        overwriting_move(path, save_path)
    else:
        os.remove(path)
