# -*- coding: utf-8 -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2014 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################

"""
ScanMaster Hosting Interface
"""

import os
import re
import shutil
import logging


customer_file_pattern = re.compile(ur'^(?P<batch_id>\d{8})\.(?P<file_type>SSXCUSP|SSXNCFP|SSXCRGP|EOT)$')

log = logging.getLogger(__name__)


def import_customer_data(data_path,
                         host_dir=ur'F:\Grocery\Host',
                         log_dir=ur'F:\Grocery\Files\Log',
                         log_backup_dir=ur'C:\rattail\scanmaster\customer-import\log',
                         sm2h2cus_path=ur'c:\Grocery\SM2H2CUS.EXE',
                         sm2h2cus_args=[u'/D']):
    """
    Prepare customer data files, and launch the ScanMaster customer data import
    utility.  This function is designed to be configured as a file monitor
    action.  Specifically, the following logic is utilized:

    Since the ScanMaster customer import utility expects three different files
    to be in place when it runs, this function in turn expects to be called
    multiple times for the same "batch" of customer data.  In order to be
    explicit about the "end" of the batch, a special file should be used as a
    marker.  This function then should be called a total of 4 times for a given
    batch, and the last call should be made with the end-of-batch marker file.

    It furthermore expects each file's name to include a batch identifier which
    is common to all files within a given batch.  This makes it possible to
    know with certainty that all files for a batch are accounted for, and to
    prevent files from different batches from being mixed together.  The batch
    identifier should be an 8-digit zero-padded integer.  Therefore a full list
    of files for a batch might be:

    * ``00000001.SSXCUSP``
    * ``00000001.SSXNCFP``
    * ``00000001.SSXCRGP``
    * ``00000001.EOT``

    Note that the contents of the EOT marker file are not relevant; it may be
    an empty file.  Contents of the other three files of course *is* important;
    they should conform to the ScanMaster file specifications.

    When this function is called with any of the data files, nothing is done.
    When the function is called with the EOT marker file, it goes to work.  Its
    first step is to make a copy of each of the data files, strip the batch
    identifier from the file names, and place them in the ScanMaster "host"
    folder.  Then it will launch the ScanMaster customer import utility.  Once
    this utility has exited, all original files from the batch are deleted.

    .. note::
       If you wish to save copies of the data files, you should do so *before*
       making the files available to this function, as it will always delete
       the files after the import is complete.
    """
    match = customer_file_pattern.match(os.path.basename(data_path))
    if not match:
        raise ValueError(u"File name does not match expected pattern: {0}".format(repr(data_path)))

    # Ignore everything until we receive EOT marker.
    if match.group(u'file_type') != u'EOT':
        return

    data_names = (u'SSXCUSP', u'SSXNCFP', u'SSXCRGP')

    # Make sure we have all our data files.
    batch_id = match.group(u'batch_id')
    data_dir = os.path.dirname(data_path)
    for name in data_names:
        path = os.path.join(data_dir, u'{0}.{1}'.format(batch_id, name))
        if not os.path.exists(path):
            raise RuntimeError(u"Data file does not exist: {0}".format(repr(path)))

    # Copy data files into host folder.
    for name in data_names:
        data_path = os.path.join(data_dir, u'{0}.{1}'.format(batch_id, name))
        host_path = os.path.join(host_dir, name)
        shutil.copyfile(data_path, host_path)

    # Launch the import utility.
    result = launch_customer_import_utility(
        sm2h2cus_path=sm2h2cus_path, sm2h2cus_args=sm2h2cus_args)
    if result != 0:
        log.error(u"import utility returned exit code {0} for batch {1}: {2}".format(
            repr(result), batch_id, repr(sm2h2cus_path)))

    # Make backups of import log files.
    log_backup_dir = os.path.join(log_backup_dir, batch_id)
    if not os.path.exists(log_backup_dir):
        os.makedirs(log_backup_dir)
    for name in (u'GR2CUST.LOG', u'GR2NCHEK.LOG', u'GR2CHARG.LOG'):
        log_path = os.path.join(log_dir, name)
        log_backup_path = os.path.join(log_backup_dir, name)
        shutil.copyfile(log_path, log_backup_path)

    # Remove original data files.
    for name in data_names + (u'EOT',):
        path = os.path.join(data_dir, u'{0}.{1}'.format(batch_id, name))
        os.remove(path)


def launch_customer_import_utility(sm2h2cus_path=ur'c:\Grocery\SM2H2CUS.EXE',
                                   sm2h2cus_args=[u'/D']): # pragma: no cover
    """
    Launch the ScanMaster customer import utility.

    :param sm2h2cus_path: Path to the ``SM2H2CUS.EXE`` binary.

    :param sm2h2cus_args: List of arguments which will be passed to
       ``SM2H2CUS.EXE``.

    :returns: Exit code of the launched utility's process.
    """
    from win32process import CreateProcess, STARTUPINFO, GetExitCodeProcess
    from win32con import STARTF_USESHOWWINDOW, SW_MINIMIZE, NORMAL_PRIORITY_CLASS
    from win32event import WaitForSingleObject, INFINITE, WAIT_OBJECT_0

    command_line = u'{0} {1}'.format(sm2h2cus_path, u' '.join(sm2h2cus_args))
    startup_info = STARTUPINFO()
    startup_info.dwFlags = STARTF_USESHOWWINDOW
    startup_info.wShowWindow = SW_MINIMIZE
    hProcess, hThread, dwProcessId, dwThreadId = CreateProcess(
        None, command_line, None, None, False, NORMAL_PRIORITY_CLASS, None, None, startup_info)
    result = WaitForSingleObject(hProcess, INFINITE)
    assert result == WAIT_OBJECT_0, u"invalid result waiting for import utility: {0}".format(repr(result))
    return GetExitCodeProcess(hProcess)
