#!/usr/bin/env python
# -*- coding: utf-8 -*-

from os.path import join
import os
import ConfigParser
import base64
import sys
import inspect
import argparse
import paramiko

# import the recipes from the local recipes folder
if os.getcwd() not in sys.path:
    sys.path.insert(0, os.getcwd())
import recipes

class EnvironmentConfigException(Exception):
    pass

class UnknownParameterException(Exception):
    pass

def rctl(argv=None):
    """Parse the command line arguments."""
    def csv(value):
        return value.split(',')
        
    parser = argparse.ArgumentParser(
        description='Run task recipes on remote hosts')
    parser.add_argument('-H','--hosts', required=True, type=csv, 
        help='List of hosts (comma separated)')
    parser.add_argument('-r','--recipe', required=True, 
        help='Name of the recipe to run')
    parser.add_argument('-e','--env', required=False, 
        help='Environment settings file (optional)')
    parser.add_argument('-a','--args', required=False, 
        help='Arguments for the recipe (optional)')
    # parser.add_argument('-l','--list', required=False, 
    #    help='List recipes çontained in the file')

    # injectable arguments
    if argv is None:
        args = vars(parser.parse_args())
    else:
        args = vars(parser.parse_args(argv))

    if not args['env']:
        args['env'] = 'environments.ini'

    recipe = getattr(recipes, args['recipe'])

    return args['hosts'], args['env'], args['args'], recipe


def run_recipe(hosts, env_filename, args, recipe):
    """Execute recipe on the list of given hosts"""
    if not os.path.exists(env_filename):
        raise EnvironmentConfigException(
                'Environment configuration file "%s" not found.' % envfile)

    envfile = ConfigParser.ConfigParser()
    envfile.read(env_filename)

    if envfile.has_section('config'):
        config = dict(envfile.items('config'))
    else:
        config = {}

    # check that all hosts are configured
    for host in hosts:
        if not envfile.has_section(host):
            raise EnvironmentConfigException(
                'Environment config is missing for host "%s".' % host)

    for host in hosts:
        env = dict(envfile.items(host))
        env['host'] = host
        if 'usr' in env:
            usr = env.pop('usr')
            if 'pwd' in env:
                pwd = base64.b64decode(env.pop('pwd'))
                run_recipe_on_host(host, usr, pwd, config, env, args, recipe)


def _call_with_kwargs(func, all_params):
    """Provide the parameters used in the function"""
    # check for alien parameters!
    for x in inspect.getargspec(func).args:
        if not x in all_params:
            raise UnknownParameterException(
                'Unknown parameter "%s" used in recipe "%s".' % 
                (x, func.__name__))
    kwargs = { x: all_params[x] for x in inspect.getargspec(func).args }
    func(**kwargs)


def run_recipe_on_host(host, usr, pwd, config, env, args, recipe):
    """Execute task on host"""
    ssh = paramiko.SSHClient()
    ssh.set_missing_host_key_policy(
        paramiko.AutoAddPolicy())
    if pwd:
        ssh.connect(host, username=usr, password=pwd)
    else:
        ssh.connect(host, username=usr)

    _call_with_kwargs(recipe, {'config': config, 'env': env, 'ssh': ssh, 'args': args})


def write_password(env_filename, host, pwd):
    """Write the encoded password to the environments.ini file"""
    if not os.path.exists(env_filename):
        raise EnvironmentConfigException(
                'Environment configuration file "%s" not found.' % envfile)
    if host == 'config':
        raise EnvironmentConfigException(
                'Environment configuration file cant write to '
                '"config" section.')
    envfile = ConfigParser.RawConfigParser()
    envfile.read(env_filename)
    if not envfile.has_section(host):
        raise EnvironmentConfigException(
            'Environment config is missing for host "%s".' % host)

    envfile.set(host,'pwd', base64.b64encode(pwd))
    with open(env_filename, 'w') as configfile:
        envfile.write(configfile)

