import smtplib
from smtplib import SMTPException
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
import logging

logger = logging.getLogger(__name__)
logger.addHandler(logging.NullHandler())

"""
This module groups together related functions for
sending emails.

Module unit test: test/TestRediEmail.py
"""

def send_email_redcap_connection_error(email_settings, subject='', msg=''):
    """
    Return True if the email was sent.
    Notify the designated `REDCap support` person about problems
    with reaching REDCap

    Parameters
    ----------
    email_settings : dict
        The dictionary with smtp server parameters
    subject : str
        The email subject
    msg : str
        The content to be emailed
    """

    sender = email_settings['redcap_support_sender_email']
    to_addr_list = email_settings['redcap_support_receiving_list']
    host = email_settings['smtp_host_for_outbound_mail']
    port = email_settings['smtp_port_for_outbound_mail']
    subject = 'Communication failure: Unable to reach REDCap instance'
    msg = 'A problem was encountered when connecting to the REDCap. Please investigate if REDCap is running.'

    logger.error('Exception: Unable to communicate with REDCap instance at: ' + email_settings['redcap_uri'])
    return send_email(host, str(port), sender, to_addr_list, None, subject, msg)

def send_email_input_data_unchanged(email_settings, subject='', msg=''):
    """
    Send a warning email to the `redcap_support_receiver_email`
    if the input file did not change for more than `batch_warning_days`

    :return True if the email was sent
    """
    sender = email_settings['redcap_support_sender_email']
    to_addr_list = email_settings['redcap_support_receiving_list']
    host = email_settings['smtp_host_for_outbound_mail']
    port = email_settings['smtp_port_for_outbound_mail']
    subject = 'Input data is static.'
    msg = """
    Administrators,
    For the past {} days the input data for the REDI application did not change.
    Please investigate.""".format(email_settings['batch_warning_days'])
    return send_email(host, str(port), sender, to_addr_list, None, subject, msg)

def send_email_data_import_completed(email_settings, body=''):
    """
    Email the html report after redi completed the data transfer
    :email_settings the dictionary produced by redi.get_email_settings()
    :body: the html string produced by transforming the xsl
        generated by redi.create_summary_report()

    :return a dictionary, with one entry for each recipient that was refused
    """
    sender = email_settings['batch_report_sender_email']
    to_addr_list = email_settings['batch_report_receiving_list']
    host = email_settings['smtp_host_for_outbound_mail']
    port = email_settings['smtp_port_for_outbound_mail']
    subject = 'Data Import Report'
    msg = MIMEMultipart()
    msg['From'] = sender
    msg['To'] = ",".join(to_addr_list)
    msg['Subject'] = subject
    msg.attach(MIMEText(body, 'html'))

    refused_list = {}
    try:
        smtpObj = smtplib.SMTP(host, port)
        refused_list = smtpObj.sendmail(sender, to_addr_list, msg.as_string())
        logger.info("Successfully sent email to: " + str(to_addr_list))
    except Exception:
         logger.error("Unable to send email with subject [{}] to {}" \
                .format(subject, str(to_addr_list)))
         raise
    smtpObj.quit()
    return refused_list

def send_email(
        host,
        port,
        sender,
        to_addr_list,
        cc_addr_list,
        subject,
        msg_body):
    """
    The email deliverer
    :to_addr_list: must be a list not a string

    :return True if the email was sent
    """
    #print ('host %s, port: %s' % (host, port))
    success = False
    try:
        smtp = smtplib.SMTP(host, port)
        header = 'From: %s\n' % sender
        header += 'To: %s\n' % ','.join(to_addr_list)
        if cc_addr_list:
            header += 'Cc: %s\n' % ','.join(cc_addr_list)
        if subject:
            header += 'Subject: %s\n\n' % subject
        msg = header + msg_body
        smtp.sendmail(sender, to_addr_list, msg)
        success = True
        logger.info(
            'Success: Email with subject [' +
            subject +
            '] was sent to:' +
            str(to_addr_list))
    except SMTPException:
        logger.error("Unable to send email with subject [{}] to {}" \
                .format(subject, str(to_addr_list)))
        logger.info("Please check if the recipient email is valid")
    except Exception as e:
        logger.error("Unable to send email with subject [{}] to {}\n{}" \
                .format(subject, str(to_addr_list), msg_body))
        logger.info("Please check if the smtp server is configured properly")
    return success
