"""
Manage the Mac pasteboard (aka "clipboard") supporting the
possibility of cutting and pasting rich text (HTML and RTF),
in addition to plain text.
"""

# Code inspired by and/or based on
# http://genbastechthoughts.wordpress.com/2012/05/20/reading-urls-from-os-x-clipboard-with-pyobjc/
# and # http://blog.carlsensei.com/post/88897796

# Mac OS X: put export LC_CTYPE=en_US.utf-8 in .bash_profile to avoid anoying console UnicodeError

import Foundation
from AppKit import NSPasteboard

formats = ['rtf', 'html', 'text'] # in preference order

uti = {
    'html': 'public.html',
    'rtf':  'public.rtf',
    'text': 'public.utf8-plain-text'
}

# UTIs are Uniform Type Identfifiers (http://en.wikipedia.org/wiki/Uniform_Type_Identifier)
# ...even if they sound a lot like urinary tract infections

def paste(prefer=None):
    """
    Get data from the Mac pasteboard (aka clipboard). Prefers a list of
    formats such as HTML and RTF, but takes plain text if that's what's available.
    That order of preference can be stated on a per-call basis.
    """
    prefer = prefer or formats
    pb = NSPasteboard.generalPasteboard()
    for format in prefer:
        contents = pb.stringForType_(uti[format])
        if contents:
            return unicode(contents)
    return None

def available(dyn=False):
    """
    What type of data is available on the pasteboard? By default, excludes items whose
    type starts with 'dyn.', which is a usually pointless internal format.
    """
    pb = NSPasteboard.generalPasteboard()
    if dyn:
        return pb.pasteboardItems()[0].types()
    else:
        return [ t for t in pb.pasteboardItems()[0].types() if not t.startswith('dyn.') ] 

def pasteall(dyn=False):
    """
    Return a dict of all available data types, matched to the associated
    content. Good for curiosity, if not as useful in practice.
    """
    pb = NSPasteboard.generalPasteboard()
    return { t: pb.stringForType_(t) for t in available(dyn) }

def copy(**kwargs):
    """
    Put contents onto the Mac pasteboard (aka clipboard). Called with a kwargs
    style, such as copy(rtf=some_rtf, html=some_html). This lets a caller add
    multiple representations in parallel.
    """
    pb = NSPasteboard.generalPasteboard()
    copy_utis = [ uti[f] for f in kwargs.keys() ]
    pb.declareTypes_owner_(copy_utis, None)

    for format, value in kwargs.iteritems():
        value_uti = uti[format]
        if isinstance(value, unicode):
            value = value.encode('utf-8')
        newStr = Foundation.NSString.stringWithString_(value).nsstring()
        newData = newStr.dataUsingEncoding_(Foundation.NSUTF8StringEncoding)
        pb.setData_forType_(newData, value_uti)
    
def uutf(v):
    """
    Coerce a value to unicode(), then get its utf8 encoding. 
    """
    return unicode(v).encode('utf-8')

def dict_print(d, heading=None, wrap=False, **kwargs):
    """
    Tidy printing of dictionary elements. Only here
    for demonstation and debugging.
    """
    import textwrap
    kwargs['subsequent_indent'] = kwargs.get('subsequent_indent', kwargs.get('initial_indent', '') + '    ')
    
    if heading:
        print heading
    for key, value in d.iteritems():
        item_text = "{}: {}".format(uutf(key), uutf(value))
        if wrap:
            print '\n'.join(textwrap.wrap(item_text, **kwargs))
        else:
            print item_text
    
def test_richxerox():
    print available()
    print paste()
    dict_print(pasteall(), 'ALL CONTENTS')
    copy(text="this is good!", html="this is <strong>good</strong>!")
    dict_print(pasteall(), 'AFTER COPY')

    
if __name__ == '__main__':
    test_richxerox()